/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz;

import org.eclipse.draw2d.Polyline;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.swt.SWT;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandFigure;
import de.bsvrz.buv.plugin.dobj.decorator.LinienstaerkeFigure;
import de.bsvrz.buv.plugin.dobj.decorator.model.Zoomverhalten;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dobj.vektor.LinAlgUtil;

public class LinieFigure extends Polyline
		implements DoFigure, LinienabstandFigure, LinienstaerkeFigure {

	private static final int DEFAULT_LINIENABSTAND = 0;
	private static final int DEFAULT_LINIENSTAERKE = 1;

	private PointList basePoints;
	private int linienstaerke = DEFAULT_LINIENSTAERKE;
	private IScaleProvider<Integer> linienabstandProvider = new ConstantScaleProvider(
			DEFAULT_LINIENABSTAND, 0.0001, 1.0);

	private double zoom = 1.0;
	private double vonZoom;
	private double bisZoom = Double.MAX_VALUE;
	private Zoomverhalten linienstaerkeZoomverhalten;
	private int linienstaerkeMaximum;
	private int linienstaerkeMinimum;
	private double linienstaerkeMaximaleZoomstufe;
	private double linienstaerkeMinimaleZoomstufe;
	private int linienabstand;
	private double linienabstandMaximaleZoomstufe;
	private double linienabstandMinimaleZoomstufe;
	private Zoomverhalten linienabstandZoomverhalten;
	private int linienabstandMaximum;
	private int linienabstandMinimum;

	{
		setAntialias(SWT.ON);
		setLineJoin(SWT.JOIN_ROUND);
	}

	@Override
	public Point getHotspot() {
		return getLocation();
	}

	@Override
	public void setHotspot(final Point hotspot) {
		setLocation(hotspot);
	}

	@Override
	public final int getLinienstaerke() {
		return linienstaerke;
	}

	@Override
	public final void setLinienstaerke(final int linienstaerke) {
		if (this.linienstaerke == linienstaerke) {
			return;
		}

		this.linienstaerke = linienstaerke;

		updateFigure();
	}

	@Override
	public final int getLinienabstand() {
		return linienabstand;
	}

	@Override
	public final void setLinienabstand(final int linienabstand) {
		if (this.linienabstand == linienabstand) {
			return;
		}
		linienabstandProvider = createScaleProvider();

		updateFigure();
	}

	private IScaleProvider<Integer> createScaleProvider() {
		if (linienabstandZoomverhalten
				.equals(Zoomverhalten.DYNAMISCHES_ZOOMVERHALTEN)) {
			return new DynamicScaleProvider(linienabstandMinimum,
					linienabstandMinimaleZoomstufe, linienabstandMaximum,
					linienabstandMaximaleZoomstufe);
		}

		return new ConstantScaleProvider(linienabstand, 0.0001, 1.0);
	}

	public final double getZoom() {
		return zoom;
	}

	@Override
	public final void setSichtbareZoomStufe(final double vonZoom,
			final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		updateFigure();
	}

	@Override
	public final void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateFigure();
	}

	public final PointList getBasePoints() {
		return basePoints;
	}

	public final void setBasePoints(final PointList basePoints) {
		if (DobjUtil.equals(this.basePoints, basePoints)) {
			return;
		}

		this.basePoints = basePoints;

		updateFigure();
	}

	@Override
	public Zoomverhalten getLinienstaerkeZoomverhalten() {
		return linienstaerkeZoomverhalten;
	}

	@Override
	public void setLinienstaerkeZoomverhalten(
			final Zoomverhalten zoomverhalten) {
		this.linienstaerkeZoomverhalten = zoomverhalten;
		updateFigure();
	}

	protected void updateFigure() {
		updateVisible();

		if (isSetBasePoints()) {
			setTolerance((int) (2 / getZoom()));

			setPoints(getVerschobeneBasePoints());

			if (getLinienstaerkeZoomverhalten() == Zoomverhalten.STATISCHES_ZOOMVERHALTEN) {
				setLineWidthFloat((float) (getLinienstaerke() / getZoom()));
			} else {
				final int minimaleLinienstaerke = getLinienstaerkeMinimum();
				final int maximaleLinienstaerke = getLinienstaerkeMaximum();
				final double minimaleZoomstufe = getLinienstaerkeMinimaleZoomstufe();
				final double maximaleZoomstufe = getLinienstaerkeMaximaleZoomstufe();
				if (getZoom() > maximaleZoomstufe) {
					setLineWidthFloat(
							(float) (getLinienstaerkeMaximum() / getZoom()));
				} else if (getZoom() < minimaleZoomstufe) {
					setLineWidthFloat(
							(float) (getLinienstaerkeMinimum() / getZoom()));
				} else {

					// Anstieg: m = (y2-y1)/(x2-x1)
					final double m = (maximaleLinienstaerke
							- minimaleLinienstaerke)
							/ (maximaleZoomstufe - minimaleZoomstufe);

					// Durchgang durch Y-Achse: n = y2-m*x2
					final double n = maximaleLinienstaerke
							- m * maximaleZoomstufe;

					// y = m*x+n
					setLineWidthFloat(
							(float) ((m * getZoom() + n) / getZoom()));

				}
			}
		} else {
			removeAllPoints();
		}
	}

	@Override
	public void setLinienstaerkeMaximum(final int maximaleLinienstaerke) {
		if (this.linienstaerkeMaximum == maximaleLinienstaerke) {
			return;
		}
		this.linienstaerkeMaximum = maximaleLinienstaerke;
		updateFigure();
	}

	@Override
	public int getLinienstaerkeMaximum() {
		return linienstaerkeMaximum;
	}

	@Override
	public void setLinienstaerkeMinimum(final int minimaleLinienstarke) {
		if (this.linienstaerkeMinimum == minimaleLinienstarke) {
			return;
		}
		this.linienstaerkeMinimum = minimaleLinienstarke;
		updateFigure();
	}

	@Override
	public int getLinienstaerkeMinimum() {
		return linienstaerkeMinimum;
	}

	@Override
	public void setLinienstaerkeMaximaleZoomstufe(
			final double maximaleZoomstufe) {
		if (Double.compare(this.linienstaerkeMaximaleZoomstufe,
				maximaleZoomstufe) == 0) {
			return;
		}
		this.linienstaerkeMaximaleZoomstufe = maximaleZoomstufe;
		updateFigure();
	}

	@Override
	public double getLinienstaerkeMaximaleZoomstufe() {
		return linienstaerkeMaximaleZoomstufe;
	}

	@Override
	public void setLinienstaerkeMinimaleZoomstufe(
			final double minimaleZoomstufe) {
		if (Double.compare(this.linienstaerkeMinimaleZoomstufe,
				minimaleZoomstufe) == 0) {
			return;
		}
		this.linienstaerkeMinimaleZoomstufe = minimaleZoomstufe;
		updateFigure();
	}

	@Override
	public double getLinienstaerkeMinimaleZoomstufe() {
		return linienstaerkeMinimaleZoomstufe;
	}

	protected void updateVisible() {
		setVisible(isZoomInSichtbareZoomStufe());
	}

	protected final boolean isZoomInSichtbareZoomStufe() {
		return vonZoom <= zoom && zoom <= bisZoom;
	}

	protected final PointList getVerschobeneBasePoints() {
		final int abstand = getVerschobeneBasePointsAbstand();
		if (abstand == 0) {
			return getBasePoints();
		}

		return LinAlgUtil.verschiebeUndStrecke(getBasePoints(), abstand,
				getZoom());

	}

	protected int getVerschobeneBasePointsAbstand() {
		return linienabstandProvider.getValueAt(zoom);
	}

	private boolean isSetBasePoints() {
		return getBasePoints() != null && getBasePoints().size() > 0;
	}

	@Override
	public void setBounds(final Rectangle rect) {
		/*
		 * FIXME Workaround für den Fall bounds == null
		 *
		 * Die Bounds dürfen an der Polylinie eigentlich nicht gesetzt werden,
		 * weil sie automatisch berechnet werden. Das Layout versucht dennoch
		 * die Bounds setzen.
		 */
		if (bounds != null) {
			super.setBounds(rect);
		}
	}

	@Override
	public double getLinienabstandMaximaleZoomstufe() {
		return linienabstandMaximaleZoomstufe;
	}

	@Override
	public void setLinienabstandMaximaleZoomstufe(
			final double maximaleZoomstufe) {
		if (linienabstandMaximaleZoomstufe == maximaleZoomstufe) {
			return;
		}
		linienabstandMaximaleZoomstufe = maximaleZoomstufe;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public double getLinienabstandMinimaleZoomstufe() {
		return linienabstandMinimaleZoomstufe;
	}

	@Override
	public void setLinienabstandMinimaleZoomstufe(
			final double minimaleZoomstufe) {
		if (linienabstandMinimaleZoomstufe == minimaleZoomstufe) {
			return;
		}
		linienabstandMinimaleZoomstufe = minimaleZoomstufe;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public Zoomverhalten getLinienabstandZoomverhalten() {
		return linienabstandZoomverhalten;
	}

	@Override
	public void setLinienabstandZoomverhalten(
			final Zoomverhalten zoomverhalten) {
		if (linienabstandZoomverhalten == zoomverhalten) {
			return;
		}
		linienabstandZoomverhalten = zoomverhalten;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public int getLinienabstandMaximum() {
		return linienabstandMaximum;
	}

	@Override
	public void setLinienabstandMaximum(final int maximaleLinienabstand) {
		if (Double.compare(linienabstandMaximum, maximaleLinienabstand) == 0) {
			return;
		}
		linienabstandMaximum = maximaleLinienabstand;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public int getLinienabstandMinimum() {
		return linienabstandMinimum;
	}

	@Override
	public void setLinienabstandMinimum(final int minimaleLinienabstand) {
		if (Double.compare(linienabstandMinimum, minimaleLinienabstand) == 0) {
			return;
		}
		linienabstandMinimum = minimaleLinienabstand;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}
}
