/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.ImageFigure;
import org.eclipse.draw2d.Polyline;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.swt.graphics.Image;

import com.bitctrl.lib.eclipse.draw2d.FixedSizeFigure;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.AntikollisionsalgorithmusFigure;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandFigure;
import de.bsvrz.buv.plugin.dobj.decorator.model.Zoomverhalten;
import de.bsvrz.buv.plugin.dobj.kollision.IArrangierbaresObjektMitExtraBounds;
import de.bsvrz.buv.plugin.dobj.kollision.KollisionsManager;
import de.bsvrz.buv.plugin.dobj.kollision.travers.IPunktTraverser;
import de.bsvrz.buv.plugin.dobj.kollision.travers.KegelStumpfUeberPunktAbsucher;
import de.bsvrz.buv.plugin.dobj.kollision.travers.NullPunktTraverser;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dobj.vektor.LinAlgUtil;
import de.bsvrz.buv.plugin.dobj.vektor.Vektor2D;

/**
 * Ein Symbol, welches den Antikollisionsalgorithmus nutzt.
 *
 * <p>
 * Das Symbol verwendet als Originalposition den Anfang einer Strecke. Der
 * Abstand des Symbols zu dieser Strecke ist variabel. Zwischen dem Symbol und
 * der Originalposition kann eine Verbindungslinie eingeblendet werden.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 *
 */
public class SymbolFigure extends Figure
		implements DoFigure, LinienabstandFigure,
		AntikollisionsalgorithmusFigure, IArrangierbaresObjektMitExtraBounds {

	private static final int DEFAULT_LINIENABSTAND = 0;

	private int linienabstand = 0;

	// Antikollisionsalgorithmus
	private final KollisionsManager kollisionsManager;
	private boolean antikollisionsalgorithmusAnwenden = true;
	private Point originalLocation;
	private Point aktuellerStartPunktDerSuche;
	private Vektor2D aktuellerNormalenVektorZumErstenLinienZug;

	private final Image image;

	// Child Figures
	private final FixedSizeFigure symbol;
	private final Polyline verbindungslinie;

	// Ausrichten an Strecke
	private PointList streckenabschnitt;

	// Properties
	private IScaleProvider<Integer> linienabstandProvider = new ConstantScaleProvider(
			DEFAULT_LINIENABSTAND, 0.0001, 1.0);
	private double zoom = 1.0;
	private double vonZoom;
	private double bisZoom = Double.MAX_VALUE;

	private int linienabstandMinimum;

	private int linienabstandMaximum;

	private Zoomverhalten linienabstandZoomverhalten;

	private double linienabstandMinimaleZoomstufe;

	private double linienabstandMaximaleZoomstufe;

	/**
	 * Initialisiert die Figure.
	 */
	public SymbolFigure(final KollisionsManager kollisionsManager,
			final Image image) {
		this.kollisionsManager = kollisionsManager;
		this.image = image;

		verbindungslinie = new Polyline();
		verbindungslinie.setForegroundColor(ColorConstants.gray);
		add(verbindungslinie);

		final Dimension imageSize = new Dimension(image);
		symbol = new FixedSizeFigure(imageSize);
		final ImageFigure imageFigure = new ImageFigure(image);
		imageFigure.setSize(imageSize);
		symbol.add(imageFigure);
		add(symbol);

		updateBounds();
	}

	/**
	 * Gibt das Image des Symbols zurück.
	 */
	protected final Image getImage() {
		return image;
	}

	/**
	 * Gibt die Figure des Symbols zurück.
	 */
	protected final FixedSizeFigure getSymbol() {
		return symbol;
	}

	/**
	 * Gibt die Figure der Verbindungslinie zurück.
	 */
	protected final Polyline getVerbindungslinie() {
		return verbindungslinie;
	}

	/**
	 * Gibt den Kollisionsmanager des Antikollisionsalgorithmus zurück.
	 */
	protected final KollisionsManager getKollisionsManager() {
		return kollisionsManager;
	}

	@Override
	public void removeNotify() {
		if (kollisionsManager != null) {
			kollisionsManager.remove(this);
		}

		super.removeNotify();
	}

	/**
	 * Gibt den Streckenabschnitt zurück, an dem das Symbol positioniert wird.
	 */
	public final PointList getStreckenabschnitt() {
		return streckenabschnitt;
	}

	/**
	 * Legt den Streckenabschnitt fest, an dem das Symbol positioniert wird.
	 */
	public final void setStreckenabschnitt(final PointList streckenabschnitt) {
		if (DobjUtil.equals(this.streckenabschnitt, streckenabschnitt)) {
			return;
		}

		this.streckenabschnitt = streckenabschnitt;

		updateFigure();
	}

	@Override
	public final int getLinienabstand() {
		return linienabstand;
	}

	@Override
	public final void setLinienabstand(final int linienabstand) {
		if (this.linienabstand == linienabstand) {
			return;
		}
		this.linienabstand = linienabstand;
		linienabstandProvider = createScaleProvider();

		updateFigure();
	}

	private IScaleProvider<Integer> createScaleProvider() {
		if (linienabstandZoomverhalten
				.equals(Zoomverhalten.DYNAMISCHES_ZOOMVERHALTEN)) {
			return new DynamicScaleProvider(linienabstandMinimum,
					linienabstandMinimaleZoomstufe, linienabstandMaximum,
					linienabstandMaximaleZoomstufe);
		}

		return new ConstantScaleProvider(linienabstand, 0.0001, 1.0);
	}

	@Override
	public final IPunktTraverser getAlternativeOrte() {
		if (aktuellerStartPunktDerSuche == null
				|| aktuellerNormalenVektorZumErstenLinienZug == null) {
			return new NullPunktTraverser();
		}

		return new KegelStumpfUeberPunktAbsucher(aktuellerStartPunktDerSuche,
				aktuellerNormalenVektorZumErstenLinienZug,
				symbol.getBounds().preciseWidth(), 100);
	}

	@Override
	public final boolean isSichtbar() {
		return isShowing();
	}

	@Override
	public Point getHotspot() {
		return getLocation();
	}

	@Override
	public void setHotspot(final Point hotspot) {
		setLocation(hotspot);
	}

	/**
	 * Gibt die Originalposition der Figure zurück, die sie ohne
	 * Antikollisionsalgorithmus einnimmt.
	 */
	public final Point getOriginalLocation() {
		return originalLocation;
	}

	/**
	 * Legt die Originalposition der Figure fest, die sie ohne
	 * Antikollisionsalgorithmus einnimmt.
	 */
	public final void setOriginalLocation(final Point originalLocation) {
		if (originalLocation.equals(this.originalLocation)) {
			return;
		}

		this.originalLocation = originalLocation;

		updateFigure();
	}

	/**
	 * Flag, ob die Verbindungslinie zwischen der Originalposition und der
	 * verschobenen Position angezeigt werden soll.
	 */
	public final boolean isVerbindungslinieSichtbar() {
		return verbindungslinie.isVisible();
	}

	/**
	 * Flag, ob die Verbindungslinie zwischen der Originalposition und der
	 * verschobenen Position angezeigt werden soll.
	 */
	public final void setVerbindungslinieSichtbar(
			final boolean verbindungslinieSichtbar) {
		verbindungslinie.setVisible(verbindungslinieSichtbar);
		updateFigure();
	}

	/**
	 * Flag, ob die Figure per Antikollisionsalgorithmus neu positioniert werden
	 * darf.
	 */
	@Override
	public final boolean isAntikollisionsalgorithmusAnwenden() {
		return kollisionsManager != null && antikollisionsalgorithmusAnwenden;
	}

	/**
	 * Flag, ob die Figure per Antikollisionsalgorithmus neu positioniert werden
	 * darf.
	 */
	@Override
	public final void setAntikollisionsalgorithmusAnwenden(
			final boolean antikollisionsalgorithmusAnwenden) {
		if (this.antikollisionsalgorithmusAnwenden == antikollisionsalgorithmusAnwenden) {
			return;
		}

		this.antikollisionsalgorithmusAnwenden = antikollisionsalgorithmusAnwenden;

		updateFigure();
	}

	@Override
	public final Rectangle getExtraBounds() {
		final Rectangle r = symbol.getBounds().getCopy();
		translateToParent(r);
		return r;
	}

	@Override
	public void setSichtbareZoomStufe(final double vonZoom,
			final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		updateFigure();
	}

	/**
	 * Gibt den aktuellen Zoom zurück.
	 */
	public final double getZoom() {
		return zoom;
	}

	@Override
	public final void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateFigure();
	}

	/**
	 * Wird bei Änderung der in dieser Klassen definierten zusätzlichen
	 * Eigenschaften der Figure aufgerufen.
	 */
	protected void updateFigure() {
		updateVisible();

		symbol.setLocation(new Point());
		symbol.setZoom(zoom);

		verbindungslinie.removeAllPoints();
		verbindungslinie.setLineWidthFloat((float) (1 / zoom));

		if (streckenabschnitt != null && streckenabschnitt.size() > 0) {
			final Point original = LinAlgUtil.getPunktNebenPunktListeAnfang(
					streckenabschnitt, getLinienabstand());

			Point symbolCenter;
			if (original != null) {
				originalLocation = original;
				symbolCenter = original;

				final double symbolUmkreisRadius = symbol.getBounds().height
						/ 1.5;
				aktuellerStartPunktDerSuche = LinAlgUtil
						.getPunktNebenPunktListeAnfang(streckenabschnitt,
								linienabstandProvider.getValueAt(zoom)
										+ symbolUmkreisRadius);

				if (aktuellerStartPunktDerSuche != null) {
					symbolCenter = aktuellerStartPunktDerSuche;
					aktuellerNormalenVektorZumErstenLinienZug = LinAlgUtil
							.getNormalenVektorZumErstenLinienZugVon(
									getStreckenabschnitt(), true)
							.getVektorNormiert();
					if (isAntikollisionsalgorithmusAnwenden()) {
						symbolCenter = getKollisionsManager().verorte(this);
					}
				}
			} else {
				// Wenn keine Verschiebung berechnet werden kann, psotioniere
				// das Symbol auf dem ersten Punkt
				originalLocation = streckenabschnitt.getFirstPoint();
				symbolCenter = originalLocation;
			}

			final Point symbolLocation = symbolCenter.getTranslated(
					-symbol.getBounds().width / 2,
					-symbol.getBounds().height / 2);
			symbol.setLocation(symbolLocation);

		} else {
			originalLocation = getHotspot();
		}

		verbindungslinie.addPoint(symbol.getBounds().getCenter());
		verbindungslinie.addPoint(originalLocation);

		updateBounds();
	}

	/**
	 * Berechnet die Sichtbarkeit der Figure und ruft
	 * {@link #setVisible(boolean)} auf.
	 */
	protected void updateVisible() {
		setVisible(isZoomInSichtbareZoomStufe());
	}

	/**
	 * Flag, ob der aktuelle Zoom sich innerhalb der erlaubten Zoomgrenzen
	 * befindet.
	 */
	protected final boolean isZoomInSichtbareZoomStufe() {
		return vonZoom <= zoom && zoom <= bisZoom;
	}

	/**
	 * Berechnet die Bounds der Figure und ruft {@link #setBounds(Rectangle)}
	 * auf.
	 */
	protected void updateBounds() {
		Rectangle r = symbol.getBounds();
		if (verbindungslinie.isVisible()) {
			r = r.getUnion(verbindungslinie.getBounds());
		}
		setBounds(r);
	}

	@Override
	protected void primTranslate(final int dx, final int dy) {
		// XXX Workaround damit Child-Figures nicht verschoben werden.
		bounds.x += dx;
		bounds.y += dy;
	}

	@Override
	public double getLinienabstandMaximaleZoomstufe() {
		return linienabstandMaximaleZoomstufe;
	}

	@Override
	public void setLinienabstandMaximaleZoomstufe(
			final double maximaleZoomstufe) {
		if (linienabstandMaximaleZoomstufe == maximaleZoomstufe) {
			return;
		}
		linienabstandMaximaleZoomstufe = maximaleZoomstufe;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public double getLinienabstandMinimaleZoomstufe() {
		return linienabstandMinimaleZoomstufe;
	}

	@Override
	public void setLinienabstandMinimaleZoomstufe(
			final double minimaleZoomstufe) {
		if (linienabstandMinimaleZoomstufe == minimaleZoomstufe) {
			return;
		}
		linienabstandMinimaleZoomstufe = minimaleZoomstufe;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public Zoomverhalten getLinienabstandZoomverhalten() {
		return linienabstandZoomverhalten;
	}

	@Override
	public void setLinienabstandZoomverhalten(
			final Zoomverhalten zoomverhalten) {
		if (linienabstandZoomverhalten == zoomverhalten) {
			return;
		}
		linienabstandZoomverhalten = zoomverhalten;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public int getLinienabstandMaximum() {
		return linienabstandMaximum;
	}

	@Override
	public void setLinienabstandMaximum(final int maximaleLinienabstand) {
		if (Double.compare(linienabstandMaximum, maximaleLinienabstand) == 0) {
			return;
		}
		linienabstandMaximum = maximaleLinienabstand;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

	@Override
	public int getLinienabstandMinimum() {
		return linienabstandMinimum;
	}

	@Override
	public void setLinienabstandMinimum(final int minimaleLinienabstand) {
		if (Double.compare(linienabstandMinimum, minimaleLinienabstand) == 0) {
			return;
		}
		linienabstandMinimum = minimaleLinienabstand;
		linienabstandProvider = createScaleProvider();
		updateFigure();
	}

}
