/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.geo;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.Polygon;
import org.eclipse.draw2d.Polyline;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.swt.graphics.Color;

import de.bsvrz.buv.plugin.dobj.DoFigure;

/**
 * Zeichnet ein Systemobjekt vom Typ <em>KomplexXY</em>.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class KomplexFigure extends Figure implements DoFigure {

	private double zoom = 1.0;
	private double vonZoom;
	private double bisZoom = Double.MAX_VALUE;

	private List<PointList> flaechen = new ArrayList<>();
	private List<PointList> linien = new ArrayList<>();

	@Override
	public Point getHotspot() {
		return getLocation();
	}

	@Override
	public void setHotspot(final Point hotspot) {
		setLocation(hotspot);
	}

	@Override
	public boolean containsPoint(final int x, final int y) {
		if (super.containsPoint(x, y)) {
			for (final PointList e : flaechen) {
				if (e.polygonContainsPoint(x, y)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Legt die zu zeichnenden Flächen fest.
	 */
	public void setFlaechen(final List<PointList> flaechen) {
		this.flaechen = flaechen;

		updateFigure();
	}

	/**
	 * Legt die zu zeichnenden Linien fest.
	 */
	public void setLinien(final List<PointList> linien) {
		this.linien = linien;

		updateFigure();
	}

	@Override
	public void setForegroundColor(final Color fg) {
		super.setForegroundColor(fg);
		updateFigure();
	}

	@Override
	public void setBackgroundColor(final Color bg) {
		super.setBackgroundColor(bg);
		updateFigure();
	}

	@Override
	public void setSichtbareZoomStufe(final double vonZoom,
			final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		updateFigure();
	}

	@Override
	public void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateFigure();
	}

	private void updateFigure() {
		setVisible(vonZoom <= zoom && zoom <= bisZoom);
		removeAll();

		Rectangle newBounds = null;
		if (isSetFlaechen()) {
			for (final PointList flaeche : flaechen) {
				final Polygon polygon = new Polygon();
				polygon.setForegroundColor(getForegroundColor());
				polygon.setBackgroundColor(getBackgroundColor());
				polygon.setLineWidthFloat((float) (1.0 / zoom));
				polygon.setFill(true);
				polygon.setPoints(flaeche);
				add(polygon);

				if (newBounds == null) {
					newBounds = polygon.getBounds();
				} else {
					newBounds.union(polygon.getBounds());
				}
			}
		}

		if (isSetLinien()) {
			for (final PointList linie : linien) {
				final Polyline polyline = new Polyline();
				polyline.setForegroundColor(getForegroundColor());
				polyline.setBackgroundColor(getBackgroundColor());
				polyline.setLineWidthFloat((float) (1.0 / zoom));
				polyline.setFill(true);
				polyline.setPoints(linie);
				add(polyline);

				if (newBounds == null) {
					newBounds = polyline.getBounds();
				} else {
					newBounds.union(polyline.getBounds());
				}
			}
		}

		// if (isSetLinien()) {
		// for (final PointList linie : linien) {
		// // TODO enthaltene Linien, Punkte und Komplexe zeichnen
		// }
		// }

		if (newBounds != null) {
			setBounds(newBounds);
		} else {
			setBounds(new Rectangle());
		}
	}

	private boolean isSetFlaechen() {
		return flaechen != null && !flaechen.isEmpty();
	}

	private boolean isSetLinien() {
		return linien != null && !linien.isEmpty();
	}

}
