/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.messquerschnitt;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;

import org.eclipse.draw2d.Polygon;
import org.eclipse.draw2d.ScalableFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionDimension;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.swt.graphics.Color;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.Datenstatus;
import de.bsvrz.buv.plugin.dobj.decorator.DatenstatusFigure;
import de.bsvrz.buv.plugin.dobj.decorator.DrehungFigure;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandFigure;
import de.bsvrz.buv.plugin.dobj.decorator.model.Zoomverhalten;
import de.bsvrz.buv.plugin.dobj.vektor.LinAlgUtil;

/**
 * MVC-View für einen Messquerschnitt.
 *
 * <p>
 * Ein Messquerschnitt wird als Dreieck mit den Punkten a, b, und c gezeichnet.
 * Der Punkt a ist gleichzeitig der Hotspot der Figure.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
final class MessQuerschnittFigure extends Polygon implements DoFigure,
		ScalableFigure, LinienabstandFigure, DatenstatusFigure, DrehungFigure {

	// Absoluter Abstand der Punkte bei unskalierter Darstellung.
	private static final double ABSTAND = 10;

	// Winkel zwischen ab und ac.
	private static final int WINKEL = 60;

	// Farben für die Zustände der Sendesteuerung
	private Color keineQuelleFarbe;
	private Color keineDatenFarbe;
	private Color nutzdatenFarbe;

	private double zoom = 1.0;
	private double vonZoom;
	private double bisZoom = Double.MAX_VALUE;
	private double scale = 1.0;

	private Point hotspot;
	private PointList streckenabschnitt;

	private int linienabstand;

	private double drehwinkel;

	private Datenstatus datenstatus = Datenstatus.KeineQuelle;

	private double linienabstandMaximaleZoomstufe;

	private double linienabstandMinimaleZoomstufe;

	private Zoomverhalten linienabstandZoomverhalten;

	private int linienabstandMaximum;

	private int linienabstandMinimum;

	{
		setFill(true);
		updateFigure();
	}

	@Override
	public Point getHotspot() {
		return hotspot;
	}

	@Override
	public void setHotspot(final Point hotspot) {
		if (this.hotspot != null && this.hotspot.equals(hotspot)) {
			return;
		}

		this.hotspot = hotspot;

		updateFigure();
	}

	/**
	 * Setzt den Streckenabschnitt, auf dem sich der MQ befindet.
	 *
	 * @param streckenabschnitt
	 *            der Streckenabschnitt in Form zweier Punkte.
	 */
	void setStreckenabschnitt(final Point[] streckenabschnitt) {
		if (streckenabschnitt != null) {
			this.streckenabschnitt = new PointList();
			this.streckenabschnitt.addPoint(streckenabschnitt[0]);
			this.streckenabschnitt.addPoint(streckenabschnitt[1]);
		} else {
			this.streckenabschnitt = null;
		}

		updateFigure();
	}

	@Override
	public int getLinienabstand() {
		return linienabstand;
	}

	@Override
	public void setLinienabstand(final int linienabstand) {
		if (this.linienabstand == linienabstand) {
			return;
		}

		this.linienabstand = linienabstand;

		updateFigure();
	}

	@Override
	public Datenstatus getDatenstatus() {
		return datenstatus;
	}

	@Override
	public void setDatenstatus(final Datenstatus datenstatus) {
		if (this.datenstatus == datenstatus) {
			return;
		}

		this.datenstatus = datenstatus;
		refreshDatenstatus();
	}

	private void refreshDatenstatus() {
		switch (datenstatus) {
		case KeineQuelle:
			setBackgroundColor(keineQuelleFarbe);
			break;
		case KeineDaten:
			setBackgroundColor(keineDatenFarbe);
			break;
		case Nutzdaten:
			setBackgroundColor(nutzdatenFarbe);
			break;
		default:
			throw new IllegalArgumentException(
					"Unbekannter Datenstatus: " + datenstatus);
		}
	}

	@Override
	public void setScale(final double scale) {
		if (this.scale == scale) {
			return;
		}

		this.scale = scale;

		updateFigure();
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	public void setSichtbareZoomStufe(final double vonZoom,
			final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		updateFigure();
	}

	@Override
	public void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateFigure();
	}

	private int calculateLinienabstand() {
		if (Zoomverhalten.DYNAMISCHES_ZOOMVERHALTEN
				.equals(linienabstandZoomverhalten)) {
			if (zoom > linienabstandMaximaleZoomstufe) {
				return linienabstandMaximum;
			} else if (zoom < linienabstandMinimaleZoomstufe) {
				return linienabstandMinimum;
			}

			final double m = (linienabstandMaximum - linienabstandMinimum)
					/ (linienabstandMaximaleZoomstufe
							- linienabstandMinimaleZoomstufe);

			final double n = linienabstandMaximum
					- m * linienabstandMaximaleZoomstufe;

			// y = m*x+n
			return Math.round((float) (m * zoom + n));
		}

		return linienabstand;

	}

	private void updateFigure() {
		setVisible(hotspot != null && vonZoom <= zoom && zoom <= bisZoom);
		removeAllPoints();

		final double combinedScale = zoom / scale;
		setLineWidthFloat((float) (1 / zoom));

		Point a = null;
		double winkel = 90;
		if (isAnLinieAusrichten()) {
			/**
			 * Verschiebe a, je nach Skalierung bzw. Zoomstufe
			 */
			a = LinAlgUtil.getPunktNebenPunktListeAnfang(streckenabschnitt,
					calculateLinienabstand() / combinedScale);

			if (a != null) {
				// Am Streckenabschnitt ausrichten
				final double dX = streckenabschnitt.getPoint(0).preciseX()
						- streckenabschnitt.getPoint(1).preciseX();
				final double dY = streckenabschnitt.getPoint(0).preciseY()
						- streckenabschnitt.getPoint(1).preciseY();
				winkel = Math.toDegrees(Math.atan2(dY, dX));
			}
		}

		if (a == null) {
			if (hotspot != null) {
				a = hotspot;
			} else {
				a = new PrecisionPoint(0, ABSTAND / 2);
			}
		}

		final Point b = new PrecisionPoint(
				a.preciseX() + (ABSTAND * Math
						.cos(Math.toRadians(winkel - (90 - (WINKEL / 2))))),
				a.preciseY() + (ABSTAND * Math
						.sin(Math.toRadians(winkel - (90 - (WINKEL / 2))))));
		final Point c = new PrecisionPoint(
				a.preciseX() + (ABSTAND * Math
						.cos(Math.toRadians(winkel - (90 + (WINKEL / 2))))),
				a.preciseY() + (ABSTAND * Math
						.sin(Math.toRadians(winkel - (90 + (WINKEL / 2))))));

		// Relative Verschiebung der anderen Punkte bestimmen
		final Dimension db = new PrecisionDimension(b.preciseX() - a.preciseX(),
				b.preciseY() - a.preciseY());
		final Dimension dc = new PrecisionDimension(c.preciseX() - a.preciseX(),
				c.preciseY() - a.preciseY());

		final PointList polygonPunkte = new PointList();
		polygonPunkte.addPoint(a);
		Dimension d = db.getCopy().scale(1 / combinedScale);
		polygonPunkte.addPoint(a.getCopy().translate(d));
		d = dc.getCopy().scale(1 / combinedScale);
		polygonPunkte.addPoint(a.getCopy().translate(d));

		if (drehwinkel > 0) {
			final double rad = drehwinkel / 360 * 2 * Math.PI;
			final Point2D[] altePunkte = new Point2D[polygonPunkte.size()];

			for (int i = 0; i < polygonPunkte.size(); i++) {
				final Point p = polygonPunkte.getPoint(i);
				altePunkte[i] = new Point2D.Float(p.x - a.x, p.y - a.y);
			}

			final AffineTransform transform = new AffineTransform();
			transform.rotate(rad);
			// transform.scale(scale, scale);

			final Point2D[] neuePunkte = new Point2D[polygonPunkte.size()];
			transform.transform(altePunkte, 0, neuePunkte, 0,
					polygonPunkte.size());

			for (int i = 0; i < polygonPunkte.size(); i++) {
				final Point2D p = neuePunkte[i];
				addPoint(a.getCopy()
						.translate(new PrecisionPoint(p.getX(), p.getY())));
			}
		} else {
			setPoints(polygonPunkte);
		}
	}

	private boolean isAnLinieAusrichten() {
		return streckenabschnitt != null && streckenabschnitt.size() > 1;
	}

	@Override
	public Color getKeineQuelleFarbe() {
		return keineQuelleFarbe;
	}

	@Override
	public void setKeineQuelleFarbe(final Color keineQuelleFarbe) {
		this.keineQuelleFarbe = keineQuelleFarbe;
		refreshDatenstatus();
	}

	@Override
	public Color getKeineDatenFarbe() {
		return keineDatenFarbe;
	}

	@Override
	public void setKeineDatenFarbe(final Color keineDatenFarbe) {
		this.keineDatenFarbe = keineDatenFarbe;
		refreshDatenstatus();
	}

	@Override
	public Color getNutzdatenFarbe() {
		return nutzdatenFarbe;
	}

	@Override
	public void setNutzdatenFarbe(final Color nutzdatenFarbe) {
		this.nutzdatenFarbe = nutzdatenFarbe;
		refreshDatenstatus();
	}

	@Override
	public float getDrehwinkel() {
		return (float) drehwinkel;
	}

	@Override
	public void setDrehwinkel(final float drehwinkel) {
		if (this.drehwinkel == drehwinkel) {
			return;
		}

		this.drehwinkel = drehwinkel;

		updateFigure();
	}

	@Override
	public double getLinienabstandMaximaleZoomstufe() {
		return linienabstandMaximaleZoomstufe;
	}

	@Override
	public void setLinienabstandMaximaleZoomstufe(
			final double maximaleZoomstufe) {
		if (linienabstandMaximaleZoomstufe == maximaleZoomstufe) {
			return;
		}
		linienabstandMaximaleZoomstufe = maximaleZoomstufe;
		updateFigure();
	}

	@Override
	public double getLinienabstandMinimaleZoomstufe() {
		return linienabstandMinimaleZoomstufe;
	}

	@Override
	public void setLinienabstandMinimaleZoomstufe(
			final double minimaleZoomstufe) {
		if (linienabstandMinimaleZoomstufe == minimaleZoomstufe) {
			return;
		}
		linienabstandMinimaleZoomstufe = minimaleZoomstufe;
		updateFigure();
	}

	@Override
	public Zoomverhalten getLinienabstandZoomverhalten() {
		return linienabstandZoomverhalten;
	}

	@Override
	public void setLinienabstandZoomverhalten(
			final Zoomverhalten zoomverhalten) {
		if (linienabstandZoomverhalten == zoomverhalten) {
			return;
		}
		linienabstandZoomverhalten = zoomverhalten;
		updateFigure();
	}

	@Override
	public int getLinienabstandMaximum() {
		return linienabstandMaximum;
	}

	@Override
	public void setLinienabstandMaximum(final int maximaleLinienabstand) {
		if (Double.compare(linienabstandMaximum, maximaleLinienabstand) == 0) {
			return;
		}
		linienabstandMaximum = maximaleLinienabstand;
		updateFigure();
	}

	@Override
	public int getLinienabstandMinimum() {
		return linienabstandMinimum;
	}

	@Override
	public void setLinienabstandMinimum(final int minimaleLinienabstand) {
		if (Double.compare(linienabstandMinimum, minimaleLinienabstand) == 0) {
			return;
		}
		linienabstandMinimum = minimaleLinienabstand;
		updateFigure();
	}
}
