/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.messquerschnitt;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.GraphicalEditPart;

import de.bsvrz.buv.plugin.dobj.HotspotConverter;
import de.bsvrz.buv.plugin.dobj.decorator.Datenstatus;
import de.bsvrz.buv.plugin.dobj.decorator.DatenstatusMediator;
import de.bsvrz.buv.plugin.dobj.decorator.DrehungMediator;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandZoomMediator;
import de.bsvrz.buv.plugin.dobj.decorator.SkalierungMediator;
import de.bsvrz.buv.plugin.dobj.decorator.ZoomVerhaltenFixMediator;
import de.bsvrz.buv.plugin.dobj.editparts.PunktEditPart;
import de.bsvrz.buv.plugin.netz.model.MessQuerschnittNBAHanDoModel;
import de.bsvrz.buv.plugin.netz.model.MessQuerschnittNBAHanDoTyp;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;

/**
 * MVC-Controller für einen Messquerschnitt NBAHan.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @author BitCtrl Systems GmbH, Christian Hösel
 *
 */
public final class MessQuerschnittNBAHanEditPart extends
		PunktEditPart<MessQuerschnittAllgemein, MessQuerschnittNBAHanFigure> {

	private MessQuerschnittEmpfaenger empfaenger;
	private MessQuerschnittLegendeBaustein legendeBaustein;

	@Override
	public MessQuerschnittNBAHanDoModel getModel() {
		return (MessQuerschnittNBAHanDoModel) super.getModel();
	}

	private MessQuerschnittNBAHanDoTyp getDoTyp() {
		return (MessQuerschnittNBAHanDoTyp) getModel().getDoTyp();
	}

	@Override
	protected MessQuerschnittNBAHanFigure createFigure() {
		return new MessQuerschnittNBAHanFigure();
	}

	@Override
	protected IFigure getToolTip() {
		// Außerhalb des Editor wird der dynamische Tooltip per Edit Policy
		// verwendet, deswegen hier keinen zurückgeben.

		if (!isEditor()) {
			return null;
		}

		return super.getToolTip();
	}

	@Override
	protected Point getWgs84Location() {
		vorladen(MessQuerschnittAllgemein.PID, KdPunktLiegtAufLinienObjekt.PID);
		vorladen(MessQuerschnittAllgemein.PID, KdPunktKoordinaten.PID);

		return super.getWgs84Location();
	}

	@Override
	protected void createEditPolicies() {
		super.createEditPolicies();

		if (!isEditor()) {
			installEditPolicy(EditPolicy.SELECTION_FEEDBACK_ROLE,
					new MessQuerschnittPopUpEditPolicy());
		}
	}

	@Override
	public void deactivate() {
		disposeLegende();

		super.deactivate();
	}

	/**
	 * Meldet sich fuer den assoziierten MQ auf Online-Daten der Attributgruppe
	 * "atg.verkehrsDatenKurzZeitMq" unter dem Aspekt "asp.analyse" an.
	 */
	@Override
	public void geheOnline() {
		super.geheOnline();

		final SystemObject so = getModel().getSystemObject();
		if (so != null) {
			empfaenger = new MessQuerschnittEmpfaenger(so, getFigure(),
					getDavZeitStempelManager());
			empfaenger.anmelden();
		}
	}

	/**
	 * Meldet sich fuer den assoziierten MQ von Online-Daten der Attributgruppe
	 * "atg.verkehrsDatenKurzZeitMq" unter dem Aspekt "asp.analyse" ab.
	 */
	@Override
	public void geheOffline() {
		getFigure().setDatenstatus(Datenstatus.KeineQuelle);

		if (empfaenger != null) {
			empfaenger.abmelden();
			empfaenger = null;
		}

		super.geheOffline();
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		final Object notifier = notification.getNotifier();
		if (notification.getEventType() != Notification.REMOVING_ADAPTER
				&& (notifier instanceof MessQuerschnittNBAHanDoTyp
						|| notifier instanceof MessQuerschnittNBAHanDoModel)) {
			refreshVisuals();
		}
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		refreshLocation();

		new DrehungMediator(this).mediate();
		new SkalierungMediator(this).mediate();
		new LinienabstandZoomMediator(this).mediate();
		new DatenstatusMediator(this).mediate();
		new ZoomVerhaltenFixMediator(this).mediate();

		correctSelectionBoundsIfNecessary();
	}

	private void refreshLocation() {
		final MessQuerschnittNBAHanDoModel model = getModel();
		final MessQuerschnittNBAHanFigure f = getFigure();

		if (model.getLocation() != null) {
			f.setHotspot(model.getAbsoluteLocation());
			f.setStreckenabschnitt(null);
		} else {
			f.setHotspot(getLocation());
			f.setStreckenabschnitt(getStreckenAbschnitt());
		}

		((GraphicalEditPart) getParent()).setLayoutConstraint(this, f,
				f.getBounds());
	}

	/**
	 * Beim Drehen oder Skalieren können sich die Bounds ändern. Wenn das
	 * passiert muss falls das Objekt markiert ist, auch der Markierungsrahmen
	 * um das Objekt angepasst werden.
	 */
	private void correctSelectionBoundsIfNecessary() {
		if (getSelected() != SELECTED_NONE) {
			final int oldValue = getSelected();
			setSelected(SELECTED_NONE);
			setSelected(oldValue);
		}
	}

	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") final Class key) {
		if (ILegendeBaustein.class.equals(key)) {
			return getLegendeBaustein();
		} else if (HotspotConverter.class.equals(key)) {
			return new HotspotConverter() {

				@Override
				public Point convertViewToModel(final Point viewLocation) {
					final Rectangle bounds = getFigure().getBounds();
					final Point hotspot = getFigure().getHotspot();
					final Dimension offset = hotspot
							.getDifference(bounds.getLocation());
					return viewLocation.getTranslated(offset);
				}

				@Override
				public Point convertModelToView(final Point modelLocation) {
					final Rectangle bounds = getFigure().getBounds();
					final Point hotspot = getFigure().getHotspot();
					final Dimension offset = hotspot
							.getDifference(bounds.getLocation());
					return modelLocation.getTranslated(offset.getNegated());
				}

			};
		}

		return super.getAdapter(key);
	}

	private ILegendeBaustein getLegendeBaustein() {
		disposeLegende();
		legendeBaustein = new MessQuerschnittLegendeBaustein(getDoTyp(),
				createFigure());
		return legendeBaustein;
	}

	private void disposeLegende() {
		if (legendeBaustein != null) {
			legendeBaustein.dispose();
			legendeBaustein = null;
		}
	}

}
