/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.messquerschnitt;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.Polygon;
import org.eclipse.draw2d.ScalableFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.swt.graphics.Color;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.Datenstatus;
import de.bsvrz.buv.plugin.dobj.decorator.DatenstatusFigure;
import de.bsvrz.buv.plugin.dobj.decorator.DrehungFigure;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandFigure;
import de.bsvrz.buv.plugin.dobj.decorator.ZoomVerhaltenFixFigure;
import de.bsvrz.buv.plugin.dobj.decorator.model.Zoomverhalten;
import de.bsvrz.buv.plugin.dobj.vektor.LinAlgUtil;

/**
 * MVC-View für einen Messquerschnitt NBAHan.
 *
 * <p>
 * Dieser Messquerschnitt wird als Viereck gezeichnet. Der Punkt a ist
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @author BitCtrl Systems GmbH, Christian Hösel
 * @author BitCtrl Systems GmbH, Daniel Hammerschmidt
 */
final class MessQuerschnittNBAHanFigure extends Polygon
		implements DoFigure, ScalableFigure, LinienabstandFigure,
		DatenstatusFigure, DrehungFigure, ZoomVerhaltenFixFigure {

	// Absoluter Abstand der Punkte bei unskalierter Darstellung.
	private static final int ABSTAND = 10;

	// Farben für die Zustände der Sendesteuerung
	private Color keineQuelleFarbe;
	private Color keineDatenFarbe;
	private Color nutzdatenFarbe;

	private double zoom = 1.0;
	private double vonZoom;
	private double bisZoom = Double.MAX_VALUE;
	private double scale = 1.0;

	private Point hotspot;
	private PointList streckenabschnitt;

	private int linienabstand;

	private float drehwinkel;

	private boolean zoomVerhaltenFix = true;

	private final Polygon innerPolygon = new Polygon();

	private Datenstatus datenstatus = Datenstatus.KeineQuelle;

	private int linienabstandMinimum;

	private int linienabstandMaximum;

	private Zoomverhalten linienabstandZoomverhalten;

	private double linienabstandMinimaleZoomstufe;

	private double linienabstandMaximaleZoomstufe;

	{
		setFill(true);
		setOutline(false);
		innerPolygon.setFill(true);
		innerPolygon.setOutline(false);
		updateFigure();
	}

	@Override
	public Point getHotspot() {
		return hotspot;
	}

	@Override
	public void setHotspot(final Point hotspot) {
		if (this.hotspot != null && this.hotspot.equals(hotspot)) {
			return;
		}

		this.hotspot = hotspot;

		updateFigure();
	}

	@Override
	protected void fillShape(final Graphics g) {
		final Color fg = g.getForegroundColor();
		final Color bg = g.getBackgroundColor();

		g.setBackgroundColor(fg);
		g.fillPolygon(getPoints());
		g.setBackgroundColor(bg);
		g.fillPolygon(innerPolygon.getPoints());

	}

	/**
	 * Setzt den Streckenabschnitt, auf dem sich der MQ befindet.
	 *
	 * @param streckenabschnitt
	 *            der Streckenabschnitt in Form zweier Punkte.
	 */
	void setStreckenabschnitt(final Point[] streckenabschnitt) {
		if (streckenabschnitt != null) {
			this.streckenabschnitt = new PointList();
			this.streckenabschnitt.addPoint(streckenabschnitt[0]);
			this.streckenabschnitt.addPoint(streckenabschnitt[1]);
		} else {
			this.streckenabschnitt = null;
		}

		updateFigure();
	}

	@Override
	public int getLinienabstand() {
		return linienabstand;
	}

	@Override
	public void setLinienabstand(final int linienabstand) {
		if (this.linienabstand == linienabstand) {
			return;
		}

		this.linienabstand = linienabstand;

		updateFigure();
	}

	@Override
	public Datenstatus getDatenstatus() {
		return datenstatus;
	}

	@Override
	public void setDatenstatus(final Datenstatus datenstatus) {
		if (this.datenstatus == datenstatus) {
			return;
		}

		this.datenstatus = datenstatus;
		refreshDatenstatus();
	}

	private void refreshDatenstatus() {
		switch (datenstatus) {
		case KeineQuelle:
			setBackgroundColor(keineQuelleFarbe);
			break;
		case KeineDaten:
			setBackgroundColor(keineDatenFarbe);
			break;
		case Nutzdaten:
			setBackgroundColor(nutzdatenFarbe);
			break;
		default:
			throw new IllegalArgumentException(
					"Unbekannter Datenstatus: " + datenstatus);
		}
	}

	@Override
	public void setScale(final double scale) {
		if (this.scale == scale) {
			return;
		}

		this.scale = scale;

		updateFigure();
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	public void setSichtbareZoomStufe(final double vonZoom,
			final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		updateFigure();
	}

	@Override
	public void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateFigure();
	}

	private void computeAndAddPolygonPoints(final Polygon polygon, final int x,
			final int y, final int w, final int h, final double winkel) {

		final Rectangle rect = new Rectangle(x, y, w, h);

		final PointList polygonPunkte = new PointList();
		polygonPunkte.addPoint(rect.getTopRight());
		polygonPunkte.addPoint(rect.getBottomRight());
		polygonPunkte.addPoint(rect.getBottomLeft());
		polygonPunkte.addPoint(rect.getTopLeft());

		final Point2D[] altePunkte = new Point2D[polygonPunkte.size()];

		for (int i = 0; i < polygonPunkte.size(); i++) {
			final Point p = polygonPunkte.getPoint(i);
			altePunkte[i] = new Point2D.Float(p.x, p.y);
		}

		final AffineTransform transform = new AffineTransform();
		transform.rotate(Math.toRadians(winkel), rect.getCenter().preciseX(),
				rect.getCenter().preciseY());

		final Point2D[] neuePunkte = new Point2D[polygonPunkte.size()];
		transform.transform(altePunkte, 0, neuePunkte, 0, polygonPunkte.size());

		for (int i = 0; i < polygonPunkte.size(); i++) {
			final Point2D p = neuePunkte[i];
			polygon.addPoint(new PrecisionPoint(p.getX(), p.getY()));
		}
	}

	private void updateFigure() {
		setVisible(vonZoom <= zoom && zoom <= bisZoom);
		removeAllPoints();

		double combinedScale;
		if (zoomVerhaltenFix) {
			combinedScale = zoom / scale;
		} else {
			combinedScale = 1 / scale;
		}
		setLineWidthFloat((float) (1 / zoom));

		final Point a;
		final double winkel;
		if (isAnLinieAusrichten()) {
			/**
			 * Verschiebe a, je nach Skalierung bzw. Zoomstufe
			 */
			a = LinAlgUtil.getPunktNebenPunktListeAnfang(streckenabschnitt,
					calculateLinienabstand() / combinedScale);
			a.translate(new PrecisionPoint((-ABSTAND) / combinedScale,
					(-ABSTAND / 2) / combinedScale));

			// Am Streckenabschnitt ausrichten
			final double dX = streckenabschnitt.getPoint(0).preciseX()
					- streckenabschnitt.getPoint(1).preciseX();
			final double dY = streckenabschnitt.getPoint(0).preciseY()
					- streckenabschnitt.getPoint(1).preciseY();
			winkel = Math.toDegrees(Math.atan2(dY, dX));
		} else {
			if (hotspot != null) {
				a = hotspot;
			} else {
				a = new Point(0, ABSTAND / 2);
			}

			winkel = 90;
		}

		innerPolygon.removeAllPoints();

		final Dimension dim = new Dimension(ABSTAND / 2, ABSTAND);
		dim.scale(1 / combinedScale);

		computeAndAddPolygonPoints(this, a.x, a.y, dim.width, dim.height,
				winkel + drehwinkel);
		computeAndAddPolygonPoints(innerPolygon, a.x + 1, a.y + 1,
				dim.width - 2, dim.height - 2, winkel + drehwinkel);

	}

	private boolean isAnLinieAusrichten() {
		return streckenabschnitt != null;
	}

	@Override
	public Color getKeineQuelleFarbe() {
		return keineQuelleFarbe;
	}

	@Override
	public void setKeineQuelleFarbe(final Color keineQuelleFarbe) {
		this.keineQuelleFarbe = keineQuelleFarbe;
		refreshDatenstatus();
	}

	@Override
	public Color getKeineDatenFarbe() {
		return keineDatenFarbe;
	}

	@Override
	public void setKeineDatenFarbe(final Color keineDatenFarbe) {
		this.keineDatenFarbe = keineDatenFarbe;
		refreshDatenstatus();
	}

	@Override
	public Color getNutzdatenFarbe() {
		return nutzdatenFarbe;
	}

	@Override
	public void setNutzdatenFarbe(final Color nutzdatenFarbe) {
		this.nutzdatenFarbe = nutzdatenFarbe;
		refreshDatenstatus();
	}

	@Override
	public float getDrehwinkel() {
		return drehwinkel;
	}

	@Override
	public void setDrehwinkel(final float drehwinkel) {
		if (this.drehwinkel == drehwinkel) {
			return;
		}

		this.drehwinkel = drehwinkel;

		updateFigure();
	}

	@Override
	public boolean isZoomVerhaltenFix() {
		return zoomVerhaltenFix;
	}

	@Override
	public void setZoomVerhaltenFix(final boolean zoomVerhaltenFix) {
		if (this.zoomVerhaltenFix == zoomVerhaltenFix) {
			return;
		}
		this.zoomVerhaltenFix = zoomVerhaltenFix;
		updateFigure();

	}

	@Override
	public double getLinienabstandMaximaleZoomstufe() {
		return linienabstandMaximaleZoomstufe;
	}

	@Override
	public void setLinienabstandMaximaleZoomstufe(
			final double maximaleZoomstufe) {
		if (linienabstandMaximaleZoomstufe == maximaleZoomstufe) {
			return;
		}
		linienabstandMaximaleZoomstufe = maximaleZoomstufe;
		updateFigure();
	}

	@Override
	public double getLinienabstandMinimaleZoomstufe() {
		return linienabstandMinimaleZoomstufe;
	}

	@Override
	public void setLinienabstandMinimaleZoomstufe(
			final double minimaleZoomstufe) {
		if (linienabstandMinimaleZoomstufe == minimaleZoomstufe) {
			return;
		}
		linienabstandMinimaleZoomstufe = minimaleZoomstufe;
		updateFigure();
	}

	@Override
	public Zoomverhalten getLinienabstandZoomverhalten() {
		return linienabstandZoomverhalten;
	}

	@Override
	public void setLinienabstandZoomverhalten(
			final Zoomverhalten zoomverhalten) {
		if (linienabstandZoomverhalten == zoomverhalten) {
			return;
		}
		linienabstandZoomverhalten = zoomverhalten;
		updateFigure();
	}

	@Override
	public int getLinienabstandMaximum() {
		return linienabstandMaximum;
	}

	@Override
	public void setLinienabstandMaximum(final int maximaleLinienabstand) {
		if (Double.compare(linienabstandMaximum, maximaleLinienabstand) == 0) {
			return;
		}
		linienabstandMaximum = maximaleLinienabstand;
		updateFigure();
	}

	@Override
	public int getLinienabstandMinimum() {
		return linienabstandMinimum;
	}

	@Override
	public void setLinienabstandMinimum(final int minimaleLinienabstand) {
		if (Double.compare(linienabstandMinimum, minimaleLinienabstand) == 0) {
			return;
		}
		linienabstandMinimum = minimaleLinienabstand;
		updateFigure();
	}

	private int calculateLinienabstand() {
		if (Zoomverhalten.DYNAMISCHES_ZOOMVERHALTEN
				.equals(linienabstandZoomverhalten)) {
			if (zoom > linienabstandMaximaleZoomstufe) {
				return linienabstandMaximum;
			} else if (zoom < linienabstandMinimaleZoomstufe) {
				return linienabstandMinimum;
			}

			final double m = (linienabstandMaximum - linienabstandMinimum)
					/ (linienabstandMaximaleZoomstufe
							- linienabstandMinimaleZoomstufe);

			final double n = linienabstandMaximum
					- m * linienabstandMaximaleZoomstufe;

			// y = m*x+n
			return Math.round((float) (m * zoom + n));
		}

		return linienabstand;

	}

}
