/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.messquerschnitt;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Comparator;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.netz.internal.NetzPlugin;
import de.bsvrz.buv.plugin.netz.model.MessQuerschnittDoTyp;
import de.bsvrz.buv.plugin.netz.ui.DObjPopUp;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum.Status;
import de.bsvrz.sys.funclib.bitctrl.modell.fachmodellglobal.attribute.AttProzent;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AttGeschwindigkeit;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AttVerkehrsStaerkeIntervall;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdFahrStreifen;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.FahrStreifen;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.onlinedaten.OdVerkehrsDatenKurzZeitIntervall;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.onlinedaten.OdVerkehrsDatenKurzZeitMq;

/**
 * Tooltip fuer MessQuerschnittEditPart.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 *
 */
final class MessQuerschnittPopUp extends DObjPopUp<MessQuerschnittAllgemein>
		implements DatensatzUpdateListener {

	MessQuerschnittPopUp(final Shell shell, final MessQuerschnittAllgemein mst,
			final DoModel doModel) {
		super(shell, mst, doModel);
		mqTyp = (MessQuerschnittDoTyp) doModel.getDoTyp();
		final String zeitStempelFormat = mqTyp.getTooltipZeitstempelFormat();
		if (zeitStempelFormat != null) {
			try {
				mqDateFormater = new SimpleDateFormat(zeitStempelFormat);
			} catch (final IllegalArgumentException ex) {
				NetzPlugin.getDefault().getLog()
						.log(new org.eclipse.core.runtime.Status(
								IStatus.WARNING, NetzPlugin.PLUGIN_ID,
								"Ungültiges Datumsformat für MQ Tooltips", ex));
			}
		}

	}

	private SimpleDateFormat mqDateFormater = FORMAT_DATUM_ZEIT_GENAU;
	private final MessQuerschnittDoTyp mqTyp;
	private Label zeitstempel;
	private Label b;
	private Label qKfz;
	private Label qPkw;
	private Label qLkw;
	private Label vKfz;
	private Label vPkw;
	private Label vLkw;
	private TableViewer fsTable;
	private SortedMap<FahrStreifen, OdVerkehrsDatenKurzZeitIntervall.Daten> fsDaten;

	@Override
	protected Control createDialogArea(final Composite parent) {
		final Composite composite = (Composite) super.createDialogArea(parent);
		composite.setLayout(new GridLayout(1, true));
		composite.setLayoutData(GridDataFactory.fillDefaults()
				.minSize(400, SWT.DEFAULT).grab(true, true).create());

		createVerkehrsDatenKurzZeitMq(composite);
		if (getSystemObjekt() instanceof MessQuerschnitt) {
			createFahrStreifenViewer(composite);
		}

		return composite;
		// return container;
	}

	private void createVerkehrsDatenKurzZeitMq(final Composite parent) {

		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayoutData(GridDataFactory.swtDefaults()
				.align(SWT.FILL, SWT.FILL).grab(true, false).create());
		container.setLayout(new GridLayout(6, true));

		Label label = new Label(container, SWT.NONE);
		label.setText("Zeit:");
		zeitstempel = new Label(container, SWT.NONE);
		zeitstempel.setLayoutData(
				GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER)
						.grab(true, false).span(5, 1).create());

		final String infoText = getInfoText();
		if (infoText != null) {
			infoTextLabel = new Label(container, SWT.NONE);
			infoTextLabel.setLayoutData(
					GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER)
							.grab(true, false).span(5, 1).create());
			infoTextLabel.setText("  Info: " + infoText);
		}
		final String beschreibungText = getBeschreibungsText();
		if (beschreibungText != null) {
			beschreibungsTextLabel = new Label(container, SWT.NONE);
			beschreibungsTextLabel.setLayoutData(
					GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER)
							.grab(true, false).span(5, 1).create());
			beschreibungsTextLabel.setText("  Beschreibung: " + infoText);
		}
		if (doModel != null && doModel.getName() != null
				&& doModel.getName().length() > 0) {
			doModelNameLabel = new Label(container, SWT.NONE);
			doModelNameLabel.setLayoutData(
					GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER)
							.grab(true, false).span(5, 1).create());
			doModelNameLabel.setText("  Model: " + doModel.getName());
		}

		label = new Label(container, SWT.NONE);
		label.setText("B:");
		b = new Label(container, SWT.NONE);
		b.setLayoutData(
				GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER)
						.grab(true, false).span(5, 1).create());

		final GridData gd = GridDataFactory.swtDefaults()
				.align(SWT.FILL, SWT.CENTER).create();
		gd.minimumWidth = 100;

		label = new Label(container, SWT.NONE);
		label.setText("QKfz:");
		qKfz = new Label(container, SWT.NONE);
		qKfz.setLayoutData(gd);

		label = new Label(container, SWT.NONE);
		label.setText("QPkw:");
		qPkw = new Label(container, SWT.NONE);
		qPkw.setLayoutData(gd);

		label = new Label(container, SWT.NONE);
		label.setText("QLkw:");
		qLkw = new Label(container, SWT.NONE);
		qLkw.setLayoutData(gd);

		label = new Label(container, SWT.NONE);
		label.setText("VKfz:");
		vKfz = new Label(container, SWT.NONE);
		vKfz.setLayoutData(gd);

		label = new Label(container, SWT.NONE);
		label.setText("VPkw:");
		vPkw = new Label(container, SWT.NONE);
		vPkw.setLayoutData(gd);

		label = new Label(container, SWT.NONE);
		label.setText("VLkw:");
		vLkw = new Label(container, SWT.NONE);
		vLkw.setLayoutData(gd);

		refresh(getSystemObjekt().getOdVerkehrsDatenKurzZeitMq()
				.getDatum(OdVerkehrsDatenKurzZeitMq.Aspekte.Analyse));
		getSystemObjekt().getOdVerkehrsDatenKurzZeitMq().addUpdateListener(
				OdVerkehrsDatenKurzZeitMq.Aspekte.Analyse, this);
	}

	private void createFahrStreifenViewer(final Composite parent) {
		final MessQuerschnitt mq = (MessQuerschnitt) getSystemObjekt();
		final Collection<FahrStreifen> fahrStreifen = mq.getFahrStreifen();

		final Composite tableComposite = new Composite(parent, SWT.NONE);
		final TableColumnLayout tableColumnLayout = new TableColumnLayout();
		tableComposite.setLayout(tableColumnLayout);

		fsTable = new TableViewer(tableComposite, SWT.NONE);
		int tableHeight = fsTable.getTable().getHeaderHeight();
		tableHeight += fsTable.getTable().getItemHeight()
				* (fahrStreifen.size() + 1);
		tableHeight += 5;
		tableComposite.setLayoutData(GridDataFactory.swtDefaults()
				.align(SWT.FILL, SWT.FILL).grab(true, true)
				.hint(SWT.DEFAULT, tableHeight).create());
		fsTable.getTable().setHeaderVisible(true);
		fsTable.setContentProvider(new ArrayContentProvider());

		TableViewerColumn col = new TableViewerColumn(fsTable, SWT.NONE);
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(150, 200, true));
		col.getColumn().setText("Fahrstreifen");
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				return datum.dGetDatensatz().getSystemObjekt().getName();
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("Zeit");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(140, 140, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;

				return mqDateFormater.format(datum.dGetZeitstempel().getTime());
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("Status");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(85, 85, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				return datum.dGetDatenStatus().toString();
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("B");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(40, 40, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttProzent wert = datum.getB().getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("QKfz");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(75, 75, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttVerkehrsStaerkeIntervall wert = datum.getQKfz()
						.getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("QPkw");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(75, 75, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttVerkehrsStaerkeIntervall wert = datum.getQPkw()
						.getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("QLkw");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(75, 75, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttVerkehrsStaerkeIntervall wert = datum.getQLkw()
						.getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("VKfz");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(60, 60, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttGeschwindigkeit wert = datum.getVKfz().getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("VPkw");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(60, 60, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttGeschwindigkeit wert = datum.getVPkw().getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		col = new TableViewerColumn(fsTable, SWT.NONE);
		col.getColumn().setText("VLkw");
		tableColumnLayout.setColumnData(col.getColumn(),
				new ColumnWeightData(60, 60, true));
		col.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) element;
				final AttGeschwindigkeit wert = datum.getVLkw().getWert();
				return wert != null ? wert.toString() : "-";
			}

		});

		fsDaten = new TreeMap<>(new Comparator<FahrStreifen>() {

			@Override
			public int compare(final FahrStreifen fs1, final FahrStreifen fs2) {
				final KdFahrStreifen.Daten datum1 = fs1.getKdFahrStreifen()
						.getDatum();
				final KdFahrStreifen.Daten datum2 = fs2.getKdFahrStreifen()
						.getDatum();
				return datum1.getLage().compareTo(datum2.getLage());
			}

		});

		for (final FahrStreifen fs : fahrStreifen) {
			final OdVerkehrsDatenKurzZeitIntervall.Daten datum = fs
					.getOdVerkehrsDatenKurzZeitIntervall().getDatum(
							OdVerkehrsDatenKurzZeitIntervall.Aspekte.ExterneErfassung);
			fsDaten.put(fs, datum);
			fs.getOdVerkehrsDatenKurzZeitIntervall().addUpdateListener(
					OdVerkehrsDatenKurzZeitIntervall.Aspekte.ExterneErfassung,
					this);
		}
		fsTable.setInput(fsDaten);
	}

	@Override
	public boolean close() {
		getSystemObjekt().getOdVerkehrsDatenKurzZeitMq().removeUpdateListener(
				OdVerkehrsDatenKurzZeitMq.Aspekte.Analyse, this);
		if (fsDaten != null) {
			for (final FahrStreifen fs : fsDaten.keySet()) {
				fs.getOdVerkehrsDatenKurzZeitIntervall().removeUpdateListener(
						OdVerkehrsDatenKurzZeitIntervall.Aspekte.ExterneErfassung,
						this);
			}
		}
		return super.close();
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (event.getDatensatz() instanceof OdVerkehrsDatenKurzZeitMq) {
			final OdVerkehrsDatenKurzZeitMq.Daten datum = (OdVerkehrsDatenKurzZeitMq.Daten) event
					.getDatum();
			refresh(datum);
		} else if (event
				.getDatensatz() instanceof OdVerkehrsDatenKurzZeitIntervall) {
			final OdVerkehrsDatenKurzZeitIntervall.Daten datum = (OdVerkehrsDatenKurzZeitIntervall.Daten) event
					.getDatum();
			refresh(datum);
		}
	}

	private void refresh(final OdVerkehrsDatenKurzZeitMq.Daten datum) {
		if (getShell() == null || getShell().isDisposed()) {
			return;
		}

		getShell().getDisplay().asyncExec(new Runnable() {

			@Override
			public void run() {
				if (getShell() == null || getShell().isDisposed()) {
					return;
				}

				zeitstempel.setText(mqDateFormater
						.format(datum.dGetZeitstempel().getTime()));

				final Status datenStatus = datum.dGetDatenStatus();
				setInfoText(datenStatus.toString());

				if (datum.dContainsDaten()) {
					b.setText(datum.getB().getWert().toString());
					qKfz.setText(datum.getQKfz().getWert().toString());
					qPkw.setText(datum.getQPkw().getWert().toString());
					qLkw.setText(datum.getQLkw().getWert().toString());
					vKfz.setText(datum.getVKfz().getWert().toString());
					vPkw.setText(datum.getVPkw().getWert().toString());
					vLkw.setText(datum.getVLkw().getWert().toString());
				} else {
					b.setText("-");
					qKfz.setText("-");
					qPkw.setText("-");
					qLkw.setText("-");
					vKfz.setText("-");
					vPkw.setText("-");
					vLkw.setText("-");
				}
			}

		});
	}

	private void refresh(final OdVerkehrsDatenKurzZeitIntervall.Daten datum) {
		if (getShell() == null || getShell().isDisposed()) {
			return;
		}

		getShell().getDisplay().asyncExec(new Runnable() {

			@Override
			public void run() {
				if (getShell() == null || getShell().isDisposed()
						|| !datum.dContainsDaten()) {
					return;
				}

				fsDaten.put(
						(FahrStreifen) datum.dGetDatensatz().getSystemObjekt(),
						datum);
				fsTable.setInput(fsDaten.values());
			}

		});
	}

}
