/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.rdsmeldung;

import java.util.Date;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.netz.ui.DObjPopUp;
import de.bsvrz.buv.plugin.netz.ui.PopUpSizingHelper;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum.Status;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsNachrichtenDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsNachrichtenKlasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;

/**
 * Tooltip für RDS-Meldungen.
 *
 * @author BitCtrl Systems GmbH, bitcontrol
 *
 */
final class RdsMeldungsPopUp extends DObjPopUp<RdsMeldung>
		implements DatensatzUpdateListener {

	/** der Datensatz mit den anzuzeigenden Daten. */
	private final PdRdsMeldung rdsDatensatz;

	/** Die Widgets fuer die Anzeige der Datenelemente. */
	private Text meldungsTextLabel;
	private Label aktualisierungsZeit;
	private Label ablaufZeit;
	private Label datenStatusLabel;

	RdsMeldungsPopUp(final Shell shell, final RdsMeldung meldung,
			final DoModel doModel) {
		super(shell, meldung, doModel);
		rdsDatensatz = getSystemObjekt().getPdRdsMeldung();
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		final Composite composite = (Composite) super.createDialogArea(parent);

		composite.setLayout(new GridLayout(1, true));
		composite.setLayoutData(GridDataFactory.fillDefaults()
				.minSize(400, SWT.MULTI).grab(true, true).create());

		final GridData gd = new GridData(GridData.FILL_HORIZONTAL);

		meldungsTextLabel = new Text(composite, SWT.MULTI | SWT.WRAP);
		meldungsTextLabel.setEditable(false);
		meldungsTextLabel.setLayoutData(gd);

		final String infoText = getInfoText();
		if (infoText != null) {
			infoTextLabel = new Label(composite, SWT.NONE);
			infoTextLabel.setLayoutData(GridDataFactory.swtDefaults()
					.align(SWT.FILL, SWT.CENTER).create());
			infoTextLabel.setText("  Info: " + infoText);
		}
		final String beschreibungText = getBeschreibungsText();
		if (beschreibungText != null) {
			beschreibungsTextLabel = new Label(composite, SWT.NONE);
			beschreibungsTextLabel.setLayoutData(GridDataFactory.swtDefaults()
					.align(SWT.FILL, SWT.CENTER).create());
			beschreibungsTextLabel
					.setText("  Beschreibung: " + beschreibungText);
		}
		if (doModel != null && doModel.getName() != null
				&& doModel.getName().length() > 0) {
			doModelNameLabel = new Label(composite, SWT.NONE);
			doModelNameLabel.setLayoutData(GridDataFactory.swtDefaults()
					.align(SWT.FILL, SWT.CENTER).create());
			doModelNameLabel.setText("  Model: " + doModel.getName());
		}

		aktualisierungsZeit = new Label(composite, SWT.NONE);
		aktualisierungsZeit.setLayoutData(gd);
		ablaufZeit = new Label(composite, SWT.NONE);
		ablaufZeit.setLayoutData(gd);
		datenStatusLabel = new Label(composite, SWT.NONE);
		datenStatusLabel.setLayoutData(gd);

		datenSatzListenerAnmelden();

		return composite;
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		final PdRdsMeldung.Daten daten = rdsDatensatz.getDatum();

		getShell().getDisplay().asyncExec(new Runnable() {

			@Override
			public void run() {
				if (getShell() == null || getShell().isDisposed()) {
					return;
				}

				meldungsTextLabel.setText("<leerer Meldungstext>");
				aktualisierungsZeit
						.setText("Aktualisierungszeit:  nicht ermittelbar");
				ablaufZeit.setText("Ablaufzeit:  nicht ermittelbar");

				setInfoText(event.getDatum().dGetDatenStatus().toString());

				if (event.getDatum().dGetDatenStatus() == Status.DATEN) {

					long zeit = 0;
					if (daten.getVersion().getVerwaltungsInformationen()
							.getAktualisierungsZeit() != null) {
						zeit = daten.getVersion().getVerwaltungsInformationen()
								.getAktualisierungsZeit().getTime();
					}
					if (zeit > 0) {
						aktualisierungsZeit.setText("Aktualisierungszeit:  "
								+ FORMAT_DATUM_ZEIT.format(new Date(zeit)));
					}
					zeit = 0;
					if (daten.getVersion().getVerwaltungsInformationen()
							.getAblaufZeit() != null) {
						zeit = daten.getVersion().getVerwaltungsInformationen()
								.getAblaufZeit().getTime();
					}
					if (zeit > 0) {
						ablaufZeit.setText("Ablaufzeit:  "
								+ FORMAT_DATUM_ZEIT.format(new Date(zeit)));
					}

					final String strNummerText = getNachrichtenText(
							AttRdsNachrichtenKlasse.ZUSTAND_0_STRASSENNUMMER,
							daten);
					final String infoText = getNachrichtenText(
							AttRdsNachrichtenKlasse.ZUSTAND_2_INHALT, daten);
					final String strSegmentText = getNachrichtenText(
							AttRdsNachrichtenKlasse.ZUSTAND_1_SEGMENT, daten);
					String meldungsText = null;

					if (strNummerText != null) {
						meldungsText = strNummerText;
					}
					if (strSegmentText != null) {
						if (meldungsText != null) {
							meldungsText += " " + strSegmentText;
						} else {
							meldungsText = strSegmentText;
						}
					}

					String meldungUndInfo = "";
					if (meldungsText != null && meldungsText.length() > 0) {
						meldungUndInfo += meldungsText;
					}
					if (infoText != null && infoText.length() > 0) {
						if (meldungUndInfo.length() > 0) {
							meldungUndInfo += "\n";
						}
						meldungUndInfo += infoText;
					}

					meldungsTextLabel.setText(part(meldungUndInfo, 70));

				} else {
					datenStatusLabel
							.setText(daten.dGetDatenStatus().toString());
					datenStatusLabel.setForeground(ColorConstants.red);
				}

				PopUpSizingHelper.resize(aktualisierungsZeit);
				PopUpSizingHelper.resize(ablaufZeit);
				PopUpSizingHelper.resize(datenStatusLabel);

				getShell().pack();
				constrainShellSize();
			}
		});
	}

	/**
	 * Bricht einen langen Text mit Newlines um.
	 *
	 * @param longText
	 *            ein langer Text.
	 * @param maxLineLen
	 *            maximale Zeilenlaenge.
	 * @return umgebrochener Text.
	 */
	private static String part(final String longText, final int maxLineLen) {
		final StringBuffer result = new StringBuffer();

		if (longText != null) {
			final String[] words = longText.split(" ");
			for (int i = 0, currentLineLen = 0; i < words.length - 1; i++) {
				result.append(words[i]);
				currentLineLen += words[i].length();
				if (currentLineLen > maxLineLen) {
					result.append('\n');
					currentLineLen = 0;
				} else {
					result.append(' ');
				}
			}
			result.append(words[words.length - 1]);
		}

		return result.toString();
	}

	/**
	 * Liefert den fuer die uebergebene Klasse innerhalb der Meldung gelieferten
	 * Texteintrag. Wenn kein entsprechender Eintrag vorliegt, liefert die
	 * Funktion den Wert <code>null</code>.
	 *
	 * @param klasse
	 *            die Klasse der Meldungstextes.
	 * @param datum
	 *            der Datensatz aus der RdsMeldung.
	 * @return den ermittelten Text oder <code>null</code>.
	 */
	private String getNachrichtenText(final AttRdsNachrichtenKlasse klasse,
			final PdRdsMeldung.Daten datum) {
		String result = null;
		if (datum != null) {
			final Feld<AtlRdsNachrichtenDaten> nachrichtenDaten = datum
					.getVersion().getVerkehrsInformationen().getNachrichten()
					.getNachrichtenDaten();
			for (final AtlRdsNachrichtenDaten eintrag : nachrichtenDaten) {
				if (klasse == eintrag.getNachrichtenKlasse()) {
					result = eintrag.getNachrichtenText();
					break;
				}
			}
		}

		return result;
	}

	/**
	 * Fuehrt DatenABmeldung durch.
	 */
	private void datenSatzListenerAbmelden() {
		rdsDatensatz.removeUpdateListener(RdsConverter.ASP, this);
	}

	/**
	 * Fuehrt DatenANmeldung durch.
	 */
	private void datenSatzListenerAnmelden() {
		rdsDatensatz.addUpdateListener(RdsConverter.ASP, this);
	}

	@Override
	public boolean close() {
		datenSatzListenerAbmelden();
		return super.close();
	}

}
