/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.situation;

import java.util.Calendar;

import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.dobj.util.IDavZeitStempelManager;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Situation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.parameter.PdSituationsEigenschaften;

public abstract class SituationEmpfaenger implements DatensatzUpdateListener {

	private final SituationEditPart<? extends Situation, ? extends SituationFigure> editPart;

	private PointList streckenabschnitt;

	SituationEmpfaenger(
			final SituationEditPart<? extends Situation, ? extends SituationFigure> editPart) {
		this.editPart = editPart;
	}

	protected final SituationEditPart<? extends Situation, ? extends SituationFigure> getEditPart() {
		return editPart;
	}

	protected final Situation getSituation() {
		return editPart.getModel().getSystemObjekt();
	}

	protected final SituationFigure getFigure() {
		return editPart.getFigure();
	}

	protected final Projektion getProjektion() {
		return editPart.getProjektion();
	}

	protected final PointList getStreckenabschnitt() {
		return streckenabschnitt;
	}

	public void anmelden() {
		getSituation().getPdSituationsEigenschaften().addUpdateListener(this);
	}

	public void abmelden() {
		getSituation().getPdSituationsEigenschaften()
				.removeUpdateListener(this);
	}

	@Override
	public final void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (!editPart.isActive()) {
			return;
		}

		updateZeitstempelLetzterDatensatz(event);

		if (PlatformUI.getWorkbench().getDisplay() != null) {
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					handleDatensatzAktualisiert(event.getDatum());
					editPart.refreshLocation();
				}

			});
		}
	}

	private void updateZeitstempelLetzterDatensatz(
			final DatensatzUpdateEvent event) {
		final IDavZeitStempelManager davZeitStempelManager = editPart
				.getDavZeitStempelManager();
		if (event.getDatum().dContainsDaten()
				&& davZeitStempelManager != null) {
			davZeitStempelManager.setLetztenEmpfangenenZeitStempel(
					event.getDatum().dGetZeitstempel().getTime(),
					event.getObjekt().getSystemObject());
		}
	}

	protected void handleDatensatzAktualisiert(final OnlineDatum datum) {
		if (datum instanceof PdSituationsEigenschaften.Daten) {
			situationsEigenschaftenAktualisiert(
					(PdSituationsEigenschaften.Daten) datum);
		}
	}

	protected void situationsEigenschaftenAktualisiert(
			final PdSituationsEigenschaften.Daten datum) {
		if (datum.dContainsDaten()) {
			getFigure().setSituationGueltig(isValid(datum));
			streckenabschnitt = getStreckenabschnitt(datum);
		} else {
			getFigure().setSituationGueltig(false);
			streckenabschnitt = new PointList();
		}
	}

	protected abstract PointList getStreckenabschnitt(
			PdSituationsEigenschaften.Daten datum);

	private boolean isValid(final PdSituationsEigenschaften.Daten datum) {
		if (datum.getStrassenSegment().isEmpty()) {
			return false;
		}

		final long situationStart = datum.getStartZeit().getTime();
		final long situationEnd = situationStart + datum.getDauer().getTime();
		final long todayStart = getTodayStart();
		final long todayEnd = getTodayEnd();

		final boolean situationStartetHeute = todayStart <= situationStart
				&& situationStart <= todayEnd;
		final boolean situationEndetHeute = todayStart <= situationEnd
				&& situationEnd <= todayEnd;
		final boolean situationStartetInVergangenheitUndEndetInZukunft = situationStart <= todayStart
				&& todayEnd <= situationEnd;

		return situationStartetHeute || situationEndetHeute
				|| situationStartetInVergangenheitUndEndetInZukunft;
	}

	private long getTodayStart() {
		final Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTimeInMillis();
	}

	private long getTodayEnd() {
		final Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		cal.add(Calendar.HOUR_OF_DAY, 24);
		return cal.getTimeInMillis();
	}

}
