/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.situation;

import java.text.DateFormat;
import java.util.Date;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.netz.ui.DObjPopUp;
import de.bsvrz.buv.plugin.netz.ui.PopUpSizingHelper;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum.Status;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Situation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.parameter.PdSituationsEigenschaften;
import de.bsvrz.sys.funclib.bitctrl.modell.util.SituationUtil;

/**
 * Allgemeiner Tooltip eines SituationEditPart.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 *
 */
public final class SituationPopUp extends DObjPopUp<Situation>
		implements DatensatzUpdateListener {

	/** der Datensatz mit den anzuzeigenden Daten. */
	private final PdSituationsEigenschaften situationsDatensatz;

	/** die Widgets fuer die Anzeige der Datenelemente. */
	private Label zeitstempel;
	private Label strassenLabel;
	private Label vonKnotenLabel;
	private Label folgeKnotenLabel;
	private Label laengeLabel;
	private Label beginnLabel;
	private Label endeLabel;
	private Label dauerLabel;
	private Label datenStatusLabel;

	/**
	 * Initialisiert das Pop-Up.
	 */
	public SituationPopUp(final Shell shell, final Situation situation,
			final DoModel doModel) {
		super(shell, situation, doModel);
		situationsDatensatz = getSystemObjekt().getPdSituationsEigenschaften();
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		final Composite composite = (Composite) super.createDialogArea(parent);

		composite.setLayout(new GridLayout(1, true));
		composite.setLayoutData(GridDataFactory.fillDefaults()
				.minSize(400, SWT.DEFAULT).grab(true, true).create());

		final GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		zeitstempel = new Label(composite, SWT.NONE);
		zeitstempel.setLayoutData(gd);

		final String infoText = getInfoText();
		if (infoText != null) {
			infoTextLabel = new Label(composite, SWT.NONE);
			infoTextLabel.setLayoutData(gd);
			infoTextLabel.setText("  Info: " + infoText);
		}
		final String beschreibungText = getBeschreibungsText();
		if (beschreibungText != null) {
			beschreibungsTextLabel = new Label(composite, SWT.NONE);
			beschreibungsTextLabel.setLayoutData(gd);
			beschreibungsTextLabel
					.setText("  Beschreibung: " + beschreibungText);
		}
		if (doModel != null && doModel.getName() != null
				&& doModel.getName().length() > 0) {
			doModelNameLabel = new Label(composite, SWT.NONE);
			doModelNameLabel.setLayoutData(gd);
			doModelNameLabel.setText("  Model: " + doModel.getName());
		}

		strassenLabel = new Label(composite, SWT.NONE);
		strassenLabel.setLayoutData(gd);
		vonKnotenLabel = new Label(composite, SWT.NONE);
		vonKnotenLabel.setLayoutData(gd);
		folgeKnotenLabel = new Label(composite, SWT.NONE);
		folgeKnotenLabel.setLayoutData(gd);
		laengeLabel = new Label(composite, SWT.NONE);
		laengeLabel.setLayoutData(gd);
		beginnLabel = new Label(composite, SWT.NONE);
		beginnLabel.setLayoutData(gd);
		endeLabel = new Label(composite, SWT.NONE);
		endeLabel.setLayoutData(gd);
		dauerLabel = new Label(composite, SWT.NONE);
		dauerLabel.setLayoutData(gd);
		datenStatusLabel = new Label(composite, SWT.NONE);
		datenStatusLabel.setLayoutData(gd);

		datenSatzListenerAnmelden();

		return composite;
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		final PdSituationsEigenschaften.Daten daten = situationsDatensatz
				.getDatum();

		getShell().getDisplay().asyncExec(new Runnable() {

			@Override
			public void run() {
				if (getShell() == null || getShell().isDisposed()) {
					return;
				}

				zeitstempel.setText(
						"Zeit: " + FORMAT_DATUM_ZEIT_GENAU.format(new Date(
								event.getDatum().dGetZeitstempel().getTime())));

				strassenLabel.setText("Straße: nicht ermittelbar");
				vonKnotenLabel.setText("Vonknoten: nicht ermittelbar");
				folgeKnotenLabel.setText("Folgeknoten: nicht ermittelbar");
				laengeLabel.setText("Länge: unbekannt");
				beginnLabel.setText("Beginn: unbekannt");
				endeLabel.setText("Ende: unbekannt");
				dauerLabel.setText("Dauer: unbekannt");

				setInfoText(event.getDatum().dGetDatenStatus().toString());

				if (event.getDatum().dGetDatenStatus() == Status.DATEN) {

					datenStatusLabel.setText("");

					final Strasse strasse = SituationUtil.getStrasse(daten);
					if (strasse != null) {
						strassenLabel.setText("Straße: " + strasse.getName());
					}

					StrassenKnoten knoten = SituationUtil.getVonKnoten(daten);
					if (knoten != null) {
						vonKnotenLabel
								.setText("Vonknoten: " + knoten.getName());
					}

					knoten = SituationUtil.getFolgeKnoten(daten);
					if (knoten != null) {
						folgeKnotenLabel
								.setText("Folgeknoten: " + knoten.getName());
					}

					laengeLabel.setText("Länge: " + daten.getLaenge());
					beginnLabel.setText("Beginn: " + FORMAT_DATUM_ZEIT
							.format(new Date(daten.getStartZeit().getTime())));
					endeLabel.setText("Ende: " + FORMAT_DATUM_ZEIT
							.format(new Date(daten.getStartZeit().getTime()
									+ daten.getDauer().getTime())));

					dauerLabel.setText(
							"Dauer: " + getDauer(daten.getDauer().getTime()));
				} else {
					datenStatusLabel
							.setText(daten.dGetDatenStatus().toString());
					datenStatusLabel.setForeground(ColorConstants.red);
				}

				PopUpSizingHelper.resize(zeitstempel);
				PopUpSizingHelper.resize(strassenLabel);
				PopUpSizingHelper.resize(vonKnotenLabel);
				PopUpSizingHelper.resize(folgeKnotenLabel);
				PopUpSizingHelper.resize(laengeLabel);
				PopUpSizingHelper.resize(beginnLabel);
				PopUpSizingHelper.resize(endeLabel);
				PopUpSizingHelper.resize(dauerLabel);

				getShell().pack();
				constrainShellSize();
			}
		});
	}

	private static final long SEC_MS = 1000L;

	private static final long MIN_MS = SEC_MS * 60L;

	private static final long HOUR_MS = MIN_MS * 60L;

	private static final long DAY_MS = HOUR_MS * 24L;

	private static final long WEEK_MS = DAY_MS * 7L;

	private static final long FOUR_WEEKS_MS = WEEK_MS * 4L;

	private static final long MONTH_MS = DAY_MS * 30L;

	private static final long YEAR_MS = DAY_MS * 365L;

	private static String getDauer(final long dauerInMillis) {
		String dauer = "";

		if (dauerInMillis < DAY_MS) {
			dauer = DateFormat.getTimeInstance()
					.format(new Date(dauerInMillis - HOUR_MS));
		} else if (dauerInMillis < FOUR_WEEKS_MS) {
			dauer = Math.round((double) dauerInMillis / DAY_MS) + " Tag(e)";
		} else if (dauerInMillis >= FOUR_WEEKS_MS && dauerInMillis < YEAR_MS) {
			dauer = Math.round((double) dauerInMillis / WEEK_MS) + " Woche(n)";
		} else {
			dauer = Math.round((double) dauerInMillis / MONTH_MS) + " Monat(e)";
		}

		return dauer;
	}

	private void datenSatzListenerAbmelden() {
		situationsDatensatz.removeUpdateListener(this);
	}

	private void datenSatzListenerAnmelden() {
		situationsDatensatz.addUpdateListener(this);
	}

	@Override
	public boolean close() {
		datenSatzListenerAbmelden();
		return super.close();
	}

}
