/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.stoerfall;

import static de.bsvrz.buv.plugin.dobj.util.StoerfallConstants.ATG_STOERFALL_ZUSTAND;
import static de.bsvrz.buv.plugin.dobj.util.StoerfallConstants.DEFAULT_ASP_STOERFALL_VERFAHREN;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.dialogs.ChoiceDialog;
import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.netz.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;

/**
 * Aktion zum Umschalten des Störfallverfahrens der Störfallindikatoren. Der
 * Benutzer wählt das Verfahren mittels Auswahldialog. Die Edit Parts können die
 * aktuelle Auswahl ({@link Aspect}) als Viewer Property
 * StoerfallConstants#VIEWER_PROPERTY_ASPECT_STOERFALL_VERFAHREN auslesen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class StoerfallVerfahrenAction extends Action {

	private final IWorkbenchPart part;

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part mit dem die Action arbeiten soll.
	 */
	public StoerfallVerfahrenAction(final IWorkbenchPart part) {
		super("Störfallverfahren");
		this.part = part;
	}

	@Override
	public void run() {
		final Shell shell = part.getSite().getShell();
		final ChoiceDialog dlg = new ChoiceDialog(shell,
				getAspects().toArray());
		dlg.setTitle("Störfallverfahren");
		dlg.setMessage("Bitte wählen Sie das gewünschte Störfallverfahren.");
		dlg.setInitialElementSelections(
				Collections.singletonList(getCurrentAspect()));

		if (dlg.open() == Window.OK) {
			final Aspect asp = (Aspect) dlg.getResult()[0];
			setCurrentAspect(asp);
		}
	}

	private Collection<Aspect> getAspects() {
		final DataModel model = getDataModel();
		final AttributeGroup atg = model
				.getAttributeGroup(ATG_STOERFALL_ZUSTAND);
		final List<Aspect> aspects = new ArrayList<>(atg.getAspects());
		Collections.sort(aspects,
				(asp1, asp2) -> asp1.toString().compareTo(asp2.toString()));
		return aspects;
	}

	private Aspect getCurrentAspect() {
		final Darstellung darstellung = part.getAdapter(Darstellung.class);
		final String stoerfallverfahren = darstellung.getStoerfallverfahren();
		Aspect result = null;
		if (stoerfallverfahren != null) {
			result = getDataModel().getAspect(stoerfallverfahren);
		}

		if (result == null) {
			result = getDataModel().getAspect(DEFAULT_ASP_STOERFALL_VERFAHREN);
		}

		Assert.isNotNull(result);
		return result;
	}

	private void setCurrentAspect(final Aspect asp) {
		final Darstellung darstellung = part.getAdapter(Darstellung.class);

		if (!Objects.equals(darstellung.getStoerfallverfahren(),
				asp.getPid())) {
			final CommandStack commandStack = part
					.getAdapter(CommandStack.class);
			final SetCommand cmd = new SetCommand(darstellung,
					DarstellungPackage.Literals.DARSTELLUNG__STOERFALLVERFAHREN,
					asp.getPid());
			commandStack.execute(cmd);
		}
	}

	private DataModel getDataModel() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();
		final DataModel model = dav.getDataModel();
		Assert.isNotNull(model);
		return model;
	}

}
