/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.stoerfall.anzeigeverfahren;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import de.bsvrz.buv.plugin.netz.CacheMenge;
import de.bsvrz.buv.plugin.netz.internal.NetzPlugin;
import de.bsvrz.buv.plugin.netz.internal.RahmenwerkService;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;

/**
 * Allgemeiner Provider fuer Darstellungsverfahren der Verkehrslage nach
 * SE-02.00.00.00.00-AFo-5.0.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 */
public abstract class AbstractAnzeigeVerfahrenPolygonProvider
		implements AnzeigeVerfahrenPolygonProvider {

	/**
	 * Zu ladende Caches.
	 */
	protected static final CacheMenge CACHES = new CacheMenge(RahmenwerkService
			.getService().getCacheService().getAnzeigeVerfahrenCache());

	private final Set<AnzeigeVerfahrenListener> listeners;
	private final Map<MessQuerschnittAllgemein, WGS84Polygon> mqToPolygon = new HashMap<>();

	private boolean initialisiert;

	AbstractAnzeigeVerfahrenPolygonProvider() {
		listeners = new HashSet<>();
	}

	protected final void initializeInBackground() {
		new Thread(createAnzeigeVerfahrenInitializer()).start();
	}

	protected abstract AnzeigeVerfahrenInitializer createAnzeigeVerfahrenInitializer();

	@Override
	public final WGS84Polygon getPolygonBy(final MessQuerschnittAllgemein mqa) {
		if (!initialisiert) {
			throw new RuntimeException(this + " noch nicht initialisiert");
		}
		final WGS84Polygon p = mqToPolygon.get(mqa);
		if (p == null) {
			return LEERES_POLYGON;
		}
		return p;
	}

	/**
	 * Benachrichtigt alle angemeldeten Listener und meldet sie dann ab.
	 */
	protected final void fireAnzeigeVerfahrenInitialisiertAndRemoveAllListeners() {
		initialisiert = true;
		synchronized (listeners) {
			for (final AnzeigeVerfahrenListener listener : listeners) {
				listener.anzeigeVerfahrenInitialisiert();
			}
			listeners.clear();
		}

	}

	@Override
	public final void addInitialisierungsListener(
			final AnzeigeVerfahrenListener listener) {
		synchronized (listeners) {
			if (initialisiert) {
				listener.anzeigeVerfahrenInitialisiert();
			} else {
				listeners.add(listener);
			}
		}
	}

	@Override
	public final String getId() {
		return toString();
	}

	/**
	 * Erfragt eine Map mit proallgemeinen Messquerschnitt ein Polygon
	 * (Punktliste).
	 *
	 * @return Map: Ordnet jedem allgemeinen Messquerschnitt ein Polygon
	 *         (Punktliste) zu.
	 */
	public final Map<MessQuerschnittAllgemein, WGS84Polygon> getMqToPolygon() {
		return mqToPolygon;
	}

	protected abstract class AnzeigeVerfahrenInitializer implements Runnable {

		@Override
		public final void run() {
			waitForCaches();
			initialize();

		}

		private void waitForCaches() {
			while (!CACHES.isInitialisiert()) {
				try {
					Thread.sleep(1000L);
				} catch (final InterruptedException ex) {
					NetzPlugin.getDefault().getLogger().error(
							"Initialisierung des Anzeigeverfahren II abgebrochen.",
							ex);
				}
			}
		}

		protected abstract void initialize();

	}

}
