/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.stoerfall.anzeigeverfahren;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.dialogs.SelectionDialog;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.model.AnzeigeVerfahren;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.netz.internal.NetzPlugin;

/**
 * Action zum Umschalten des Anzeigeverfahrens der Verkehrslage für
 * Messquerschnitte. Der Benutzer kann das Verfahren in einem Auswahldialog
 * auswählen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class AnzeigeVerfahrenAction extends Action {

	private final IWorkbenchPart part;

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part mit dem die Action arbeiten soll.
	 */
	public AnzeigeVerfahrenAction(final IWorkbenchPart part) {
		super("Anzeigeverfahren");
		this.part = part;
	}

	@Override
	public void run() {
		final Shell shell = part.getSite().getShell();
		final AuswahlDialog dlg = new AuswahlDialog(shell);
		dlg.setTitle("Anzeigeverfahren");
		dlg.setMessage("Bitte wählen Sie das gewünschte Anzeigeverfahren.");
		dlg.setInitialElementSelections(
				Collections.singletonList(getCurrentAnzeigeVerfahren()));

		if (dlg.open() == Window.OK) {
			final AnzeigeVerfahren anzeigeVerfahren = (AnzeigeVerfahren) dlg
					.getResult()[0];
			setCurrentAnzeigeVerfahren(anzeigeVerfahren);

		}
	}

	private AnzeigeVerfahren getCurrentAnzeigeVerfahren() {
		final Darstellung darstellung = part.getAdapter(Darstellung.class);
		AnzeigeVerfahren anzeigeVerfahren = darstellung.getAnzeigeVerfahren();

		if (anzeigeVerfahren == null) {
			anzeigeVerfahren = AnzeigeVerfahren.VERFAHREN_I;
		}
		Assert.isNotNull(anzeigeVerfahren);
		return anzeigeVerfahren;
	}

	private void setCurrentAnzeigeVerfahren(
			final AnzeigeVerfahren anzeigeVerfahren) {
		final Darstellung darstellung = part.getAdapter(Darstellung.class);

		if (!Objects.equals(darstellung.getAnzeigeVerfahren(),
				anzeigeVerfahren)) {
			final CommandStack commandStack = part
					.getAdapter(CommandStack.class);
			final SetCommand cmd = new SetCommand(darstellung,
					DarstellungPackage.Literals.DARSTELLUNG__ANZEIGE_VERFAHREN,
					anzeigeVerfahren);
			commandStack.execute(cmd);
		}
	}

	private static final class AuswahlDialog extends SelectionDialog {

		private Object anzeigeVerfahren;
		private Image legendeImage;

		private AuswahlDialog(final Shell parentShell) {
			super(parentShell);
		}

		@Override
		protected Control createDialogArea(final Composite parent) {
			final Composite composite = (Composite) super.createDialogArea(
					parent);
			createMessageArea(composite);

			for (final Object obj : AnzeigeVerfahren.values()) {
				final Button radioButton = new Button(composite, SWT.RADIO);
				radioButton.setText(obj.toString());

				final List<?> initialElements = getInitialElementSelections();
				if (initialElements != null && !initialElements.isEmpty()
						&& initialElements.get(0).equals(obj)) {
					radioButton.setSelection(true);
					anzeigeVerfahren = obj;
				}

				radioButton.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						anzeigeVerfahren = obj;
					}

				});
			}

			final ImageDescriptor imageDescriptor = NetzPlugin.getDefault()
					.getImageDescriptor("images/AnzeigeVerfahrenLegende.png");
			legendeImage = imageDescriptor.createImage();

			final Label l = new Label(composite, SWT.NONE);
			l.setImage(legendeImage);

			return composite;
		}

		@Override
		protected void okPressed() {
			if (anzeigeVerfahren == null) {
				return;
			}

			setResult(Collections.singletonList(anzeigeVerfahren));
			super.okPressed();
		}

		@Override
		public boolean close() {
			legendeImage.dispose();
			return super.close();
		}

	}

}
