/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.stoerfall.anzeigeverfahren;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.swt.widgets.Display;

import de.bsvrz.buv.plugin.darstellung.model.AnzeigeVerfahren;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.buv.plugin.netz.stoerfall.StoerfallIndikatorEditPart;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StoerfallIndikator;

/**
 * Basisklasse für Störfallindikatoren, die als zusätzliche Eigenschaft das
 * Anzeigeverfahren besitzen. Das Anzeigeverfahren bezieht sich auf einen
 * Messquerschnitt.
 *
 * <p>
 * Das Anzeigeverfahren bestimmt, ob das Polygon des Störfallindikators vor,
 * nach oder über der Position des Messquerschnitts liegt. Default ist das
 * Verfahren&nbsp;I.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public abstract class StoerfallIndikatorMitAnzeigeVerfahrenEditPart<T extends StoerfallIndikator>
		extends StoerfallIndikatorEditPart<T> {

	private final AnzeigeVerfahrenListener anzeigeVerfahrenListener = new AnzeigeVerfahrenListener() {

		@Override
		public void anzeigeVerfahrenInitialisiert() {
			initialisiert = true;
			Display.getDefault().asyncExec(new Runnable() {

				@Override
				public void run() {
					refreshVisuals();
				}

			});
		}

	};

	private final PropertyChangeListener viewerPropertyChangeListener = new PropertyChangeListener() {

		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			if (!AnzeigeVerfahren.class.getName()
					.equals(evt.getPropertyName())) {
				return;
			}

			final AnzeigeVerfahren anzeigeVerfahren = (AnzeigeVerfahren) evt
					.getNewValue();
			anzeigeVerfahrenPolygonProvider = getPolygonProvider(
					anzeigeVerfahren);
			initialisiert = false;
			anzeigeVerfahrenPolygonProvider
					.addInitialisierungsListener(anzeigeVerfahrenListener);
		}

	};

	private AnzeigeVerfahrenPolygonProvider anzeigeVerfahrenPolygonProvider;
	private boolean initialisiert;

	@Override
	public void activate() {
		super.activate();

		getViewer().addPropertyChangeListener(viewerPropertyChangeListener);

		anzeigeVerfahrenPolygonProvider = getPolygonProvider(
				AnzeigeVerfahren.VERFAHREN_I);
		anzeigeVerfahrenPolygonProvider
				.addInitialisierungsListener(anzeigeVerfahrenListener);
	}

	@Override
	public void deactivate() {
		getViewer().removePropertyChangeListener(viewerPropertyChangeListener);

		super.deactivate();
	}

	private AnzeigeVerfahrenPolygonProvider getPolygonProvider(
			final AnzeigeVerfahren anzeigeVerfahren) {
		switch (anzeigeVerfahren) {
		case VERFAHREN_I:
			return AnzeigeVerfahren1PolygonProvider.INSTANCE;
		case VERFAHREN_II:
			return AnzeigeVerfahren2PolygonProvider.INSTANCE;
		case VERFAHREN_III:
			return AnzeigeVerfahren3PolygonProvider.INSTANCE;
		default:
			throw new IllegalArgumentException(
					"Unbekannters Anzeigeverfahren: " + anzeigeVerfahren);
		}

	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		refreshGeometrie();
	}

	private void refreshGeometrie() {
		getFigure().setBasePoints(getPolygon());
	}

	private PointList getPolygon() {
		final PointList punkte = new PointList();

		if (initialisiert && getMessQuerschnitt() != null) {
			final WGS84Polygon polygon = anzeigeVerfahrenPolygonProvider
					.getPolygonBy(getMessQuerschnitt());

			final Projektion projektion = getProjektion();
			for (final WGS84Punkt p : polygon.getKoordinaten()) {
				final Point pMitProjektion = projektion.umrechnenWeltNachModel(
						new PrecisionPoint(p.getLaenge(), p.getBreite()));
				punkte.addPoint(pMitProjektion);
			}
		}

		return punkte;
	}

	/**
	 * Gibt den Messquerschnitt zurück, der als Basis für die Berechnung des
	 * Polygons der Figure mit Hilfe des Anzeigeverfahrens verwendet wird.
	 */
	protected abstract MessQuerschnittAllgemein getMessQuerschnitt();

}
