/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.strassenknoten;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.PrecisionRectangle;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.GraphicalEditPart;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.HintergrundfarbeMediator;
import de.bsvrz.buv.plugin.dobj.decorator.VordergrundfarbeMediator;
import de.bsvrz.buv.plugin.dobj.editparts.BitCtrlDoModelEditPart;
import de.bsvrz.buv.plugin.dobj.legende.DoLegendeBaustein;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.buv.plugin.netz.internal.NetzPlugin;
import de.bsvrz.buv.plugin.netz.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.netz.model.StrassenKnotenDoModel;
import de.bsvrz.buv.plugin.netz.model.StrassenKnotenDoTyp;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;
import de.bsvrz.buv.rw.bitctrl.CacheService;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdLinienKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcPunkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcCache;

/**
 * MVC-Controller für Systemobjekte vom Typ <em>StraßenKnoten</em>.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class StrassenKnotenEditPart
		extends BitCtrlDoModelEditPart<StrassenKnoten, StrassenKnotenFigure> {

	private DoLegendeBaustein legendeBaustein;

	private final PropertyChangeListener propertyListener = new PropertyChangeListener() {

		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			final String prop = evt.getPropertyName();

			if (Projektion.class.toString().equals(prop)) {
				cleanModellKoordinaten();
				refreshVisuals();
			}
		}

	};

	private Point location;
	private TmcCache tmcCache;
	private PropertyChangeListener tmcCacheListener;

	@Override
	public StrassenKnotenDoModel getModel() {
		return (StrassenKnotenDoModel) super.getModel();
	}

	private StrassenKnotenDoTyp getDoTyp() {
		return (StrassenKnotenDoTyp) getModel().getDoTyp();
	}

	@Override
	protected StrassenKnotenFigure createFigure() {
		return new StrassenKnotenFigure();
	}

	@Override
	public void activate() {
		super.activate();

		if (isEditor()) {
			getViewer().addPropertyChangeListener(propertyListener);
		}

	}

	@Override
	public void deactivate() {
		if (isEditor()) {
			getViewer().removePropertyChangeListener(propertyListener);
		}

		if ((tmcCache != null) && (tmcCacheListener != null)) {
			tmcCache.removePropertyChangeListener(AbstractCache.PROP_STATE,
					tmcCacheListener);
			tmcCacheListener = null;
		}

		disposeLegende();

		super.deactivate();
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		final Object notifier = notification.getNotifier();
		if (notification.getEventType() != Notification.REMOVING_ADAPTER
				&& (notifier instanceof StrassenKnotenDoModel
						|| notifier instanceof StrassenKnotenDoTyp)) {
			refreshVisuals();
		}
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		refreshLocation();
		refreshDurchmesser();

		new VordergrundfarbeMediator(this).mediate();
		new HintergrundfarbeMediator(this).mediate();
	}

	private void refreshLocation() {
		final DoFigure f = getFigure();
		final Point p = getModel().getAbsoluteLocation();
		if (p != null) {
			f.setHotspot(p);
		} else {
			f.setHotspot(getLocation());
			((GraphicalEditPart) getParent()).setLayoutConstraint(this, f,
					f.getBounds());
		}
	}

	private void refreshDurchmesser() {
		final StrassenKnotenDoTyp doTyp = getDoTyp();
		final StrassenKnotenFigure f = getFigure();
		f.setAusgangsdurchmesserInMeter(doTyp.getAusgangsdurchmesserInMeter());
		f.setMinimalerDurchmesserInPixel(
				doTyp.getMinimalerDurchmesserInPixel());
	}

	/**
	 * Bestimmt die Position des Straßenknoten als Mittelpunkt der enthaltenen
	 * inneren Straßensegmente.
	 *
	 * @return <code>(0,0)</code>, wenn die Position nicht bestimmt werden kann.
	 */
	private Point getLocation() {
		if (location == null) {
			Point p = getLcl();
			if (p == null) {
				if (NetzPlugin.getDefault().isDebugging()) {
					NetzPlugin.getDefault().getLogger()
							.error("Straßenknoten ohne Mittelpunkt: "
									+ getModel().getSystemObject());
				}

				p = getMittelpunkt();
			}

			location = p;
		}

		return location;
	}

	/**
	 * Gibt die Position des Straßenknoten in Bildschirmkoordinaten nach LCL
	 * (TMC Location Code List) zurück.
	 *
	 * @return der Punkt oder <code>null</code>, wenn aus der LCL kein
	 *         Mittelpunkt bestimmt werden kann.
	 */
	private Point getLcl() {

		final CacheService cacheService = RahmenwerkService.getService()
				.getCacheService();
		tmcCache = cacheService.getTmcCache(cacheService.getDefaultNetzPid());
		if (tmcCache.isInitialisiert()) {
			final TmcPunkt tmcPunkt = tmcCache
					.getTmcLocationCode(getModel().getSystemObjekt());

			if (tmcPunkt != null) {
				vorladen(TmcPunkt.PID, KdPunktKoordinaten.PID);

				final KdPunktKoordinaten.Daten punktKoordinatenDatum = tmcPunkt
						.getKdPunktKoordinaten().getDatum();

				if (punktKoordinatenDatum != null) {
					final double x = punktKoordinatenDatum.getX().getValue();
					final double y = punktKoordinatenDatum.getY().getValue();
					return getProjektion()
							.umrechnenWeltNachModel(new PrecisionPoint(x, y));
				}
			}

			if (NetzPlugin.getDefault().isDebugging()) {
				NetzPlugin.getDefault().getLogger().error(
						"Location Code des Straßenknoten ist nicht bestimmbar: "
								+ getModel().getSystemObjekt());
			}

		} else {
			tmcCacheListener = new PropertyChangeListener() {

				@Override
				public void propertyChange(final PropertyChangeEvent evt) {
					if (AbstractCache.PROP_STATE
							.equals(evt.getPropertyName())) {
						if (tmcCache.isInitialisiert()) {
							getViewer().getControl().getDisplay()
									.asyncExec(new Runnable() {
										@Override
										public void run() {
											refreshVisuals();
											tmcCache.removePropertyChangeListener(
													AbstractCache.PROP_STATE,
													tmcCacheListener);
											tmcCacheListener = null;
										}
									});
						}
					}
				}
			};

			tmcCache.addPropertyChangeListener(AbstractCache.PROP_STATE,
					tmcCacheListener);
		}

		return null;
	}

	/**
	 * Gibt die Position des Straßenknoten in Bildschirmkoordinaten als
	 * Mittelpunkt der Rechteckfläche um alle inneren Straßensegmente des
	 * Straßenknoten.
	 *
	 * @return der Punkt, nie <code>null</code>.
	 */
	private Point getMittelpunkt() {
		vorladen(StrassenSegment.PID, KdBestehtAusLinienObjekten.PID);
		vorladen(StrassenTeilSegment.PID, KdLinienKoordinaten.PID);

		Rectangle rect = null;
		for (final InneresStrassenSegment iss : getModel().getSystemObjekt()
				.getInnereStrassenSegmente()) {
			if (rect == null) {
				rect = getBounds(iss);
			} else {
				rect.union(getBounds(iss));
			}
		}

		if (rect != null) {
			return getProjektion().umrechnenWeltNachModel(rect.getCenter());
		}

		if (NetzPlugin.getDefault().isDebugging()) {
			NetzPlugin.getDefault().getLogger()
					.error("Straßenknoten ohne innere Straßensegmente: "
							+ getModel().getSystemObjekt());
		}
		return new Point();
	}

	private Rectangle getBounds(final StrassenSegment strassenSegment) {
		PrecisionRectangle rect = null;
		for (final Linie linie : strassenSegment.getKdBestehtAusLinienObjekten()
				.getDatum().getLinienReferenz()) {
			final StrassenTeilSegment sts = (StrassenTeilSegment) linie;
			final KdLinienKoordinaten.Daten koordinaten = sts
					.getKdLinienKoordinaten().getDatum();
			for (int i = 0; i < koordinaten.getX().size(); ++i) {
				final PrecisionPoint p = new PrecisionPoint(
						koordinaten.getX().get(i).doubleValue(),
						koordinaten.getY().get(i).doubleValue());

				if (rect == null) {
					rect = new PrecisionRectangle();
					rect.setLocation(p);
					rect.setSize(0, 0);
				} else {
					rect.union(p);
				}
			}
		}
		return rect;
	}

	/**
	 * Entfernt alle gecachten Modellkoordinaten (Pixelkoordinaten), damit diese
	 * neu berechnet werden.
	 */
	protected void cleanModellKoordinaten() {
		location = null;
	}

	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") final Class key) {
		if (ILegendeBaustein.class.equals(key)) {
			return getLegendeBaustein();
		}

		return super.getAdapter(key);
	}

	private ILegendeBaustein getLegendeBaustein() {
		disposeLegende();
		legendeBaustein = new StrassenKnotenLegendeBaustein(getDoTyp(),
				createFigure());
		return legendeBaustein;
	}

	private void disposeLegende() {
		if (legendeBaustein != null) {
			legendeBaustein.dispose();
			legendeBaustein = null;
		}
	}
}
