/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.strassensegment;

import java.util.List;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandZoomMediator;
import de.bsvrz.buv.plugin.dobj.decorator.LinienstaerkeZoomMediator;
import de.bsvrz.buv.plugin.dobj.decorator.VordergrundfarbeMediator;
import de.bsvrz.buv.plugin.dobj.editparts.LinieEditPart;
import de.bsvrz.buv.plugin.dobj.tools.IStreckenSelektierbar;
import de.bsvrz.buv.plugin.netz.LinieFigure;
import de.bsvrz.buv.plugin.netz.LinieLegendeBaustein;
import de.bsvrz.buv.plugin.netz.model.StrassenSegmentDoModel;
import de.bsvrz.buv.plugin.netz.model.StrassenSegmentDoTyp;
import de.bsvrz.buv.plugin.netz.model.StrassenTyp;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdLinienKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AttStrassenTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AttStrassenTypErweitert;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasseErweitert;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenSegment.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;

/**
 * MVC-Controller für ein Straßensegment.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @author BitCtrl Systems GmbH, Thierfelder
 */
public final class StrassenSegmentEditPart
		extends LinieEditPart<StrassenSegment, LinieFigure>
		implements IStreckenSelektierbar {

	private LinieLegendeBaustein legendeBaustein;

	@Override
	public StrassenSegmentDoModel getModel() {
		return (StrassenSegmentDoModel) super.getModel();
	}

	private StrassenSegmentDoTyp getDoTyp() {
		return (StrassenSegmentDoTyp) getModel().getDoTyp();
	}

	@Override
	protected LinieFigure createFigure() {
		return new LinieFigure();
	}

	@Override
	protected List<Point> getWgs84Points() {
		vorladen(StrassenSegment.PID, KdBestehtAusLinienObjekten.PID);
		vorladen(StrassenTeilSegment.PID, KdLinienKoordinaten.PID);

		return super.getWgs84Points();
	}

	@Override
	public void deactivate() {
		disposeLegende();

		super.deactivate();
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		final Object notifier = notification.getNotifier();
		if (notification.getEventType() != Notification.REMOVING_ADAPTER
				&& (notifier instanceof StrassenSegmentDoTyp
						|| notifier instanceof StrassenSegmentDoModel)) {
			refreshVisuals();
		}
	}

	private double getZoomSkalierung() {
		final Darstellung darstellung = getDarstellung();
		if (darstellung != null) {
			return darstellung.getZoomSkalierung();
		}

		// fallback
		return 1.0;
	}

	private StrassenTyp getStrassenTyp() {
		vorladen(StrassenSegment.PID, KdStrassenSegment.PID);
		vorladen(Strasse.PID, KdStrasse.PID);

		final StrassenSegment segment = getModel().getSystemObjekt();
		final Daten kdSSegmentDatum = segment.getKdStrassenSegment().getDatum();
		if (kdSSegmentDatum != null) {
			final Strasse strasse = kdSSegmentDatum.getGehoertZuStrasse();

			if (strasse == null) {
				return StrassenTyp.SONSTIGE;
			}

			final KdStrasse.Daten kdStrasseDatum = strasse.getKdStrasse()
					.getDatum();
			final KdStrasseErweitert.Daten kdStrasseErweitertDatum = strasse
					.getKdStrasseErweitert().getDatum();
			if (kdStrasseDatum != null) {
				final AttStrassenTyp typ = kdStrasseDatum.getTyp();
				if (AttStrassenTyp.ZUSTAND_1_AUTOBAHN.equals(typ)) {
					return StrassenTyp.AUTOBAHN;
				} else if (AttStrassenTyp.ZUSTAND_2_BUNDESSTRASSE.equals(typ)) {
					return StrassenTyp.BUNDESSTRASSE;
				} else if (AttStrassenTyp.ZUSTAND_3_LANDSTRASSE.equals(typ)) {
					return StrassenTyp.LANDSTRASSE;
				} else if (AttStrassenTyp.ZUSTAND_4_KREISSTRASSE.equals(typ)) {
					return StrassenTyp.KREISSTRASSE;
				}
			} else if (kdStrasseErweitertDatum != null) {
				final AttStrassenTypErweitert typ = kdStrasseErweitertDatum
						.getTyp();
				if (AttStrassenTypErweitert.ZUSTAND_1_AUTOBAHN.equals(typ)) {
					return StrassenTyp.AUTOBAHN;
				} else if (AttStrassenTypErweitert.ZUSTAND_2_BUNDESSTRASSE
						.equals(typ)) {
					return StrassenTyp.BUNDESSTRASSE;
				} else if (AttStrassenTypErweitert.ZUSTAND_3_LANDES_ODER_STAATSSTRASSE
						.equals(typ)) {
					return StrassenTyp.LANDSTRASSE;
				} else if (AttStrassenTypErweitert.ZUSTAND_4_KREISSTRASSE
						.equals(typ)) {
					return StrassenTyp.KREISSTRASSE;
				}
			}
		}
		return StrassenTyp.SONSTIGE;
	}

	@Override
	protected void refreshVisuals() {

		super.refreshVisuals();

		refreshBasePoints();
		new VordergrundfarbeMediator(this).mediate();
		new LinienabstandZoomMediator(this).mediate();
		new LinienstaerkeZoomMediator(this).mediate();

		final StrassenTyp strassenTyp = getStrassenTyp();
		switch (strassenTyp) {
		case AUTOBAHN:
			if (!getDoTyp().isDefaultDarstellungAutobahn()) {
				setzeAutobahnAttribute();
			}
			break;
		case BUNDESSTRASSE:
			if (!getDoTyp().isDefaultDarstellungBundesstrasse()) {
				setzeBundesstrasseAttribute();
			}
			break;
		case LANDSTRASSE:
			if (!getDoTyp().isDefaultDarstellungLandstrasse()) {
				setzeLandstrasseAttribute();
			}
			break;
		case KREISSTRASSE:
			if (!getDoTyp().isDefaultDarstellungKreisstrasse()) {
				setzeKreisstrasseAttribute();
			}
			break;
		default:
			// nüscht
		}
	}

	private void setzeLandstrasseAttribute() {
		final double zoomSkalierung = getZoomSkalierung();
		final RGB rgb = getDoTyp().getVordergrundFarbeLandstrasse();
		if (rgb != null) {
			final Color color = getResourceManager().createColor(rgb);
			getFigure().setForegroundColor(color);
		}
		getFigure().setLinienabstand(getDoTyp().getLinienabstandLandstrasse());
		getFigure().setLinienstaerkeZoomverhalten(
				getDoTyp().getLinienstaerkeZoomverhaltenLandstrasse());
		getFigure().setLinienstaerke(getDoTyp().getLinienstaerkeLandstrasse());
		getFigure().setLinienstaerkeMaximum(
				getDoTyp().getLinienstaerkeMaximumLandstrasse());
		getFigure().setLinienstaerkeMinimum(
				getDoTyp().getLinienstaerkeMinimumLandstrasse());

		double maximaleZoomstufe = getDoTyp()
				.getLinienstaerkeMaximaleZoomstufeLandstrasse();
		maximaleZoomstufe /= 100;
		maximaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMaximaleZoomstufe(maximaleZoomstufe);

		double minimaleZoomstufe = getDoTyp()
				.getLinienstaerkeMinimaleZoomstufeLandstrasse();
		minimaleZoomstufe /= 100;
		minimaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMinimaleZoomstufe(minimaleZoomstufe);
	}

	private void setzeKreisstrasseAttribute() {
		final double zoomSkalierung = getZoomSkalierung();
		final RGB rgb = getDoTyp().getVordergrundFarbeKreisstrasse();
		if (rgb != null) {
			final Color color = getResourceManager().createColor(rgb);
			getFigure().setForegroundColor(color);
		}
		getFigure().setLinienabstand(getDoTyp().getLinienabstandKreisstrasse());
		getFigure().setLinienstaerkeZoomverhalten(
				getDoTyp().getLinienstaerkeZoomverhaltenKreisstrasse());
		getFigure().setLinienstaerke(getDoTyp().getLinienstaerkeKreisstrasse());
		getFigure().setLinienstaerkeMaximum(
				getDoTyp().getLinienstaerkeMaximumKreisstrasse());
		getFigure().setLinienstaerkeMinimum(
				getDoTyp().getLinienstaerkeMinimumKreisstrasse());
		double maximaleZoomstufe = getDoTyp()
				.getLinienstaerkeMaximaleZoomstufeKreisstrasse();
		maximaleZoomstufe /= 100;
		maximaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMaximaleZoomstufe(maximaleZoomstufe);

		double minimaleZoomstufe = getDoTyp()
				.getLinienstaerkeMinimaleZoomstufeKreisstrasse();
		minimaleZoomstufe /= 100;
		minimaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMinimaleZoomstufe(minimaleZoomstufe);
	}

	private void setzeBundesstrasseAttribute() {
		final double zoomSkalierung = getZoomSkalierung();
		final RGB rgb = getDoTyp().getVordergrundFarbeBundesstrasse();
		if (rgb != null) {
			final Color color = getResourceManager().createColor(rgb);
			getFigure().setForegroundColor(color);
		}
		getFigure()
				.setLinienabstand(getDoTyp().getLinienabstandBundesstrasse());
		getFigure().setLinienstaerkeZoomverhalten(
				getDoTyp().getLinienstaerkeZoomverhaltenBundesstrasse());
		getFigure()
				.setLinienstaerke(getDoTyp().getLinienstaerkeBundesstrasse());
		getFigure().setLinienstaerkeMaximum(
				getDoTyp().getLinienstaerkeMaximumBundesstrasse());
		getFigure().setLinienstaerkeMinimum(
				getDoTyp().getLinienstaerkeMinimumBundesstrasse());

		double maximaleZoomstufe = getDoTyp()
				.getLinienstaerkeMaximaleZoomstufeBundesstrasse();
		maximaleZoomstufe /= 100;
		maximaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMaximaleZoomstufe(maximaleZoomstufe);

		double minimaleZoomstufe = getDoTyp()
				.getLinienstaerkeMinimaleZoomstufeBundesstrasse();
		minimaleZoomstufe /= 100;
		minimaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMinimaleZoomstufe(minimaleZoomstufe);
	}

	private void setzeAutobahnAttribute() {
		final double zoomSkalierung = getZoomSkalierung();
		final RGB rgb = getDoTyp().getVordergrundFarbeAutobahn();
		if (rgb != null) {
			final Color color = getResourceManager().createColor(rgb);
			getFigure().setForegroundColor(color);
		}
		getFigure().setLinienabstand(getDoTyp().getLinienabstandAutobahn());

		getFigure().setLinienstaerkeZoomverhalten(
				getDoTyp().getLinienstaerkeZoomverhaltenAutobahn());
		getFigure().setLinienstaerke(getDoTyp().getLinienstaerkeAutobahn());

		getFigure().setLinienstaerkeMaximum(
				getDoTyp().getLinienstaerkeMaximumAutobahn());
		getFigure().setLinienstaerkeMinimum(
				getDoTyp().getLinienstaerkeMinimumAutobahn());

		double maximaleZoomstufe = getDoTyp()
				.getLinienstaerkeMaximaleZoomstufeAutobahn();
		maximaleZoomstufe /= 100;
		maximaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMaximaleZoomstufe(maximaleZoomstufe);

		double minimaleZoomstufe = getDoTyp()
				.getLinienstaerkeMinimaleZoomstufeAutobahn();
		minimaleZoomstufe /= 100;
		minimaleZoomstufe /= zoomSkalierung;
		getFigure().setLinienstaerkeMinimaleZoomstufe(minimaleZoomstufe);

	}

	private void refreshBasePoints() {
		final LinieFigure f = getFigure();
		if (getModel().getSystemObject() != null
				&& getModel().getLocation() == null
				&& getProjektion() != null) {

			f.setBasePoints(getPoints());
			((GraphicalEditPart) getParent()).setLayoutConstraint(this, f,
					f.getBounds());
		} else {
			f.setBasePoints(null);
		}
	}

	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") final Class key) {
		if (ILegendeBaustein.class.equals(key)) {
			return getLegendeBaustein();
		}

		return super.getAdapter(key);
	}

	private ILegendeBaustein getLegendeBaustein() {
		disposeLegende();
		legendeBaustein = new LinieLegendeBaustein(getDoTyp(), createFigure());
		return legendeBaustein;
	}

	private void disposeLegende() {
		if (legendeBaustein != null) {
			legendeBaustein.dispose();
			legendeBaustein = null;
		}
	}

}
