/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz.tmcmeldung;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import com.bitctrl.lib.eclipse.log.PluginLogger;

import de.bsvrz.buv.plugin.netz.internal.NetzPlugin;
import de.bsvrz.buv.plugin.netz.internal.RahmenwerkService;
import de.bsvrz.buv.rw.bitctrl.CacheService;
import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.att.WerteBereich;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsLocation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcPunkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.attribute.AtlTmcLokationsDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.objekte.TmcVerkehrsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.parameter.PdTmcVerkehrsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.parameter.PdTmcVerkehrsMeldung.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StreckenAbschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StreckenAbschnittCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcLocationCodeCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcLinieWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcPunktWrapper;

/**
 * Converter-Klasse zur Verortung der TMC-Meldungen auf der Karte.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 *
 */
final class TmcConverter {

	/**
	 * Aspekt f&uuml;r die Daten der TMC-Meldung.
	 */
	public static final Aspekt ASP = PdTmcVerkehrsMeldung.Aspekte.TmcEmpfangen;

	private static TmcConverter instance;
	private Map<Integer, List<AeusseresStrassenSegment>> tmcLocationCode2aess = new HashMap<>();

	private final Map<Integer, TmcPunkt> tmcPunktMap = new HashMap<>();
	private final PluginLogger pluginLogger;

	private final Logger logger = Logger
			.getLogger(TmcConverter.class.getName());

	private final TmcMeldungEditPart<?> editpart;

	private static class StartSegmente {
		protected int getLocationIndex() {
			return locationIndex;
		}

		protected List<AeusseresStrassenSegment> getAss() {
			return ass;
		}

		StartSegmente(final int locationIndex,
				final List<AeusseresStrassenSegment> ass) {
			super();
			this.locationIndex = locationIndex;
			this.ass = ass;
		}

		private final int locationIndex;

		private final List<AeusseresStrassenSegment> ass;
	}

	private static class RdsRoute {
		public List<StrassenSegment> getSegmente() {
			return segmente;
		}

		public AtlRdsLocation getLetzteLokationUnbekannt() {
			return letzteLokationUnbekannt;
		}

		public void setLetzteLokationUnbekannt(
				final AtlRdsLocation letzteLokationUnbekannt) {
			this.letzteLokationUnbekannt = letzteLokationUnbekannt;
		}

		public void addSegmente(final List<StrassenSegment> neuesegmente) {
			segmente.addAll(neuesegmente);
		}

		public void addSegment(final StrassenSegment segment) {
			segmente.add(segment);
		}

		RdsRoute() {
		}

		private final List<StrassenSegment> segmente = new ArrayList<>();

		private AtlRdsLocation letzteLokationUnbekannt;

	}

	private TmcConverter(final ObjektFactory objektFactory,
			final PluginLogger logger, final TmcMeldungEditPart<?> editPart) {
		pluginLogger = logger;
		// factory = objektFactory;
		initCache(objektFactory);
		editpart = editPart;
	}

	/**
	 * Gibt die (einzige) Instanz der KLasse zur&uuml;ck.
	 *
	 * @param objektFactory
	 *            ObjektFactory
	 * @param pluginLogger
	 *            PluginLogger
	 * @return Instanz
	 */
	static TmcConverter getInstance(final TmcMeldungEditPart<?> editPart) {
		if (instance == null) {
			instance = new TmcConverter(
					RahmenwerkService.getService().getObjektFactory(),
					NetzPlugin.getDefault().getLogger(), editPart);
		}

		return instance;
	}

	/**
	 * Wird nur für Unit-Test verwendet.
	 */
	static TmcConverter getInstance(final ObjektFactory objektFactory,
			final PluginLogger pluginLogger) {
		if (instance == null) {
			instance = new TmcConverter(objektFactory, pluginLogger, null);
		}

		return instance;
	}

	private void initCache(final ObjektFactory objektFactory) {
		tmcLocationCode2aess = new HashMap<>();
		final List<? extends SystemObjekt> ass = objektFactory
				.bestimmeModellobjekte(AeusseresStrassenSegment.PID);
		for (final SystemObjekt so : ass) {
			final AeusseresStrassenSegment segment = (AeusseresStrassenSegment) so;

			cacheLocationCode(segment);
		}
	}

	private void cacheLocationCode(final AeusseresStrassenSegment aess) {
		final KdAeusseresStrassenSegment.Daten assKd = aess
				.getKdAeusseresStrassenSegment().getDatum();

		if (assKd != null && assKd.getTmcPrimaerOrt() != null) {
			final TmcPunkt tmcPunkt = assKd.getTmcPrimaerOrt();

			// AeussereStrassensegmente-Map
			if (tmcPunkt.getKdTmcLocationCode() != null
					&& tmcPunkt.getKdTmcLocationCode().getDatum() != null) {

				final KdTmcLocationCode.Daten locationCodeDaten = tmcPunkt
						.getKdTmcLocationCode().getDatum();
				final AttTmcLocationCode locationCode = locationCodeDaten
						.getTmcLocationCode();

				if (locationCode != null) {
					tmcPunktMap.put(locationCode.getValue(), tmcPunkt);
					List<AeusseresStrassenSegment> aessSet = tmcLocationCode2aess
							.get(locationCode.getValue());
					if (aessSet == null) {
						aessSet = new ArrayList<>();
						tmcLocationCode2aess.put(locationCode.getValue(),
								aessSet);
					}
					aessSet.add(aess);
				}
			}
		}
	}

	// private StartSegmente findeASS(final Feld<AtlRdsLocation> rdsLocation,
	// final int startindex, final AttTmcRichtung attTmcRichtung,
	// final TmcVerkehrsMeldung meldung) {
	//
	// final List<AeusseresStrassenSegment> startAss = new
	// ArrayList<AeusseresStrassenSegment>();
	// int index = 0;
	//
	// if (startindex > (rdsLocation.size() - 2)) {
	// return null;
	// }
	//
	// List<AeusseresStrassenSegment> assSet = null;
	// AtlRdsLocation loc = null;
	//
	// // finde die erste bekannte Lokation
	// for (index = startindex + 1; index < rdsLocation.size(); index++) {
	// loc = rdsLocation.get(index);
	//
	// assSet = tmcLocationCode2aess.get(loc.getLocationCode().intValue());
	//
	// if (assSet == null || assSet.size() == 0) {
	// // die Lokation ist in der VRZ nicht bekannt
	// // --> wir versuchen die nächste in der Liste
	// continue;
	// }
	//
	// // das sind alle ASS, die an der Location enden
	// for (final AeusseresStrassenSegment ass : assSet) {
	// if (isAssNachKnoten(ass, attTmcRichtung, loc)) {
	// // die Linienlocation ist eindeutig zur Location, d.h.
	// // weitere ASS brauchen nicht untersucht zu werden
	// startAss.add(ass);
	// }
	// }
	//
	// if (startAss.size() > 0) {
	// break;
	// }
	// }
	//
	// // if (assSet == null || assSet.size() == 0) {
	// // // logError("Die RDS-Meldung "
	// // // + meldung.getPid()
	// // // +
	// // //
	// // " konnte nicht verortet werden (die Meldung enthält keine bekannten
	// // Lokationen)");
	// //
	// // return null;
	// // }
	// //
	// // // das sind alle ASS, die an der Location enden
	// // for (final AeusseresStrassenSegment ass : assSet) {
	// // if (isAssNachKnoten(ass, attTmcRichtung, loc)) {
	// // // die Linienlocation ist eindeutig zur Location, d.h.
	// // // weitere ASS brauchen nicht untersucht zu werden
	// // startAss.add(ass);
	// // }
	// // }
	//
	// if (startAss.size() == 0) {
	// return null;
	// // // kein ASS gefunden, dass in der geforderten Richtung auf
	// // // die Lokation zuführt
	// // // da die Lokation aber existiert, handelt es sich um eine
	// // // einseitige in der Gegenrichtung
	// // final AeusseresStrassenSegment start =
	// // findeStartSegmentFuerUndefinierteEinseitigeLokation(
	// // rdsLocation, attTmcRichtung, index);
	// // if (start != null) {
	// // startAss.add(start);
	// // }
	// }
	//
	// return new StartSegmente(index, startAss);
	// }

	/**
	 * Verortet die Meldung.
	 *
	 * @param meldung
	 *            TMC-Meldung
	 * @return TmcMeldungVerortung oder null, wenn nicht verortet werden konnte.
	 */
	public TmcMeldungVerortung verorte(final TmcVerkehrsMeldung meldung) {

		if (!editpart.cachesOK()) {
			return null;
		}

		// Es werden nur 'empfangene' Meldungen verortet
		final PdTmcVerkehrsMeldung pdRdsMeldung = meldung
				.getPdTmcVerkehrsMeldung();
		final Daten datum = pdRdsMeldung.getDatum(ASP);

		if (!datum.dContainsDaten()) {
			logError("Die TMC-Meldung " + meldung.getPid()
					+ " besitzt keine Daten unter dem Aspekt '" + ASP + "'");
			return null;
		}

		final Feld<AtlTmcLokationsDaten> lokationsDaten = datum.getTMCDaten()
				.getVerortungsInformationen().getLokationsDaten();

		if (lokationsDaten.size() == 0) {
			logError("Die TMC-Meldung " + meldung.getPid()
					+ " kann nicht verortet werden, da sie keine Lokationen besitzt");
			return null;
		}

		if (lokationsDaten.size() > 1) {
			// TODO: was tun wir hier
		}

		final AtlTmcLokationsDaten atlTmcLokationsDaten = lokationsDaten.get(0);
		final AttTmcRichtung tmcRichtung = atlTmcLokationsDaten
				.getTMCRichtung();

		// Die TMC-Richtung der Meldung ist verkehrt herum (beginnt mit Anfang
		// der Behinderung)
		final AttTmcRichtung attTmcRichtung = tmcRichtung == AttTmcRichtung.ZUSTAND_1_POSITIV
				? AttTmcRichtung.ZUSTAND_1N_NEGATIV
				: AttTmcRichtung.ZUSTAND_1_POSITIV;

		final AttTmcLocationCode primaerLokation = atlTmcLokationsDaten
				.getPrimaerLokation();
		final AttTmcLocationCode sekundaerLokation = atlTmcLokationsDaten
				.getSekundaerLokation();
		// Collections.reverse(rdsLocation);

		if (sekundaerLokation.getValue() == WerteBereich
				.getWerteBereich(AttTmcLocationCode.class).getMinimum()) {
			return verortePunktLokation(primaerLokation, attTmcRichtung,
					meldung);
		}

		return verorteMehrereLokationen(primaerLokation, sekundaerLokation,
				attTmcRichtung, meldung);
	}

	private StrassenKnoten getKnoten(final AttTmcLocationCode lokation) {
		final TmcLocationCodeCache tmcLocationCodeCache = editpart
				.getTmcLocationCodeCache();
		final TmcPunktWrapper tmcPunktWrapper = tmcLocationCodeCache
				.getTmcPunktWrapper(lokation);

		if (tmcPunktWrapper == null) {
			return null;
		}
		final TmcCache tmcCache = editpart.getTmcCache();
		return tmcCache.getStrassenKnoten(tmcPunktWrapper.getTmcPunkt());
	}

	private Strasse getStrasse(final AttTmcLocationCode lokation) {
		final CacheService cs = RahmenwerkService.getService()
				.getCacheService();

		final TmcLocationCodeCache tmcLocationCodeCache = editpart
				.getTmcLocationCodeCache();
		final TmcPunktWrapper tmcPunktWrapper = tmcLocationCodeCache
				.getTmcPunktWrapper(lokation);
		if (tmcPunktWrapper == null) {
			return null;
		}

		final TmcCache tmcCache = cs.getTmcCache(cs.getDefaultNetzPid());

		final TmcLinieWrapper tmcLinieWrapper = tmcLocationCodeCache
				.getTmcLinieWrapper(tmcPunktWrapper.getIstTeilvonTmcLinie(),
						AttRdsTMCRichtung.ZUSTAND_0_POSITIV);

		return tmcCache.getStrasse(tmcLinieWrapper.getTmcLinie());
	}

	private TmcMeldungVerortung verorteMehrereLokationen(
			final AttTmcLocationCode primaerLokation,
			final AttTmcLocationCode sekundaerLokation,
			final AttTmcRichtung attTmcRichtung,
			final TmcVerkehrsMeldung meldung) {

		final StreckenAbschnittCache streckenAbschnittCache = editpart
				.getStreckenAbschnittCache();

		final StrassenKnoten startKnoten = getKnoten(primaerLokation);
		final StrassenKnoten endKnoten = getKnoten(sekundaerLokation);

		final Strasse strasse = getStrasse(primaerLokation);
		if (strasse == null) {
			return null;
		}

		if (startKnoten == null && endKnoten == null) {
			return null;
		}

		if (startKnoten != null && endKnoten == null) {
			return verortePunktLokation(primaerLokation, attTmcRichtung,
					meldung);
		}

		if (endKnoten != null && startKnoten == null) {
			return verortePunktLokation(sekundaerLokation, attTmcRichtung,
					meldung);
		}

		final List<StreckenAbschnitt> streckenAbschnitte;
		try {
			streckenAbschnitte = streckenAbschnittCache.getStreckenAbschnitte(
					strasse, attTmcRichtung, endKnoten, startKnoten);
		} catch (final IllegalArgumentException ex) {
			logError("Die TMC-Meldung '" + meldung.getPid()
					+ "' kann nicht verortet werden: "
					+ ex.getLocalizedMessage() + " "
					+ meldungLogInfo(meldung, primaerLokation,
							sekundaerLokation, attTmcRichtung));
			return null;
		}

		if (streckenAbschnitte == null || streckenAbschnitte.size() == 0) {
			logError("Die TMC-Meldung '" + meldung.getPid()
					+ "' kann nicht verortet werden: Es konnten kein Streckenabschnitte bestimmt werden "
					+ meldungLogInfo(meldung, primaerLokation,
							sekundaerLokation, attTmcRichtung));
			return null;
		}

		final StreckenAbschnitt a = streckenAbschnitte.get(0);
		final List<StrassenSegment> strassenSegmente = a.getStrassenSegmente();

		if (strassenSegmente == null || strassenSegmente.size() == 0) {
			logError("Die TMC-Meldung '" + meldung.getPid()
					+ "' kann nicht verortet werden: Es konnten keine Segmente bestimmt werden "
					+ meldungLogInfo(meldung, primaerLokation,
							sekundaerLokation, attTmcRichtung));
			return null;
		}

		// final List<StrassenSegment> segmente = verorteWeiter(startSegmente,
		// rdsLocation, attTmcRichtung, meldung);
		//
		return erzeugeVerortung(strassenSegmente, meldung);
	}

	private AeusseresStrassenSegment findeZufuehrendesAss(
			final AttTmcLocationCode rdsLocation,
			final AttTmcRichtung attTmcRichtung,
			final TmcVerkehrsMeldung meldung) {
		List<AeusseresStrassenSegment> assSet = null;
		assSet = tmcLocationCode2aess.get(rdsLocation.intValue());

		if (assSet == null || assSet.size() == 0) {
			// die Lokation ist in der VRZ nicht bekannt
			logError("Die TMC-Meldung " + meldung.getPid()
					+ " konnte nicht verortet werden (die Lokation der Meldung ist in der VRZ nicht bekannt)");

			return null;
		}

		// das sind alle ASS, die an der Location enden
		for (final AeusseresStrassenSegment ass : assSet) {
			if (isAssNachKnoten(ass, attTmcRichtung, rdsLocation)) {
				// die Linienlocation ist eindeutig zur Location, d.h.
				// weitere ASS brauchen nicht untersucht zu werden
				return ass;
			}
		}

		// kein Segment mit passender Richtung gefunden -> wir nehmen das erste
		// beste
		return assSet.get(0);
	}

	private TmcMeldungVerortung verortePunktLokation(
			final AttTmcLocationCode rdsLocation,
			final AttTmcRichtung attTmcRichtung,
			final TmcVerkehrsMeldung meldung) {

		final AeusseresStrassenSegment segment = findeZufuehrendesAss(
				rdsLocation, attTmcRichtung, meldung);

		if (segment != null) {
			return erzeugeVerortungPunkt(segment, meldung);
		}

		return null;
	}

	// private List<StrassenSegment> verorteWeiter(
	// final StartSegmente startSegmente,
	// final Feld<AtlRdsLocation> rdsLocation,
	// final AttTmcRichtung attTmcRichtung, final RdsMeldung meldung) {
	//
	// if (startSegmente == null) {
	// return null;
	// }
	//
	// final List<RdsRoute> routen = new ArrayList<RdsRoute>();
	//
	// for (final AeusseresStrassenSegment ass : startSegmente.getAss()) {
	// final RdsRoute rdsroute = new RdsRoute();
	// rdsroute.addSegment(ass);
	// routen.add(rdsroute);
	// AeusseresStrassenSegment startAss = ass;
	//
	// for (int i = startSegmente.getLocationIndex() + 1; i < rdsLocation
	// .size(); i++) {
	// final AtlRdsLocation loc = rdsLocation.get(i);
	//
	// if (startAss == null) {
	// break;
	// }
	//
	// final List<StrassenSegment> route = findeSegmente(startAss,
	// loc);
	// if (route == null) {
	// // dies passiert bei Lokationen, die im Netz der VRZ in der
	// // angegebenen TMC-Richtung nicht vorhanden sind
	// // --> wir versuchen die nächste in der Liste
	// rdsroute.setLetzteLokationUnbekannt(loc);
	// continue;
	// }
	//
	// rdsroute.setLetzteLokationUnbekannt(null);
	//
	// if (route.size() > 0) {
	// rdsroute.addSegmente(route);
	//
	// final StrassenSegment strassenSegment = route
	// .get(route.size() - 1);
	// if (strassenSegment instanceof AeusseresStrassenSegment) {
	// startAss = (AeusseresStrassenSegment) strassenSegment;
	// } else {
	// break;
	// }
	// }
	// }
	// }
	//
	// // beste Route auswählen (meiste Segmente)
	// Collections.sort(routen, new Comparator<RdsRoute>() {
	//
	// @Override
	// public int compare(final RdsRoute o1, final RdsRoute o2) {
	// return ((Integer) o2.getSegmente().size())
	// .compareTo(o1.getSegmente().size());
	// }
	// });
	//
	// final RdsRoute ergebnis = routen.get(0);
	//
	// if (ergebnis.getLetzteLokationUnbekannt() != null) {
	// AeusseresStrassenSegment lastass = null;
	//
	// for (int i = ergebnis.getSegmente().size() - 1; i >= 0; i++) {
	// final StrassenSegment s = ergebnis.getSegmente().get(i);
	// if (s instanceof AeusseresStrassenSegment) {
	// lastass = (AeusseresStrassenSegment) s;
	// break;
	// }
	// }
	//
	// if (lastass != null) {
	// final KdAeusseresStrassenSegment.Daten assKd = lastass
	// .getKdAeusseresStrassenSegment().getDatum();
	// logWarning("Die RDS-Meldung " + meldung.getPid()
	// + " konnte nicht vollständig verortet werden (es existiert kein Weg in "
	// + attTmcRichtung + "er TMC-Richtung von Lokation "
	// + assKd.getTmcPrimaerOrt().getKdTmcLocationCode()
	// .getDatum().getTmcLocationCode()
	// + " nach Lokation "
	// + ergebnis.getLetzteLokationUnbekannt()
	// .getLocationCode()
	// + " auf der Linienlokation "
	// + ergebnis.getLetzteLokationUnbekannt()
	// .getLocationCodeLinienReferenz().intValue()
	// + " am Ende der Lokationenliste)");
	// }
	// }
	//
	// return ergebnis.getSegmente();
	// }

	private TmcMeldungVerortung erzeugeVerortung(
			final List<StrassenSegment> segmentliste,
			final TmcVerkehrsMeldung meldung) {

		if (segmentliste == null || segmentliste.size() == 0) {
			return null;
		}

		// erstes (inneres Teilsegment) weglassen
		if (segmentliste.size() > 1
				&& (segmentliste.get(0) instanceof InneresStrassenSegment)) {
			segmentliste.remove(0);
		}

		// Offsets bestimmen
		final int startoffset = 0;
		final int endOffset = segmentliste.get(segmentliste.size() - 1)
				.getKdStrassenSegment().getDatum().getLaenge().intValue();

		return new TmcMeldungVerortung(segmentliste, startoffset, endOffset);
	}

	// private List<StrassenSegment> findeSegmente(
	// final AeusseresStrassenSegment startAss, final AtlRdsLocation loc) {
	// final List<StrassenSegment> route = new ArrayList<StrassenSegment>();
	//
	// final KdAeusseresStrassenSegment.Daten assKd = startAss
	// .getKdAeusseresStrassenSegment().getDatum();
	//
	// final StrassenKnoten knoten = assKd.getNachKnoten();
	//
	// if (isAssNachKnoten(startAss, assKd.getTmcRichtung(), loc)) {
	// return route;
	// }
	//
	// for (final InneresStrassenSegment segment : knoten
	// .getInnereStrassenSegmente()) {
	// final KdInneresStrassenSegment.Daten kdIss = segment
	// .getKdInneresStrassenSegment().getDatum();
	//
	// if (kdIss.getVonStrassenSegment() != null
	// && kdIss.getVonStrassenSegment() == startAss) {
	// // KdStrassenSegment.Daten kdIssSeg = segment
	// // .getKdStrassenSegment().getDatum();
	// // if (kdIssSeg.getGehoertZuStrasse() != null
	// // && kdIssSeg.getGehoertZuStrasse() == assKdSeg
	// // .getGehoertZuStrasse()) {
	// final AeusseresStrassenSegment nachStrassenSegment = kdIss
	// .getNachStrassenSegment();
	//
	// if (nachStrassenSegment != null && isAssNachKnoten(
	// nachStrassenSegment, assKd.getTmcRichtung(), loc)) {
	//
	// route.add(segment);
	// route.add(nachStrassenSegment);
	// return route;
	// }
	// // }
	// }
	// }
	//
	// return null;
	// }
	//
	private boolean isAssNachKnoten(final AeusseresStrassenSegment ass,
			final AttTmcRichtung tmcRichtung, final AttTmcLocationCode loc) {
		final KdAeusseresStrassenSegment.Daten assKd = ass
				.getKdAeusseresStrassenSegment().getDatum();

		if (assKd.getTmcRichtung() == tmcRichtung) {
			final TmcPunkt tmcPunkt = assKd.getTmcPrimaerOrt();
			final KdTmcLocationCode.Daten datum = tmcPunkt
					.getKdTmcLocationCode().getDatum();
			if (datum.getTmcLocationCode().intValue() != loc.intValue()) {
				return false;
			}

			// // Linienlocation überprüfen
			// final TmcLinie istTeilvonTmcLinie = tmcPunkt.getKdTmcPunkt()
			// .getDatum().getIstTeilvonTmcLinie();
			// final AttTmcLocationCode tmcLocationCode = istTeilvonTmcLinie
			// .getKdTmcLocationCode().getDatum().getTmcLocationCode();
			//
			// if (tmcLocationCode.intValue() != loc
			// .getLocationCodeLinienReferenz().intValue()) {
			// logError("Die Linienlokation des TmcPrimärortes des ASS ("
			// + tmcLocationCode.intValue() + ", " + tmcPunkt
			// + ") stimmt nicht mit der Linienlokation der Meldung ("
			// + loc.getLocationCodeLinienReferenz().intValue()
			// + ") überein.");
			// return false;
			// }

			return true;
		}

		return false;
	}

	/**
	 * Erzeugt eine Verortung f&uum;r eine Punktlokation.
	 *
	 * Es wird das k&uuml;rzeste weiterf&uuml;hrende innere Straßensegment
	 * gesucht und die Meldung in der Mitte dieses Segmentes verortet. Nicht
	 * weiterführende Segmente werden nicht benutzt, da damit keine
	 * Punktverortung im Knoten sinnvoll möglich ist. Wenn kein
	 * weiterf&uuml;hrendes inneres Stra&szlig;ensegment gefunden werden kann,
	 * wird die Meldung am Ende des zuf&uuml;hrenden &auml;u&szlig;eren
	 * Segmesntes verortet.
	 */
	private TmcMeldungVerortung erzeugeVerortungPunkt(
			final AeusseresStrassenSegment ass,
			final TmcVerkehrsMeldung meldung) {

		StrassenSegment segment = null;
		int startoffset = 0;

		final KdAeusseresStrassenSegment.Daten assKd = ass
				.getKdAeusseresStrassenSegment().getDatum();

		final StrassenKnoten nachKnoten = assKd.getNachKnoten();

		if (nachKnoten == null) {
			return null;
		}

		InneresStrassenSegment weiterfuehrendesIss = null;

		// es wird das kürzeste weiterführende innere Straßensegment gesucht
		// nicht weiterführende werden nicht benutzt, da damit keine
		// Punktverortung im Knoten sinnvoll möglich ist
		for (final InneresStrassenSegment iss : nachKnoten
				.getInnereStrassenSegmente()) {

			final AeusseresStrassenSegment vonStrassenSegment = iss
					.getKdInneresStrassenSegment().getDatum()
					.getVonStrassenSegment();

			if (vonStrassenSegment != ass) {
				continue;
			}

			if (iss.getKdInneresStrassenSegment().getDatum()
					.getNachStrassenSegment() != null) {
				if (weiterfuehrendesIss == null || weiterfuehrendesIss
						.getKdStrassenSegment().getDatum().getLaenge()
						.intValue() > iss.getKdStrassenSegment().getDatum()
								.getLaenge().intValue()) {
					weiterfuehrendesIss = iss;
				}

			}
			segment = weiterfuehrendesIss;

			if (segment != null) {
				final int isslaenge = segment.getKdStrassenSegment().getDatum()
						.getLaenge().intValue();

				if (isslaenge > 0) {
					startoffset = isslaenge / 2;
				}
			}
		}

		// keine inneren Strassensegmente:
		// -> Verortung am Ende des ASS
		if (segment == null) {
			segment = ass;
			startoffset = ass.getKdStrassenSegment().getDatum().getLaenge()
					.intValue();
		}

		final List<StrassenSegment> segmente = new ArrayList<>();
		segmente.add(segment);

		// Länge wird auf 0 gesetzt
		return new TmcMeldungVerortung(segmente, startoffset, startoffset);
	}

	private void logWarning(final String msg) {
		if (pluginLogger != null) {
			pluginLogger.error(msg);
		} else {
			logger.severe(msg);
		}
	}

	private void logError(final String msg) {
		if (pluginLogger != null) {
			pluginLogger.error(msg);
		} else {
			logger.severe(msg);
		}
	}

	private String meldungLogInfo(final TmcVerkehrsMeldung meldung,
			final AttTmcLocationCode prim, final AttTmcLocationCode sek,
			final AttTmcRichtung attTmcRichtung) {

		final StringBuffer text = new StringBuffer();

		final AttRdsTMCRichtung tmcRichtung = attTmcRichtung == AttTmcRichtung.ZUSTAND_1_POSITIV
				? AttRdsTMCRichtung.ZUSTAND_1_NEGATIV
				: AttRdsTMCRichtung.ZUSTAND_0_POSITIV;

		text.append("(TMC-Richtung der Meldung: " + tmcRichtung);

		text.append(", Primärlokation: " + prim.intValue());
		text.append(", Sekundärlokation: " + sek.intValue());

		text.append(")");

		return text.toString();
	}
}
