/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.netz.tmcmeldung;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.draw2d.geometry.PointList;

import de.bsvrz.buv.plugin.dobj.decorator.LinienabstandZoomMediator;
import de.bsvrz.buv.plugin.dobj.editparts.BitCtrlDoModelEditPart;
import de.bsvrz.buv.plugin.netz.internal.RahmenwerkService;
import de.bsvrz.buv.rw.bitctrl.CacheService;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcverkehrsmeldung.objekte.TmcVerkehrsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StreckenAbschnittCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcLocationCodeCache;

/**
 * Basisklasse für MVC-Controller zur Darstellung einer <em>TMC-Meldung</em>.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 *
 */
abstract class TmcMeldungEditPart<F extends TmcMeldungFigure>
		extends BitCtrlDoModelEditPart<TmcVerkehrsMeldung, F> {

	private TmcMeldungEmpfaenger empfaenger;

	private static final CacheService CACHE_SERVICE = RahmenwerkService
			.getService().getCacheService();

	private static final TmcCache TMC_CACHE = CACHE_SERVICE
			.getTmcCache(CACHE_SERVICE.getDefaultNetzPid());

	private static final StreckenAbschnittCache ABSCHNITT_CACHE = CACHE_SERVICE
			.getStreckenAbschnittCache(CACHE_SERVICE.getDefaultNetzPid());

	private static final TmcLocationCodeCache LOCATION_CACHE = CACHE_SERVICE
			.getTmcLocationCodeCache();

	private PropertyChangeListener cacheListener;

	@Override
	protected void geheOnline() {
		super.geheOnline();

		if (cachesOK()) {
			if (getSystemObjekt() != null) {
				empfaenger = createTmcMeldungEmpfaenger();
				empfaenger.anmelden();
			}
		}
	}

	protected abstract TmcMeldungEmpfaenger createTmcMeldungEmpfaenger();

	@Override
	public void activate() {
		super.activate();

		cacheListener = new PropertyChangeListener() {

			@Override
			public void propertyChange(final PropertyChangeEvent evt) {
				if (TMC_CACHE.isInitialisiert()) {
					TMC_CACHE.removePropertyChangeListener(
							AbstractCache.PROP_STATE, this);
				}

				if (ABSCHNITT_CACHE.isInitialisiert()) {
					ABSCHNITT_CACHE.removePropertyChangeListener(
							AbstractCache.PROP_STATE, this);
				}

				if (LOCATION_CACHE.isInitialisiert()) {
					LOCATION_CACHE.removePropertyChangeListener(
							AbstractCache.PROP_STATE, this);
				}

				if (cachesOK()) {
					if (getSystemObjekt() != null) {
						empfaenger = createTmcMeldungEmpfaenger();
						empfaenger.anmelden();
					}
					refreshVisuals();
				}
			}
		};

		if (!TMC_CACHE.isInitialisiert()) {
			TMC_CACHE.addPropertyChangeListener(AbstractCache.PROP_STATE,
					cacheListener);
		}

		if (!ABSCHNITT_CACHE.isInitialisiert()) {
			ABSCHNITT_CACHE.addPropertyChangeListener(AbstractCache.PROP_STATE,
					cacheListener);
		}

		if (LOCATION_CACHE.isInitialisiert()) {
			LOCATION_CACHE.addPropertyChangeListener(AbstractCache.PROP_STATE,
					cacheListener);
		}
	}

	@Override
	public void deactivate() {
		if ((TMC_CACHE != null) && (cacheListener != null)) {
			TMC_CACHE.removePropertyChangeListener(AbstractCache.PROP_STATE,
					cacheListener);
		}

		if ((ABSCHNITT_CACHE != null) && (cacheListener != null)) {
			ABSCHNITT_CACHE.removePropertyChangeListener(
					AbstractCache.PROP_STATE, cacheListener);
		}

		if ((LOCATION_CACHE != null) && (cacheListener != null)) {
			LOCATION_CACHE.removePropertyChangeListener(
					AbstractCache.PROP_STATE, cacheListener);
		}

		cacheListener = null;

		super.deactivate();
	}

	@Override
	protected void geheOffline() {
		if (empfaenger != null) {
			empfaenger.abmelden();
			empfaenger = null;
		}

		super.geheOffline();
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		refreshLocation();

		new LinienabstandZoomMediator(this).mediate();
	}

	final void refreshLocation() {
		if (empfaenger != null) {
			getFigure().setStreckenabschnitt(empfaenger.getStreckenabschnitt());
		} else {
			getFigure().setStreckenabschnitt(new PointList());
		}
	}

	public TmcCache getTmcCache() {
		return TMC_CACHE;
	}

	public boolean cachesOK() {
		return TMC_CACHE.isInitialisiert() && ABSCHNITT_CACHE.isInitialisiert()
				&& LOCATION_CACHE.isInitialisiert();
	}

	public StreckenAbschnittCache getStreckenAbschnittCache() {
		return ABSCHNITT_CACHE;
	}

	public TmcLocationCodeCache getTmcLocationCodeCache() {
		return LOCATION_CACHE;
	}

}
