/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.beispiel;

import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

import de.bsvrz.buv.plugin.param.editors.AbstractParameterFormPage;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.helper.UzVerifyListener;
import de.bsvrz.buv.rw.basislib.legende.ILegende;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.puk.param.lib.Parameter;

/**
 * Editorseiten-Stub als Beispiel, wie angepasste Parameterdialoge nach TBuV-138
 * erstellt werden können.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class BeispielParamEditorFormPage extends AbstractParameterFormPage {

	/**
	 * Schlüssel für den Namen des DAF-Attributs, welches von einem {@link Text}
	 * -Widget bearbeitet wird.
	 */
	public static final String DATAKEY_ITEMNAME = "itemName";

	/**
	 * Die vier atomaren Subattribute in der Attributgruppe
	 * atg.verkehrsLageVerfahren1.
	 */
	private final String[] itemNames = { "v1", "v2", "k1", "k2" };

	/**
	 * Konstruktor reicht die Werte nur durch.
	 *
	 * @param id
	 *            die ID
	 * @param title
	 *            Titel des Editors
	 */
	public BeispielParamEditorFormPage(final String id, final String title) {
		super(id, title);
	}

	/**
	 * Konstruktor reicht die Werte nur durch.
	 *
	 * @param editor
	 *            der Editor
	 * @param id
	 *            die ID
	 * @param title
	 *            Titel des Editors
	 */
	public BeispielParamEditorFormPage(final FormEditor editor, final String id, final String title) {
		super(editor, id, title);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.editor.FormPage#createFormContent(org.eclipse.ui
	 * .forms.IManagedForm)
	 */
	@Override
	protected void createFormContent(final IManagedForm managedForm) {
		final ScrolledForm form = managedForm.getForm();
		final FormToolkit toolkit = managedForm.getToolkit();

		final TableWrapLayout layout = new TableWrapLayout();
		form.getBody().setLayout(layout);
		layout.numColumns = 2;
		final StringBuilder b = new StringBuilder("Beispiel-Parameterdialog: ");
		final ParameterEditorInput pei = (ParameterEditorInput) getEditorInput();
		final Parameter[] parameters = pei.getParameters();
		b.append(parameters.length);
		b.append(" Objekt");
		if (parameters.length > 1) {
			b.append('e');
		}
		b.append(" in Bearbeitung");
		form.setText(b.toString());

		final Map<SystemObject, Text[]> editedControls = erzeugeAbschnittDaten(form, toolkit, parameters);
		final Map<SystemObject, Parameter> referenzParameter = new LinkedHashMap<>(parameters.length);
		for (final Parameter p : parameters) {
			referenzParameter.put(p.getObjekt(), p);
		}
		final BeispielParamEditorFormPart formPart = new BeispielParamEditorFormPart(editedControls, referenzParameter);
		managedForm.addPart(formPart);
		final IAction exportAction = new Action() {

			/*
			 * (non-Javadoc)
			 *
			 * @see org.eclipse.jface.action.Action#run()
			 */
			@Override
			public void run() {
				MessageDialog.openInformation(null, "Export",
						"Hier könnte der Inhalt des " + "Beispieleditors exportiert werden");
			}
		};
		setExportAction(exportAction);

		super.createFormContent(managedForm);
	}

	/**
	 * Erzeugt den Abschnitt mit den Steuerelementen für das Bearbeiten. Diese sind
	 * tabellarisch angeordnet und haben je 5 Spalten: 1. SPalte it der Objektname,
	 * die anderen 4 Spalten sind die 4 atomaren Attribute der
	 * atg.verkehrsLageVerfahren1.
	 *
	 * @param form
	 *            das Form, wo die Steuerelemente Platz finden sollen
	 * @param toolkit
	 *            das Toolkit zum Erzeugen von Steuerelementen
	 * @param parameters
	 *            die zu bearbeitenden Parameter.
	 * @return eine Zuordnung, die jedem bearbeiteten Parameter, sprich
	 *         Systemobjekt, die 4 Text-Steuerelemente zuordnet. Über deren
	 *         Data-Member und den Schlüssel {@link #DATAKEY_ITEMNAME} kann dann
	 *         beim Speichern die DAF-Data gefüllt werden.
	 */
	private Map<SystemObject, Text[]> erzeugeAbschnittDaten(final ScrolledForm form, final FormToolkit toolkit,
			final Parameter[] parameters) {
		final Section sektion = toolkit.createSection(form.getBody(), Section.DESCRIPTION
				| ExpandableComposite.TITLE_BAR | ExpandableComposite.TWISTIE | ExpandableComposite.EXPANDED);
		final TableWrapData td = new TableWrapData(TableWrapData.FILL_GRAB);
		td.colspan = 2;
		sektion.setLayoutData(td);

		sektion.setText("Daten");
		sektion.setDescription("In diesem Abschnitt erfolgt das Editieren," + " allerdings ohne Wertebereichsprüfung!");
		sektion.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				form.reflow(true);
			}
		});

		// Controls erzeugen
		final Composite sectionClient = toolkit.createComposite(sektion);
		sectionClient.setLayout(new GridLayout(1 + itemNames.length, false));
		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.heightHint = 200;
		sectionClient.setLayoutData(gd);
		Label l = toolkit.createLabel(sectionClient, "Objekt");
		gd = new GridData(SWT.FILL, SWT.FILL, false, false);
		l.setLayoutData(gd);
		int loop;
		for (loop = 0; loop < itemNames.length; ++loop) {
			l = toolkit.createLabel(sectionClient, itemNames[loop]);
			gd = new GridData(SWT.FILL, SWT.FILL, false, false);
			l.setLayoutData(gd);
		}
		final Map<SystemObject, Text[]> editedControls = new LinkedHashMap<>();
		for (final Parameter p : parameters) {
			l = toolkit.createLabel(sectionClient, p.getObjekt().toString());
			gd = new GridData(SWT.FILL, SWT.FILL, false, false);
			l.setLayoutData(gd);
			final Text[] texte = new Text[itemNames.length];
			for (loop = 0; loop < itemNames.length; ++loop) {
				final Text t = toolkit.createText(sectionClient,
						p.hasData() ? p.getData().getItem(itemNames[loop]).asUnscaledValue().getValueText()
								: "undefiniert");
				gd = new GridData(SWT.FILL, SWT.FILL, false, false);
				gd.widthHint = 50;
				t.setLayoutData(gd);
				t.addVerifyListener(new UzVerifyListener(UzVerifyListener.INT));
				t.addModifyListener(new ModifyListener() {

					@Override
					public void modifyText(final ModifyEvent e) {
						((AbstractFormPart) getManagedForm().getParts()[0]).markDirty();
					}
				});
				texte[loop] = t;
				t.setData(BeispielParamEditorFormPage.DATAKEY_ITEMNAME, itemNames[loop]);
			}
			editedControls.put(p.getObjekt(), texte);
		}

		sektion.setClient(sectionClient);
		return editedControls;
	}

	/**
	 * {@inheritDoc}
	 *
	 * Das suppress warnings ist ok. Die Methodensignatur kann nicht korrigiert
	 * werden, da die Schnittstelle von Eclipse kommt.
	 */
	@Override
	public Object getAdapter(final Class adapter) {
		if (ILegende.class.isAssignableFrom(adapter)) {
			return new BeispielParamEditorLegendenAbschnitt();
		}
		return super.getAdapter(adapter);
	}
}
