/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.beispiel;

import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.AbstractFormPart;

import de.bsvrz.buv.plugin.param.editors.ParameterSaveResultEnum;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver.ParameterSaveUrsache;
import de.bsvrz.buv.plugin.param.editors.beispiel.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * FormPart für einen einfachen Editor für die Attributgruppe
 * {@link BeispielParamEditorFormPageFactory#EDITABLE_ATTRIBUTE_GROUP}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class BeispielParamEditorFormPart extends AbstractFormPart {

	/**
	 * eine Zuordnung, die jedem bearbeiteten Parameter, sprich Systemobjekt, die 4
	 * Text-Steuerelemente zuordnet. Über deren Data-Member und den Schlüssel
	 * {@link #DATAKEY_ITEMNAME} kann dann beim Speichern die DAF-Data gefüllt
	 * werden.
	 */
	private final Map<SystemObject, Text[]> editedControls;

	/**
	 * eine Zuordnung, mit der der Zugriff auf die ursprünglich in den Editor
	 * geladenen Parameter über den selben Schlüssel gefunden werden können, wir die
	 * zugehörigen Text-Steuerelemente.
	 */
	private final Map<SystemObject, Parameter> referenzParameters;

	/**
	 * Konstruktor merkt sich nur die Member.
	 *
	 * @param editedControls
	 *            eine Zuordnung, die jedem bearbeiteten Parameter, sprich
	 *            Systemobjekt, die 4 Text-Steuerelemente zuordnet. Über deren
	 *            Data-Member und den Schlüssel
	 *            {@link BeispielParamEditorFormPage#DATAKEY_ITEMNAME} kann dann
	 *            beim Speichern die DAF-Data gefüllt werden.
	 * @param referenzParameters
	 *            eine Zuordnung, mit der der Zugriff auf die ursprünglich in den
	 *            Editor geladenen Parameter über den selben Schlüssel gefunden
	 *            werden können, wir die zugehörigen Text-Steuerelemente.
	 */
	public BeispielParamEditorFormPart(final Map<SystemObject, Text[]> editedControls,
			final Map<SystemObject, Parameter> referenzParameters) {
		this.editedControls = editedControls;
		this.referenzParameters = referenzParameters;
	}

	@Override
	public void commit(final boolean onSave) {
		final Parameter[] editedParameters = new Parameter[editedControls.size()];
		int parameterLoop = 0;
		for (final Entry<SystemObject, Text[]> entry : editedControls.entrySet()) {
			SystemObject systemObject = entry.getKey();
			final Parameter referenzParameter = referenzParameters.get(systemObject);
			final ParameterInfo info = new ParameterInfo(systemObject, referenzParameter.getTyp(), referenzParameter.getAtg(),
					referenzParameter.getSim());
			final Data data;
			if (referenzParameter.hasData()) {
				data = referenzParameter.getData().createModifiableCopy();
			} else {
				data = RahmenwerkService.getService().getObjektFactory().getDav()
						.createData(referenzParameter.getAtg());
			}
			int loop;
			final Text[] texte = entry.getValue();
			for (loop = 0; loop < texte.length; ++loop) {
				final Text t = texte[loop];
				final String itemName = (String) t.getData(BeispielParamEditorFormPage.DATAKEY_ITEMNAME);
				final String valueString = t.getText();
				int value = -1;
				if (!valueString.isEmpty()) {
					value = Integer.parseInt(valueString);
				}
				data.getUnscaledValue(itemName).set(value);
			}
			editedParameters[parameterLoop] = new Parameter(info, new DataWithTime(data, System.currentTimeMillis()));
			++parameterLoop;
		}
		final ParameterSaveResultEnum saveResult = ParameterSaver.saveParameters(editedParameters, ParameterSaver
				.buildDefaultUrsache(editedParameters, editedParameters, ParameterSaveUrsache.PARAMETRIERT));
		switch (saveResult) {
		case PARAMETER_SAVE_FAILURE:
			/* Make Eclipse display an error dialog */
			throw new IllegalStateException(ParameterSaver.getLastError());
		case PARAMETER_SAVE_CANCEL:
			// Do nothing
			break;
		case PARAMETER_SAVE_SUCCESS:
			super.commit(onSave);
			break;
		default:
			break;
		}
	}
}
