/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table;

import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.nebula.paperclips.core.border.LineBorder;
import org.eclipse.nebula.paperclips.core.grid.CellBackgroundProvider;
import org.eclipse.nebula.paperclips.core.grid.DefaultGridLook;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.nebula.paperclips.core.text.TextPrint;
import org.eclipse.nebula.paperclips.core.text.TextStyle;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

import com.bitctrl.lib.eclipse.actions.ExportSWTTableViewerToCSVorHTMLAction;

import de.bsvrz.buv.plugin.param.ParamUtils;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.AbstractParameterFormPage;
import de.bsvrz.buv.plugin.param.editors.ParameterEditor;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.table.actions.TableParamEditorAction;
import de.bsvrz.buv.plugin.param.editors.table.actions.TableParamEditorInvalidateElementAction;
import de.bsvrz.buv.plugin.param.editors.table.edit.IParamTableEditorModifyListener;
import de.bsvrz.buv.plugin.param.editors.table.edit.TableParamEditorAddElementAction;
import de.bsvrz.buv.plugin.param.editors.table.edit.TableParamEditorDoubleClickListener;
import de.bsvrz.buv.plugin.param.editors.table.provider.ArrayNameAndIndex;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParamTableEditorLabelProvider;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParameterTableEditorItem;
import de.bsvrz.buv.plugin.param.editors.table.provider.PlainDataValue;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Parameter-Editor in Tabellenform, siehe TBuV-137 und diversen E-Mailverkehr.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class TableParamEditorFormPage extends AbstractParameterFormPage
		implements IParamTableEditorModifyListener, RwPrintable {

	/**
	 * Logger für Debugausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Schlüssel in die Data-Tabelle der Table, um den TableCursor zu finden.
	 */
	public static final String TABLEDATA_KEY_CURSOR = "CURSOR";

	/**
	 * Schlüssel in die Data-Tabelle des TableCursor, um das Tableobjekt zu finden,
	 * zu dem er gehört.
	 */
	public static final String CURSORDATA_KEY_TABLE = "TABLE";

	/**
	 * Default-Spaltenbreite für Tabelleneditor.
	 */
	private static final int DEFAULT_COLUMN_WIDTH = 120;

	/**
	 * Der Viewer für die Tabelle des Editors.
	 */
	private TableViewer viewer;

	/**
	 * steuer die Kontextmenüs.
	 */
	private MenuManager contextMenuManager;

	/**
	 * Aktion zum Hinzufügen eines Elements (atomar oder komplex) in ein Feld.
	 */
	private TableParamEditorAction elementAdd;

	/**
	 * Aktion zum Zurücksetzen der Daten eines Attributs (und potentiell aller
	 * anderen Attribute im übergeordneten Attributset, da Attributsets immer
	 * vollständig sein müssen).
	 */
	private TableParamEditorAction elementInvalidate;

	/**
	 * Die Sektion "Daten". Wir müssen sie ab jetzt ale Member führen, damit wir die
	 * Zahl der Einzelattribute, die in der Description auftauscht, dynamisch
	 * anpassen können.
	 */
	private Section datenSektion;

	/**
	 * Konstruktor reicht die Werte nur durch.
	 *
	 * @param editor
	 *            der Editor
	 * @param id
	 *            die ID
	 * @param index
	 *            Index der Page
	 */
	public TableParamEditorFormPage(final FormEditor editor, final String id, final int index) {
		super(editor, id, ParameterEditor.VALID_INDICES[index]);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.editor.FormPage#createFormContent(org.eclipse.ui
	 * .forms.IManagedForm)
	 */
	@Override
	protected void createFormContent(final IManagedForm managedForm) {
		final ScrolledForm form = managedForm.getForm();
		form.addControlListener(new ControlListener() {

			@Override
			public void controlResized(final ControlEvent e) {
				TableParamEditorFormPage.DEBUG.finer("form Resize!!!", form.getSize());
				layoutModified();
			}

			@Override
			public void controlMoved(final ControlEvent e) {
				// leer
			}
		});
		final FormToolkit toolkit = managedForm.getToolkit();
		final GridLayout layout = new GridLayout();
		form.getBody().setLayout(layout);
		final ParameterEditorInput pei = (ParameterEditorInput) getEditorInput();
		Parameter[] parameters = null;
		switch (getIndex()) {
		case 0:
			parameters = pei.getParameters();
			break;
		case 1:
			parameters = pei.getSollParameters();
			break;
		case 2:
			parameters = pei.getDefaultParameters();
			break;
		case 3:
			parameters = pei.getIstParameters();
			break;
		default:
			break;
		}
		if (null == parameters) {
			throw new IllegalArgumentException(
					"Index " + getIndex() + " wird nicht unterstützt.\n" + "Gültige Indices sind von 0 bis 3,"
							+ " entsprechend dens Aspekten " + Arrays.toString(ParameterEditor.VALID_INDICES));
		}
		final StringBuilder b = new StringBuilder("Tabellen-Parameterdialog: ");
		b.append(parameters.length);
		b.append(" Objekt");
		if (parameters.length > 1) {
			b.append('e');
		}
		b.append(" in Bearbeitung");
		form.setText(b.toString());

		final boolean readOnly = getIndex() > ParameterEditor.EDITED_INDEX;
		erzeugeAbschnittDaten(form, toolkit, readOnly);
		// Tabelle wird erst in TableParamEditorFormPart.refresh() mit Leben
		// gefüllt

		final TableParamEditorFormPart formPart = new TableParamEditorFormPart(readOnly, viewer);
		managedForm.addPart(formPart);
		getEditor().getSite().setSelectionProvider(viewer);

		final IToolBarManager toolBarManager = form.getToolBarManager();
		final ExportSWTTableViewerToCSVorHTMLAction exportAction = new ExportSWTTableViewerToCSVorHTMLAction(viewer);
		exportAction.setTitleText(((ParameterEditor) getEditor()).getDefaultHeader(true));
		toolBarManager.add(exportAction);
		toolBarManager.update(true);
		setExportAction(exportAction);

		super.createFormContent(managedForm);
	}

	/**
	 * Erzeugt die Modellobjekte, die in der Tabelle verarbeitet werden, aus den
	 * zugrunde liegenden Parameter-Objekten.
	 *
	 * @param parameters
	 *            die zugrunde liegenden Parameterobjekte.
	 * @return die Modellobjekte. Das Feld ist stets genau so groß wie das
	 *         übergebene Parameterfeld.
	 */
	public ParameterTableEditorItem[] createTableItems(final Parameter[] parameters) {
		final ParameterTableEditorItem[] tableItems = new ParameterTableEditorItem[parameters.length];
		int loop;
		for (loop = 0; loop < tableItems.length; ++loop) {
			tableItems[loop] = new ParameterTableEditorItem(parameters[loop], false, false);
			int innerLoop;
			for (innerLoop = 0; innerLoop < loop; ++innerLoop) {
				// tableItems[loop].catchUpBiDirectional(tableItems[innerLoop]);
				tableItems[loop].catchUpByData(tableItems[innerLoop]);
			}
		}
		for (loop = 0; loop < tableItems.length; ++loop) {
			tableItems[loop].run();
		}
		return tableItems;
	}

	/**
	 * Erzeugt den eigentlichen Abschnitt, der die Editortabelle enthält.
	 *
	 * @param form
	 *            das enthaltende Form
	 * @param toolkit
	 *            das Toolkit zum Erzeugen neuer Controls.
	 * @param readOnly
	 *            true - Tabelle ist read-only (hat also z.B. keinen Cursor).
	 */
	private void erzeugeAbschnittDaten(final ScrolledForm form, final FormToolkit toolkit, final boolean readOnly) {
		// a reasonable default value. The exact value will be determined later
		final int defaultNumColumns = 10;
		datenSektion = toolkit.createSection(form.getBody(), Section.DESCRIPTION | ExpandableComposite.TITLE_BAR
				| ExpandableComposite.TWISTIE | ExpandableComposite.EXPANDED);
		GridData gd = new GridData(SWT.LEFT, SWT.TOP, true, true);
		datenSektion.setLayoutData(gd);
		datenSektion.setText("Datenbereich");
		datenSektion.setDescription(
				"In diesem Abschnitt erfolgt das Editieren (" + defaultNumColumns + " Einzelattribute)");
		datenSektion.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				form.reflow(true);
			}
		});

		// Controls erzeugen
		final Composite sectionClient = toolkit.createComposite(datenSektion);
		sectionClient.setLayout(new GridLayout(1, false));
		gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
		gd.heightHint = 100;
		gd.widthHint = (defaultNumColumns * TableParamEditorFormPage.DEFAULT_COLUMN_WIDTH) + 150;
		sectionClient.setLayoutData(gd);

		final Table table = toolkit.createTable(sectionClient,
				SWT.MULTI | SWT.BORDER | SWT.WRAP | SWT.H_SCROLL | SWT.V_SCROLL);
		gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
		gd.heightHint = 100;
		gd.widthHint = (defaultNumColumns * TableParamEditorFormPage.DEFAULT_COLUMN_WIDTH) + 150;
		table.setLayoutData(gd);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		if (!readOnly) {
			setUpTableCursor(table);
		}
		viewer = new TableViewer(table);
		viewer.setLabelProvider(new ParamTableEditorLabelProvider((ParameterEditorInput) getEditorInput(), readOnly));
		viewer.setContentProvider(new ArrayContentProvider());
		if (!readOnly) {
			viewer.addDoubleClickListener(new TableParamEditorDoubleClickListener(this));
			createContextMenu(viewer);
		}

		datenSektion.setClient(sectionClient);
	}

	/**
	 * Füllt die Tabelle mit den übergebenen Objekten. Dabei werden zunächst alle
	 * eventuell noch vorhandenen Zeilen und Spalten entfernt.
	 *
	 * @param tableItems
	 *            die Objekte
	 */
	public void fillTable(final ParameterTableEditorItem[] tableItems) {
		clearTable();
		final Table table = viewer.getTable();
		final List<PlainDataValue> plainDataValues = tableItems[0].getPlainDataValues();
		final int numColumns = plainDataValues.size();

		TableColumn column = new TableColumn(table, SWT.NONE);
		column.setText("Objekt");
		column.setWidth(150);

		int loop;
		for (loop = 0; loop < numColumns; ++loop) {
			column = new TableColumn(table, SWT.NONE);
			column.setWidth(TableParamEditorFormPage.DEFAULT_COLUMN_WIDTH);
			final PlainDataValue plainDataValue = plainDataValues.get(loop);
			column.setText(plainDataValue.getPath());
			final String genericInfoText = ParamUtils.getInfotextForAttributeType(plainDataValue.getAttributeType(),
					true);
			String completeInfoText = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_VOLLSTAENDIER_ATTRNAME;
			completeInfoText += plainDataValue.getPath();
			completeInfoText += '\n';
			completeInfoText += genericInfoText;
			column.setToolTipText(completeInfoText);
		}
		viewer.setInput(tableItems);
		for (final TableColumn c : table.getColumns()) {
			c.pack();
		}
		// Platz für den * schaffen
		column = table.getColumn(0);
		column.setWidth(16 + column.getWidth());
		fitTable();
	}

	/**
	 * Die Breite der Tabelle (d.h. der Griddata der Tabelle) auf die Summe der
	 * Spaltenbreiten setzen. Die Höhe der Tabelle aus der Anzahl der Zeilen
	 * bestimmen.
	 */
	private void fitTable() {
		final Table table = viewer.getTable();
		final GridData tableGridData = (GridData) table.getLayoutData();
		final Composite sectionClient = (Composite) datenSektion.getClient();
		final GridData sectionClientGridData = (GridData) sectionClient.getLayoutData();
		final Point formSize = getManagedForm().getForm().getSize();

		// Breite ist Minimum aus verfügbarem und benötigtem Platz
		int realWidth = 0;
		for (final TableColumn c : table.getColumns()) {
			realWidth += c.getWidth();
		}
		tableGridData.widthHint = Math.min(formSize.x - 40, realWidth);
		sectionClientGridData.widthHint = Math.min(formSize.x - 40, realWidth);
		final int realHeight;
		if (table.getItemCount() > 0) {
			final Rectangle r = table.getItem(0).getBounds();
			int cellheight = r.height;
			if (table.getLinesVisible()) {
				++cellheight;
			}
			realHeight = table.getHeaderHeight() + (table.getItemCount() * cellheight);
		} else {
			realHeight = 26;
		}
		tableGridData.heightHint = Math.min(formSize.y - 110, realHeight);
		sectionClientGridData.heightHint = Math.min(formSize.y - 110, realHeight);

		// HINWEIS: Das ist der alte Algorithmus, der die Tabelle ohne
		// Scrollbars genau einpasst!
		// GridData gd = (GridData) table.getLayoutData();
		// int realWidth = 0;
		// for (TableColumn c : table.getColumns()) {
		// realWidth += c.getWidth();
		// }
		// gd.widthHint = realWidth;
		// final Rectangle r = table.getItem(0).getBounds();
		// int cellheight = r.height;
		// if (table.getLinesVisible()) {
		// ++cellheight;
		// }
		// gd.heightHint = table.getHeaderHeight() + table.getItemCount()
		// * cellheight;
		// DEBUG.info("fitTable Height", gd.heightHint);
		// final Composite sectionClient = (Composite) datenSektion.getClient();
		// gd = (GridData) sectionClient.getLayoutData();
		// gd.widthHint = realWidth;
		// DEBUG.info("fitTable Width", gd.widthHint);
	}

	/**
	 * Aktualisiert den Detail-Beschreibungstest der Datensektion.
	 *
	 * @param tableItems
	 *            die aktuellen Tableitems. Muss mindestens ein Element enthalten.
	 */
	public void updateDatenSectionDescription(final ParameterTableEditorItem[] tableItems) {
		final int numEinzelAttribute = tableItems[0].getPlainDataValues().size();
		datenSektion.setDescription(
				"In diesem Abschnitt erfolgt das Editieren (" + numEinzelAttribute + " Einzelattribute)");
	}

	/**
	 * Entfernt alle Zeilen und Spalten aus der Tabelle.
	 */
	public void clearTable() {
		final Table table = viewer.getTable();
		table.removeAll();
		for (final TableColumn c : table.getColumns()) {
			c.dispose();
		}
	}

	/**
	 * Erzeugen und konfigurieren eines Tablecursor zum Addressieren individueller
	 * Zellen der Tabelle mit der Maus.
	 *
	 * @param table
	 *            die Tabelle, die den Cursor enthalten soll.
	 */
	private void setUpTableCursor(final Table table) {
		final TableCursor cursor = new TableCursor(table, SWT.NONE);
		table.setData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR, cursor);
		cursor.setData(TableParamEditorFormPage.CURSORDATA_KEY_TABLE, table);
		cursor.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				// DEBUG.finest("widgetDefaultSelected row="
				// + cursor.getRow() + ", col=" + cursor.getColumn());
				// contextMenuManager.getMenu().setVisible(true);
			}

			@Override
			public void widgetSelected(final SelectionEvent e) {
				// DEBUG.finest("widgetSelected row=" + cursor.getRow()
				// + ", col=" + cursor.getColumn() + ", event=" + e);
			}
		});
		// Hide the TableCursor when the user hits the "CTRL" or "SHIFT" key.
		// This allows the user to select multiple items in the table.
		cursor.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if ((e.keyCode == SWT.CTRL) || (e.keyCode == SWT.SHIFT) || ((e.stateMask & SWT.CONTROL) != 0)
						|| ((e.stateMask & SWT.SHIFT) != 0)) {
					cursor.setVisible(false);
				}
			}
		});
		// When the user double clicks in the TableCursor, pop up an appropriate
		// Editor
		cursor.addMouseListener(new TableParamEditorDoubleClickListener(this));

		// Show the TableCursor when the user releases the "SHIFT" or "CTRL"
		// key.
		// This signals the end of the multiple selection task.
		table.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(final KeyEvent e) {
				if ((e.keyCode == SWT.CONTROL) && ((e.stateMask & SWT.SHIFT) != 0)) {
					return;
				}
				if ((e.keyCode == SWT.SHIFT) && ((e.stateMask & SWT.CONTROL) != 0)) {
					return;
				}
				if ((e.keyCode != SWT.CONTROL) && ((e.stateMask & SWT.CONTROL) != 0)) {
					return;
				}
				if ((e.keyCode != SWT.SHIFT) && ((e.stateMask & SWT.SHIFT) != 0)) {
					return;
				}

				final TableItem[] selection = table.getSelection();
				final TableItem row = (selection.length == 0) ? table.getItem(table.getTopIndex()) : selection[0];
				table.showItem(row);
				// cursor.setSelection(row, 0);
				cursor.setVisible(true);
				cursor.setFocus();
			}
		});

	}

	/**
	 * creates the popupmenus and initalizes the different actions for manipulating
	 * the array items.
	 *
	 * @param tableViewer
	 *            der TableViewer, an dem das Kontextmenü registriert werden soll
	 */
	private void createContextMenu(final TableViewer tableViewer) {
		final Table table = tableViewer.getTable();
		final TableCursor cursor = (TableCursor) table.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
		if (null == cursor) {
			return;
		}
		elementAdd = new TableParamEditorAddElementAction(tableViewer, this);
		elementInvalidate = new TableParamEditorInvalidateElementAction(tableViewer, this);
		contextMenuManager = new MenuManager();
		contextMenuManager.setRemoveAllWhenShown(true);
		contextMenuManager.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(final IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});

		final Menu menu = contextMenuManager.createContextMenu(tableViewer.getControl());
		cursor.setMenu(menu);
	}

	/**
	 * fills the context menus according to the selected {@link PlainDataValue}.
	 *
	 * @param mgr
	 *            - the IMenuManager
	 */
	void fillContextMenu(final IMenuManager mgr) {
		try {
			mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
			final TableCursor cursor = (TableCursor) viewer.getTable()
					.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
			final ParameterTableEditorItem item = (ParameterTableEditorItem) cursor.getRow().getData();
			final int effCol = cursor.getColumn() - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
			PlainDataValue editedValue = null;
			if ((effCol >= 0) && (effCol < item.getPlainDataValues().size())) {
				editedValue = item.getPlainDataValues().get(effCol);
			}
			elementAdd.setEditedValue(item, effCol);
			elementInvalidate.setEditedValue(item, effCol);
			TableParamEditorFormPage.DEBUG.finest("fillCintextMenu: selectedObject=" + editedValue);
			boolean filled = false;
			// if (einfügen möglich) {
			contextMenuManager.add(elementAdd);
			contextMenuManager.add(elementInvalidate);
			filled = true;
			// }
			mgr.setVisible(filled);
		} catch (final IllegalStateException e) {
			TableParamEditorPlugin.getDefault().getLogger().error("Fehler beim Befüllen des Kontextmenüs", e);
		}
	}

	/**
	 * Die eigentliche Behandlung der Attributpropagierung.
	 *
	 * @param item
	 *            das ursprünglich geänderte Item.
	 * @param index
	 *            der Index des geänderten PlainDataValue innerhalb item.
	 * @param oldValue
	 *            alter Wert, Typ ist abhängig vom Attributtyp (siehe letzter
	 *            Parameter). Kann auch nul sein, wenn es ein Lückenfüller-Objekt
	 *            war, was nun erstmals einen Wert zugewiesen bekam.
	 * @param newValue
	 *            neuer Wert, Typ ist abhängig vom Attributtyp (siehe letzter
	 *            Parameter). Kann auch null sein, wenn
	 *            {@link PlainDataValue#invalidate()} die Ursache für die Änderung
	 *            war.
	 * @param attributeType
	 *            Attributtyp (Referenz, Integer, Text, ...) des PlainDataValue
	 *            innerhalb item.
	 */
	private void handleAttributePropagation(final ParameterTableEditorItem item, final int index, final Object oldValue,
			final Object newValue, final AttributeType attributeType) {
		switch (getParameterEditorInput().getAttributePropagationMode()) {
		case ONLY_EDITED:
			return;
		case EDITED_AND_SELECTION:
			throw new UnsupportedOperationException("Attributpropagierung auf Auswahl wird nicht unterstützt");
		case ALL:
			break;
		default:
			break;
		}
		final PlainDataValue editedValue = item.getPlainDataValues().get(index);
		final ParameterTableEditorItem[] tableItems = (ParameterTableEditorItem[]) this.viewer.getInput();
		for (final ParameterTableEditorItem tableItem : tableItems) {
			if (item == tableItem) {
				continue;
			}
			final int firstUndefinedIndex = tableItem.getFirstUndefinedIndexOfArray(index);
			final PlainDataValue valueToModify = tableItem.getPlainDataValues().get(index);
			final ArrayNameAndIndex ari = ParameterTableEditorItem.getArrayIndexFromPath(valueToModify.getPath());
			// if (null != valueToModify.getParentAttributeSet()) {
			// final List<PlainDataValue> allValuesOfList = tableItem
			// .getValuesOfList(ari, 0, false);
			// DEBUG.finest(allValuesOfList);
			// // --firstUndefinedIndex;
			// }
			boolean modified = false;
			if (null == newValue) {
				if ((-2 == ari.getIndex()) || ((ari.getIndex() >= 0) && (((firstUndefinedIndex - 1) == ari.getIndex())
						|| (firstUndefinedIndex == ari.getIndex())))) {
					if (valueToModify.isValid()) {
						valueToModify.invalidate();
						modified = true;
					}
				}
			} else {
				if (valueToModify.isValid() || ((ari.getIndex() >= 0) && (firstUndefinedIndex == ari.getIndex()))) {
					final String oldPid = valueToModify.getPid();
					if (!editedValue.getPid().equals(oldPid)) {
						valueToModify.setPid(editedValue.getPid());
						modified = true;
					}
				}
			}
			// if (-2 == ari.index
			// || (ari.index >= 0 && (firstUndefinedIndex - 1) == ari.index)) {
			// if (null == newValue) {
			// valueToModify.invalidate();
			// } else {
			// valueToModify.setPid(editedValue.getPid());
			// }
			// }
			if (modified) {
				tableItem.setDirty(true);
			}
		}
	}

	@Override
	public void valueModified(final ParameterTableEditorItem item, final int index, final Object oldValue,
			final Object newValue, final AttributeType attributeType) {
		final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
		final PlainDataValue editedValue = plainDataValues.get(index);
		TableParamEditorFormPage.DEBUG.fine("Item modifiziert: " + editedValue + "\nAlt: " + oldValue + "\nNeu: "
				+ newValue + "\nATT: " + attributeType);
		TableParamEditorFormPart formPart = null;
		for (final IFormPart part0 : getManagedForm().getParts()) {
			if (part0 instanceof TableParamEditorFormPart) {
				formPart = (TableParamEditorFormPart) part0;
				break;
			}
		}
		if (formPart != null) {
			formPart.markDirty();
			handleAttributePropagation(item, index, oldValue, newValue, attributeType);
		}
		updateDatenSectionDescription(new ParameterTableEditorItem[] { item });
		fitTable();
		((Composite) datenSektion.getClient()).layout();
		viewer.refresh();
		moveTableCursor();
	}

	/**
	 * Den Tablecursor versetzen, damit das soeben bearbeitete Feld sichtbar wird.
	 * Wenn möglich setzen wir den Cursor eine Zelle nach rechts, wenn wir schon
	 * ganz rechts sind, eine Zelle nach links.
	 */
	private void moveTableCursor() {
		final Table table = viewer.getTable();
		final TableCursor cursor = (TableCursor) table.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
		int cursorCol = cursor.getColumn();
		if (cursorCol == (table.getColumnCount() - 1)) {
			--cursorCol;
		} else {
			++cursorCol;
		}
		cursor.setSelection(cursor.getRow(), cursorCol);
	}

	@Override
	public void showContextMenu(final PlainDataValue editedValue, final ParameterTableEditorItem editedItem,
			final int editedIndex) {
		elementAdd.setEditedValue(editedItem, editedIndex);
		contextMenuManager.setVisible(true);
	}

	@Override
	public void layoutModified() {
		fitTable();
		getManagedForm().reflow(true);
	}

	/**
	 * Liefert das Input-Objekt des Editors. Über dieses hat man Zugriff auf die zu
	 * bearbeitenden Parameter.
	 *
	 * @return das Input-Objekt.
	 */
	public ParameterEditorInput getParameterEditorInput() {
		return (ParameterEditorInput) getEditorInput();
	}

	@Override
	public PagePrint getDruckAuftrag() {
		final Table table = viewer.getTable();
		final FontData systemFontData = table.getDisplay().getSystemFont().getFontData()[0];

		final String header = ((ParameterEditor) getEditor()).getDefaultHeader(false);

		final GridPrint grid = new GridPrint();
		grid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
		grid.add(new TextPrint(header,
				new FontData(systemFontData.getName(), systemFontData.getHeight() + 2, SWT.BOLD)));

		final GridPrint tabelle = new GridPrint();
		final DefaultGridLook look = new DefaultGridLook();
		look.setHeaderBackground(new RGB(200, 200, 200));
		if (viewer.getLabelProvider() instanceof ITableColorProvider) {
			final ITableColorProvider colorProvider = (ITableColorProvider) viewer.getLabelProvider();
			look.setBodyBackgroundProvider(new CellBackgroundProvider() {

				private final RGB white = new RGB(255, 255, 255);

				@Override
				public RGB getCellBackground(final int row, final int column, final int colspan) {
					final ParameterTableEditorItem item = (ParameterTableEditorItem) table.getItem(row).getData();
					final Color bColor = colorProvider.getBackground(item, column);
					if (null == bColor) {
						return white;
					}
					return bColor.getRGB();
				}
			});
		}

		TextStyle defaultTextStyle = new TextStyle();
		final TextStyle headerStyle = defaultTextStyle.font(systemFontData.getName(), systemFontData.getHeight(),
				SWT.BOLD);
		defaultTextStyle = defaultTextStyle.font(systemFontData.getName(), systemFontData.getHeight(), SWT.NORMAL);

		look.setCellBorder(new LineBorder(new RGB(0, 0, 0)));
		tabelle.setLook(look);

		for (final TableColumn column : table.getColumns()) {
			final GridColumn gridColumn = new GridColumn(SWT.LEFT, GridColumn.DEFAULT_SIZE, 1);
			tabelle.addColumn(gridColumn);
			tabelle.addHeader(headerStyle.create(column.getText()));
		}

		for (final TableItem item : table.getItems()) {
			for (int colIdx = 0; colIdx < table.getColumns().length; ++colIdx) {
				final TextStyle tStyle = defaultTextStyle.foreground(item.getForeground(colIdx).getRGB());
				final Image image = item.getImage(colIdx);
				final TextPrint tPrint = tStyle.create(item.getText(colIdx));
				if (image != null) {
					final GridPrint combinedGrid = new GridPrint();
					combinedGrid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE,
							GridColumn.DEFAULT_WEIGHT));
					combinedGrid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE,
							GridColumn.DEFAULT_WEIGHT));
					combinedGrid.add(SWT.LEFT, SWT.CENTER, new ImagePrint(image.getImageData()));
					combinedGrid.add(SWT.LEFT, SWT.CENTER, tPrint);
					tabelle.add(combinedGrid);
				} else {
					tabelle.add(SWT.LEFT, SWT.CENTER, tPrint);
				}
			}
		}

		grid.add(tabelle);

		return new PagePrint(new BigPrint(grid));
	}

	@Override
	public String getTitel() {
		// The default provided by ParameterEditor is sufficient
		return null;
	}

}
