/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.editor.IFormPage;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.ParameterEditor;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorSaveMode;
import de.bsvrz.buv.plugin.param.editors.ParameterSaveResultEnum;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver.ParameterSaveUrsache;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParameterTableEditorItem;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.daten.DataWithTime;

/**
 * FormPart für den TabellenEditor.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class TableParamEditorFormPart extends AbstractFormPart {

	/**
	 * die einzelnen Zeilen im Tabelleneditor.
	 */
	private final List<ParameterTableEditorItem> tableItems = new ArrayList<>();

	/**
	 * true - Kein Editieren und damit kein Speichern möglich.
	 */
	private final boolean readOnly;

	/**
	 * der Tableviewer, in welchem sich {@link ParameterTableEditorItem}s befinden,
	 * über die man Zugriff auf die zu bearbeitenden Parameter hat.
	 */
	private final TableViewer viewer;

	/**
	 * Konstruktor speichert nur Member.
	 *
	 * @param readOnly
	 *            true - Kein Editieren und damit kein Speichern möglich.
	 * @param viewer
	 *            der Tableviewer, in welchem sich {@link ParameterTableEditorItem}s
	 *            befinden, über die man Zugriff auf die zu bearbeitenden Parameter
	 *            hat.
	 */
	public TableParamEditorFormPart(final boolean readOnly, final TableViewer viewer) {
		this.readOnly = readOnly;
		this.viewer = viewer;
	}

	@Override
	public void commit(final boolean onSave) {
		if (!onSave) {
			return;
		}
		if (readOnly) {
			// Should have no effect since we became never dirty but...
			super.commit(onSave);
			return;
		}
		final TableParamEditorFormPage page = (TableParamEditorFormPage) getManagedForm().getContainer();
		final ParameterEditorSaveMode saveMode = page.getParameterEditorInput().getSaveMode();
		Parameter[] editedParameters = null;
		switch (saveMode) {
		case ALL:
			editedParameters = new Parameter[tableItems.size()];
			int parameterLoop = 0;
			for (parameterLoop = 0; parameterLoop < tableItems.size(); ++parameterLoop) {
				final ParameterTableEditorItem item = tableItems.get(parameterLoop);
				editedParameters[parameterLoop] = new Parameter(tableItems.get(parameterLoop).getParameter().getInfo(),
						new DataWithTime(item.createData(), System.currentTimeMillis()));
			}
			break;
		case ONLY_MODIFIED:
			final List<Parameter> parameterToSave = new ArrayList<>(tableItems.size());
			for (parameterLoop = 0; parameterLoop < tableItems.size(); ++parameterLoop) {
				final ParameterTableEditorItem item = tableItems.get(parameterLoop);
				if (item.isDirty()) {
					parameterToSave.add(new Parameter(tableItems.get(parameterLoop).getParameter().getInfo(),
							new DataWithTime(item.createData(), System.currentTimeMillis())));
				}
			}
			editedParameters = parameterToSave.toArray(new Parameter[parameterToSave.size()]);
			break;
		default:
			break;
		}
		Assert.isNotNull(editedParameters,
				Zeichenketten.PLUGIN_PARAM_HINWEIS_ZU_SPEICHERNDE_PARAMETER_NICHT_BESTIMMBAR + saveMode);
		final ParameterSaveResultEnum saveResult = ParameterSaver.saveParameters(editedParameters, ParameterSaver
				.buildDefaultUrsache(editedParameters, editedParameters, ParameterSaveUrsache.PARAMETRIERT));
		switch (saveResult) {
		case PARAMETER_SAVE_FAILURE:
			/* Make Eclipse display an error dialog */
			throw new IllegalStateException(ParameterSaver.getLastError());
		case PARAMETER_SAVE_CANCEL:
			// Do nothing
			break;
		case PARAMETER_SAVE_SUCCESS:
			super.commit(onSave);
			getManagedForm().dirtyStateChanged();
			for (final ParameterTableEditorItem item : tableItems) {
				item.setDirty(false);
			}
			refreshViewer();
			break;
		default:
			break;
		}
	}

	@Override
	public void markStale() {
		if (!readOnly) {
			return;
		}
		final IFormPage formPage = (IFormPage) getManagedForm().getContainer();
		if (1 == formPage.getIndex()) {
			final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
			try {
				editorInput.sollParameterAbfragen();
				super.markStale();
				final Parameter[] neueSollParameter = editorInput.getSollParameters();
				viewer.setInput(((TableParamEditorFormPage) formPage).createTableItems(neueSollParameter));
				viewer.refresh();
			} catch (final ParameterClientException e) {
				TableParamEditorPlugin.getDefault().getLogger().error("Fehler bei Aktualisierung der Soll-Parameter",
						e);
			}
		}
		if (3 == formPage.getIndex()) {
			final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
			editorInput.istParameterAbfragen();
			super.markStale();
			final Parameter[] neueIstParameter = editorInput.getIstParameters();
			viewer.setInput(((TableParamEditorFormPage) formPage).createTableItems(neueIstParameter));
			viewer.refresh();
		}
	}

	@Override
	public void markDirty() {
		if (readOnly) {
			return;
		}
		final IFormPage formPage = (IFormPage) getManagedForm().getContainer();
		final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
		for (final Integer index : editorInput.getSelectedParameters()) {
			final ParameterTableEditorItem tableItem = tableItems.get(index);
			if (null != tableItem) {
				tableItem.setDirty(true);
				viewer.refresh(tableItem);
			}
		}
		super.markDirty();
	}

	@Override
	public void refresh() {
		final TableParamEditorFormPage formPage = (TableParamEditorFormPage) getManagedForm().getContainer();
		final ParameterEditorInput pei = (ParameterEditorInput) formPage.getEditorInput();
		Parameter[] parameters = null;
		switch (formPage.getIndex()) {
		case 0:
			parameters = pei.getParameters();
			break;
		case 1:
			parameters = pei.getSollParameters();
			break;
		case 2:
			parameters = pei.getDefaultParameters();
			break;
		case 3:
			parameters = pei.getIstParameters();
			break;
		default:
			break;
		}
		if (null == parameters) {
			throw new IllegalArgumentException(
					"Index " + formPage.getIndex() + " wird nicht unterstützt.\n" + "Gültige Indices sind von 0 bis 3,"
							+ " entsprechend den Aspekten " + Arrays.toString(ParameterEditor.VALID_INDICES));
		}
		final ParameterTableEditorItem[] neueTableItems;
		if (tableItems.isEmpty()) {
			neueTableItems = formPage.createTableItems(parameters);
		} else {
			neueTableItems = new ParameterTableEditorItem[tableItems.size()];
			int loop;
			for (loop = 0; loop < tableItems.size(); ++loop) {
				if (pei.getSelectedParameters().contains(loop) || readOnly) {
					neueTableItems[loop] = new ParameterTableEditorItem(parameters[loop], false, false);
				} else {
					neueTableItems[loop] = tableItems.get(loop).createModifiableCopy();
				}
				int innerLoop;
				for (innerLoop = 0; innerLoop < loop; ++innerLoop) {
					neueTableItems[loop].catchUpByData(neueTableItems[innerLoop]);
				}
			}
			for (loop = 0; loop < neueTableItems.length; ++loop) {
				neueTableItems[loop].run();
			}
			tableItems.clear();
		}
		for (final ParameterTableEditorItem item : neueTableItems) {
			tableItems.add(item);
		}
		formPage.fillTable(neueTableItems);
		formPage.updateDatenSectionDescription(neueTableItems);

		// Globalen Dirty-Status bestimmen und ggf. publizieren.
		int loop;
		boolean globalDirtyState = false;
		for (loop = 0; loop < tableItems.size(); ++loop) {
			if (tableItems.get(loop).isDirty()) {
				globalDirtyState = true;
				break;
			}
		}
		if (getManagedForm().isDirty() != globalDirtyState) {
			// dirty = false; geht nicht da private!
			super.refresh();
			getManagedForm().dirtyStateChanged();
		}
	}

	/**
	 * Aktualisieren des gesamten Viewers über eine {@link Runnable}. Die Methode
	 * kann also von jedem Thread aus benutzt werden.
	 */
	private void refreshViewer() {
		if (!viewer.getControl().getDisplay().isDisposed()) {
			viewer.getControl().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					if (!viewer.getControl().isDisposed()) {
						viewer.refresh();
					}

				}
			});
		}
	}
}
