/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.actions;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.widgets.Table;

import de.bsvrz.buv.plugin.param.editors.table.TableParamEditorFormPage;
import de.bsvrz.buv.plugin.param.editors.table.edit.IParamTableEditorModifyListener;
import de.bsvrz.buv.plugin.param.editors.table.provider.ArrayNameAndIndex;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParameterTableEditorItem;
import de.bsvrz.buv.plugin.param.editors.table.provider.PlainDataValue;

/**
 * Abstrakte Basisklasse für alle Aktionen innerhalb des Tabelleneditors.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public abstract class TableParamEditorAction extends Action {

	/**
	 * der Tableviewer, an einem dessen Elementen wir eine Aktion sind. Damit haben
	 * wir die Möglichkeit, einen globalen Refresh auszuführen, falls dies im
	 * Ergebnis unserer Aktion nötig ist.
	 */
	private final TableViewer tableViewer;

	/**
	 * Das bearbeitete Attribut, verkapselt als Modellobjekt.
	 */
	private PlainDataValue editedValue;

	/**
	 * Aufgeteilter Pfad des bearbeiteten Attributs.
	 */
	private ArrayNameAndIndex ari;

	/**
	 * Bearbeitete Tabellenzeile.
	 */
	private ParameterTableEditorItem editedItem;

	/**
	 * Nummer des Attributs (Spalte) innerhalb der bearbeiteten Zeile.
	 */
	private int editedIndex;

	/**
	 * Ermöglicht Zugriff auf Objekte, die die Attributpropagierung behandeln. Kann
	 * <code>null</code> sein.
	 */
	private final IParamTableEditorModifyListener modifyListener;

	/**
	 * Konstruktor reicht nur durch bzw. merkt sich Member.
	 *
	 * @param text
	 *            Bezeichnung der Aktion.
	 * @param tableViewer
	 *            der Tableviewer, an einem dessen Elementen wir eine Aktion sind.
	 * @param modifyListener
	 *            Ermöglicht Zugriff auf Objekte, die die Attributpropagierung
	 *            behandeln. Kann <code>null</code> sein.
	 */
	public TableParamEditorAction(final String text, final TableViewer tableViewer,
			final IParamTableEditorModifyListener modifyListener) {
		super(text);
		this.tableViewer = tableViewer;
		this.modifyListener = modifyListener;
	}

	/**
	 * Liefert das bearbeitete Attribut, verkapselt als Modellobjekt.
	 *
	 * @return das Attribut
	 */
	public PlainDataValue getEditedValue() {
		return editedValue;
	}

	/**
	 * Bestimmt das bearbeitete Attribut aus Zeile und Spaltenindex.
	 *
	 * @param editedRow
	 *            bearbeitete Zeile
	 * @param editedIdx
	 *            bearbeitete Spalte
	 */
	public void setEditedValue(final ParameterTableEditorItem editedRow, final int editedIdx) {
		this.editedItem = editedRow;
		this.editedIndex = editedIdx;
		this.editedValue = editedRow.getPlainDataValues().get(editedIdx);
		if (null == editedValue) {
			setEnabled(false);
			return;
		}
		ari = ParameterTableEditorItem.getArrayIndexFromPath(editedValue.getPath());
	}

	/**
	 * Aktualisiert die Zeile der Tabelle, in der sich das bearbeitete Attribut
	 * befindet.
	 */
	protected void refreshEditedItem() {
		final Table table = tableViewer.getTable();
		final TableCursor cursor = (TableCursor) table.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
		tableViewer.refresh(cursor.getRow().getData());
	}

	/**
	 * Liefert den Tableviewer, an einem dessen Elementen wir eine Aktion sind.
	 * Damit haben wir die Möglichkeit, einen globalen Refresh auszuführen, falls
	 * dies im Ergebnis unserer Aktion nötig ist.
	 *
	 * @return der TableViewer
	 */
	public TableViewer getTableViewer() {
		return tableViewer;
	}

	/**
	 * Liefert ein Objekt, welches Zugriff auf Objekte ermöglicht, die die
	 * Attributpropagierung behandeln. Kann <code>null</code> sein.
	 *
	 * @return das Objekt
	 */
	public IParamTableEditorModifyListener getModifyListener() {
		return modifyListener;
	}

	/**
	 * Liefert die bearbeitete Tabellenzeile.
	 *
	 * @return die Zeile
	 */
	protected ParameterTableEditorItem getEditedItem() {
		return editedItem;
	}

	/**
	 * Liefert die Nummer des Attributs (Spalte) innerhalb der bearbeiteten Zeile.
	 *
	 * @return der Index
	 */
	protected final int getEditedIndex() {
		return editedIndex;
	}

	/**
	 * Liefert den aufgeteilten Pfad des bearbeiteten Attributs.
	 *
	 * @return der Pfad
	 */
	protected final ArrayNameAndIndex getAri() {
		return ari;
	}

}
