/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.edit;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import de.bsvrz.buv.plugin.param.ParamSharedImage;
import de.bsvrz.buv.plugin.param.ParamUtils;
import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.table.TableParamEditorFormPage;
import de.bsvrz.buv.plugin.param.editors.table.actions.TableParamEditorAction;
import de.bsvrz.buv.plugin.param.editors.table.provider.ArrayInTableInfo;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParameterTableEditorItem;
import de.bsvrz.buv.plugin.param.editors.table.provider.PlainDataValue;
import de.bsvrz.dav.daf.main.config.AttributeSet;

/**
 * Aktionsklasse zum Einfügen eines Elements in ein Feld.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class TableParamEditorAddElementAction extends TableParamEditorAction {

	/**
	 * Konstruktor reicht nur die Werte durch und setzt den Image Descriptor.
	 *
	 * @param tableViewer
	 *            der Tableviewer, aus dem heraus die Aktion aufgerufen wurde. Der
	 *            wird nach dem Ausführen aktualisiert.
	 * @param modifyListener
	 *            Ein Modifylistener (kann <code>null</code> sein), der über die
	 *            Änderung benachrichtigt wird.
	 */
	public TableParamEditorAddElementAction(final TableViewer tableViewer,
			final IParamTableEditorModifyListener modifyListener) {
		super("Feldelement hinzufügen", tableViewer, modifyListener);
		setImageDescriptor(ParamSharedImage.ADD.getImageDescriptor());
	}

	@Override
	public void run() {
		final Table table = getTableViewer().getTable();
		final TableCursor cursor = (TableCursor) table.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
		final int selectedColumn = cursor.getColumn();

		final ParameterTableEditorItem[] tableItems = (ParameterTableEditorItem[]) getTableViewer().getInput();

		// Anzahl benötigter neuer Spalten bestimmen. Für atomare Felder 1
		// (Startwert). Für Felder von Listen die Anzahl der Attribute in der
		// Liste. Falls das Feld leer war, eins abziehen, da es ja schon eine
		// Platzhalterspalte gibt.
		int necessaryColumns = 1;
		final AttributeSet parentAttributeSet = getEditedValue().getParentAttributeSet();
		if (getEditedValue().getPath().endsWith(Zeichenketten.EMPTY_INDICATOR)) {
			--necessaryColumns;
		}

		// Bei atomaren Feldern fügen wir genau hinter dem editedValue ein
		int offset = 1;
		int pos = 0;
		if (null != parentAttributeSet) {
			final List<PlainDataValue> reducedList = tableItems[0].getValuesOfList(getAri(), 0, false);
			pos = ParameterTableEditorItem.bestimmePositionInnerhalbFeld(reducedList, getEditedValue().getPath());
			offset = reducedList.size() - pos;
			if (-2 == getAri().getIndex()) {
				++offset;
			}

			// Jetzt leere PlainDataValues erzeugen und in das jeweilige item
			// einfügen. Dabei implizit die Anzahl der zusätzlich erforderlichen
			// Tabellenspalten ermitteln
			final List<String> neuePfade = new ArrayList<>();
			for (final ParameterTableEditorItem item : tableItems) {
				String prefix = getAri().getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR;
				boolean firstElement;
				if (-2 == getAri().getIndex()) {
					prefix += "0";
					firstElement = true;
				} else {
					prefix += getAri().getIndex() + 1;
					firstElement = false;
				}
				final int effTargetCol = (selectedColumn + offset) - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
				final List<PlainDataValue> newValues = ParameterTableEditorItem.createPlainDataValues(prefix,
						parentAttributeSet);
				final List<String> newPaths = new ArrayList<>();
				if (newPaths.isEmpty()) {
					int pathLoop;
					for (pathLoop = 0; pathLoop < newValues.size(); ++pathLoop) {
						newPaths.add(newValues.get(pathLoop).getPath());
					}
					necessaryColumns = newValues.size();
				}
				if (firstElement) {
					--necessaryColumns;
					table.getColumn(selectedColumn).setText(newPaths.get(0));
					final PlainDataValue stayingValue = item.getPlainDataValues().get(effTargetCol - 1);
					stayingValue.setPath(newValues.get(0).getPath());
					stayingValue.setAttributeNameFromPath();
					stayingValue.setAttributeType(newValues.get(0).getAttributeType());
					stayingValue.setParentAttributeSet(newValues.get(0).getParentAttributeSet());
					newPaths.remove(0);
					newValues.remove(0);
				}
				if (neuePfade.isEmpty()) {
					neuePfade.addAll(newPaths);
				}
				item.getPlainDataValues().addAll(effTargetCol, newValues);
			}
			// Jetzt die Spalten erzeugen, an der richtigen Position.
			int newColumnLoop;
			for (newColumnLoop = 0; newColumnLoop < necessaryColumns; ++newColumnLoop) {
				final TableColumn column = new TableColumn(table, SWT.NONE, selectedColumn + offset + newColumnLoop);
				column.setWidth(100);
				column.setText(neuePfade.get(newColumnLoop));
				final String genericInfoText = ParamUtils
						.getInfotextForAttributeType(getEditedValue().getAttributeType(), true);
				String completeInfoText = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_VOLLSTAENDIER_ATTRNAME;
				completeInfoText += neuePfade.get(newColumnLoop);
				completeInfoText += '\n';
				completeInfoText += genericInfoText;
				column.setToolTipText(completeInfoText);
			}
		} else {
			// Eine oder keine neue Spalte erzeugen
			int newColumnLoop;
			for (newColumnLoop = 0; newColumnLoop < necessaryColumns; ++newColumnLoop) {
				final TableColumn column = new TableColumn(table, SWT.NONE, selectedColumn + offset + newColumnLoop);
				column.setWidth(100);
				final String neuerPfad = getAri().getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR
						+ (getAri().getIndex() + 1);
				column.setText(neuerPfad);
				final String genericInfoText = ParamUtils
						.getInfotextForAttributeType(getEditedValue().getAttributeType(), true);
				String completeInfoText = Zeichenketten.PLUGIN_PARAM_BEZEICHNER_VOLLSTAENDIER_ATTRNAME;
				completeInfoText += neuerPfad;
				completeInfoText += '\n';
				completeInfoText += genericInfoText;
				column.setToolTipText(completeInfoText);
				for (final ParameterTableEditorItem item : tableItems) {
					final PlainDataValue newValue = new PlainDataValue(null, neuerPfad);
					newValue.setAttributeType(getEditedValue().getAttributeType());
					newValue.setParentAttributeSet(null);
					newValue.setAttributeName(getEditedValue().getAttributeName());
					final int effTargetCol = (selectedColumn + offset + newColumnLoop)
							- ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
					item.getPlainDataValues().add(effTargetCol, newValue);
				}
			}
		}
		if ((necessaryColumns > 0) && (null != getModifyListener())) {
			getModifyListener().layoutModified();
		}
		getTableViewer().refresh();
	}

	@Override
	public void setEditedValue(final ParameterTableEditorItem editedItem, final int editedIndex) {
		super.setEditedValue(editedItem, editedIndex);
		boolean canAdd = false;
		if ((-2 == getAri().getIndex()) && (null != getEditedValue().getParentAttributeSet())) {
			final ArrayInTableInfo aiti = new ArrayInTableInfo(editedItem, editedIndex, true);
			if (aiti.getNumMembers() <= 0) {
				canAdd = true;
			}
		}
		if (!canAdd && (getAri().getIndex() >= 0)) {
			final int firstUndefinedIndex = editedItem.getFirstUndefinedIndexOfArray(editedIndex);
			canAdd = (getAri().getIndex() + 1) == firstUndefinedIndex;
			if (canAdd) {
				final String searchStr = getAri().getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR
						+ firstUndefinedIndex;
				int nextAtomarIdx;
				final List<PlainDataValue> plainDataValues = editedItem.getPlainDataValues();
				for (nextAtomarIdx = editedIndex + 1; nextAtomarIdx < plainDataValues.size(); ++nextAtomarIdx) {
					final PlainDataValue nextValue = plainDataValues.get(nextAtomarIdx);
					if (nextValue.getPath().startsWith(searchStr)) {
						canAdd = false;
						break;
					}
				}
			}
		}
		setEnabled(canAdd);
	}

}
