/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.edit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.bitctrl.lib.eclipse.wizards.ComposedWizardDialog;

import de.bsvrz.buv.plugin.param.editors.attribut.DoubleAttributDialog;
import de.bsvrz.buv.plugin.param.editors.attribut.IntegerAttributDialog;
import de.bsvrz.buv.plugin.param.editors.attribut.MultiLineInputDialog;
import de.bsvrz.buv.plugin.param.editors.attribut.StringLengthInputValidator;
import de.bsvrz.buv.plugin.param.editors.table.TableParamEditorFormPage;
import de.bsvrz.buv.plugin.param.editors.table.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.editors.table.provider.ArrayInTableInfo;
import de.bsvrz.buv.plugin.param.editors.table.provider.ArrayNameAndIndex;
import de.bsvrz.buv.plugin.param.editors.table.provider.ParameterTableEditorItem;
import de.bsvrz.buv.plugin.param.editors.table.provider.PlainDataValue;
import de.bsvrz.buv.rw.bitctrl.eclipse.MultipleSelection;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectAuswahlType;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.DatumZeitDialog;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.SystemObjectAuswahlWizardPage;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.ZeitAuswahlDialog;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.ZeitDauerDialog;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.AttributeSet;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Doubleklick und Mauslistener an {@link TableViewer} und {@link TableCursor},
 * der ein Objekt vom Typ {@link ParameterTableEditorItem} erwartet und dann das
 * über Spaltenindex selektierte Objekt editiert. Dazu werden abhängig vom
 * Attributtyp entsprechende Dialoge erzeugt.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class TableParamEditorDoubleClickListener implements IDoubleClickListener, MouseListener {

	/**
	 * Logger für Debugausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Ein optionaler Listener, über den Veränderungen mitgeteilt werden können.
	 */
	private final IParamTableEditorModifyListener modifyListener;

	/**
	 * Konstruktor übernimmt nur Member.
	 *
	 * @param modifyListener
	 *            der Listener oder <code>null</code>
	 */
	public TableParamEditorDoubleClickListener(final IParamTableEditorModifyListener modifyListener) {
		this.modifyListener = modifyListener;
	}

	/**
	 * Öffnet den Standarddialog zum Eingeben von absoluten Zeitpunkten oder
	 * relativen Zeitdauern.
	 *
	 * @param editedValue
	 *            der zu bearbeitende Wert.
	 * @return Ein Feld aus zwei Strings. Erstes Objekt = alter Wert, Zweites
	 *         Objekt: Neuer Wert oder <code>null</code>, wenn keine Zuweisung
	 *         erfolgte (z.B. Dialog mit 'Abbrechen' beendet).
	 */
	private Object[] openTimeDialog(final PlainDataValue editedValue) {
		final Object[] values = new Object[2];
		values[0] = editedValue.getPid();
		values[1] = null;

		final Long oldValue = Long.parseLong(editedValue.getPid());
		long newValue = oldValue;
		final TimeAttributeType attrType = (TimeAttributeType) editedValue.getAttributeType();
		ZeitAuswahlDialog azad;
		if (attrType.isRelative()) {
			azad = new ZeitDauerDialog(null, oldValue);
		} else {
			azad = new DatumZeitDialog(null, oldValue);
		}
		if (Window.OK == azad.open()) {
			newValue = azad.getSelectedValue();
		}
		if (newValue != oldValue) {
			final String pid = String.valueOf(newValue);
			editedValue.setPid(pid);
			values[1] = pid;
		}
		return values;
	}

	/**
	 * Öffnet den Standarddialog zum Bearbeiten eines Zahlenwertes (auch
	 * Kommawerte!).
	 *
	 * @param editedValue
	 *            der zu bearbeitende Wert.
	 * @return Ein Feld aus zwei Strings. Erstes Objekt = alter Wert, Zweites
	 *         Objekt: Neuer Wert oder <code>null</code>, wenn keine Zuweisung
	 *         erfolgte (z.B. Dialog mit 'Abbrechen' beendet).
	 */
	private Object[] openIntegerDialog(final PlainDataValue editedValue) {
		final Object[] values = new Object[2];
		values[0] = editedValue.getPid();
		values[1] = null;
		final Long initialWert = Long.valueOf(editedValue.getPid());
		final IntegerAttributDialog iad = new IntegerAttributDialog(null,
				(IntegerAttributeType) editedValue.getAttributeType(), initialWert);
		if (Window.OK == iad.open()) {
			final String pid = String.valueOf(iad.getWert());
			editedValue.setPid(pid);
			values[1] = pid;
		}
		return values;
	}

	/**
	 * Öffnet den Standarddialog zum Bearbeiten eines Fließkommawertes.
	 *
	 * @param editedValue
	 *            der zu bearbeitende Wert.
	 * @return Ein Feld aus zwei Strings. Erstes Objekt = alter Wert, Zweites
	 *         Objekt: Neuer Wert oder <code>null</code>, wenn keine Zuweisung
	 *         erfolgte (z.B. Dialog mit 'Abbrechen' beendet).
	 */
	private Object[] openDoubleDialog(final PlainDataValue editedValue) {
		final Object[] values = new Object[2];
		values[0] = editedValue.getPid();
		values[1] = null;
		final Double initialWert = Double.valueOf(editedValue.getPid());
		final DoubleAttributDialog dad = new DoubleAttributDialog(null,
				(DoubleAttributeType) editedValue.getAttributeType(), initialWert);
		if (Window.OK == dad.open()) {
			final String pid = String.valueOf(dad.getWert());
			editedValue.setPid(pid);
			values[1] = pid;
		}
		return values;
	}

	/**
	 * Öffnet den standarddialog zur Auswahl von Systemobjekten.
	 *
	 * @param editedValue
	 *            der zu bearbeitende Wert.
	 * @return Ein Feld aus zwei Strings. Erstes Objekt = alter Wert, Zweites
	 *         Objekt: Neuer Wert oder <code>null</code>, wenn keine Zuweisung
	 *         erfolgte (z.B. Dialog mit 'Abbrechen' beendet).
	 */
	private Object[] openReferenceDialog(final PlainDataValue editedValue) {
		final Object[] values = new Object[2];
		values[0] = editedValue.getPid();
		values[1] = null;
		final ClientDavInterface davIf = RahmenwerkService.getService().getObjektFactory().getDav();
		final SystemObject oldValue = davIf.getDataModel().getObject(editedValue.getPid());
		final boolean neu = null == oldValue;
		final ReferenceAttributeType referenceAttrType = (ReferenceAttributeType) editedValue.getAttributeType();
		SystemObjectType referencedType = referenceAttrType.getReferencedObjectType();
		String systemObjektPid;
		final SystemObjectAuswahlType auswahlType;
		if (null == referencedType) {
			referencedType = davIf.getDataModel().getType("typ.typ");
			auswahlType = SystemObjectAuswahlType.TypesOrInstances;
		} else {
			final List<SystemObject> elements = referencedType.getElements();
			if ((null == elements) || elements.isEmpty()) {
				auswahlType = SystemObjectAuswahlType.TypesOrInstances;
			} else {
				int numInstances = 0;
				int numTypes = 0;
				for (final SystemObject o : elements) {
					if (o instanceof SystemObjectType) {
						++numTypes;
					} else {
						++numInstances;
					}
				}
				if (numTypes == 0) {
					auswahlType = SystemObjectAuswahlType.OnlyInstances;
				} else if (numInstances == 0) {
					auswahlType = SystemObjectAuswahlType.OnlyTypes;
				} else {
					auswahlType = SystemObjectAuswahlType.TypesOrInstances;
				}
			}
		}
		systemObjektPid = referencedType.getPid();
		final String descLongAnfang = "Wählen Sie das gewünschte Objekt ";
		final String descLongEnde = " aus.";
		final String aktionString = neu ? " zuweisen" : " modifizieren";
		final SystemObjectAuswahlWizardPage auswahlPage = new SystemObjectAuswahlWizardPage(
				referencedType.toString() + aktionString, MultipleSelection.Single, auswahlType,
				neu ? new ArrayList<SystemObject>() : Arrays.asList(oldValue), systemObjektPid);
		auswahlPage.setDescription(descLongAnfang + referencedType.toString() + descLongEnde);
		final ComposedWizardDialog wizardDialog = new ComposedWizardDialog(referencedType.toString() + aktionString,
				auswahlPage);
		if (Window.OK == wizardDialog.open()) {
			if (!auswahlPage.getAuswahl().isEmpty()) {
				final SystemObject newValue = auswahlPage.getAuswahl().get(0);
				if (null == newValue) {
					throw new IllegalStateException("Fehler: Referenz-Object konnte nicht identifiziert werden.");
				}
				editedValue.setPid(newValue.getPid());
				values[1] = newValue.getPid();
			}
		}
		return values;
	}

	/**
	 * Öffnet einen {@link MultiLineInputDialog} zum Bearbeiten eines Textattributs.
	 *
	 * @param editedValue
	 *            der zu bearbeitende Wert.
	 * @return Ein Feld aus zwei Strings. Erstes Objekt = alter Wert, Zweites
	 *         Objekt: Neuer Wert oder <code>null</code>, wenn keine Zuweisung
	 *         erfolgte (z.B. Dialog mit 'Abbrechen' beendet).
	 */
	private Object[] openTextDialog(final PlainDataValue editedValue) {
		final Object[] values = new Object[2];
		values[1] = null;
		final String oldTextValue = editedValue.getValue();
		values[0] = oldTextValue;
		final StringAttributeType strAttr = (StringAttributeType) editedValue.getAttributeType();
		final InputDialog id = new MultiLineInputDialog(null, "Text bearbeiten",
				"Bitte bearbeiten Sie den Text. Erlaubt sind maximal " + strAttr.getMaxLength() + " Zeichen.",
				oldTextValue, new StringLengthInputValidator(strAttr.getMaxLength()));
		if (Window.OK == id.open()) {
			values[1] = id.getValue();
			editedValue.setPid(id.getValue());
		}
		return values;
	}

	/**
	 * Toplevelroutine zum Öffnen eines Dialogs. Die entsprechend dem zu
	 * bearbeitenden Typ zu öffnenden Spezialeditoren werden anhand des
	 * DAF-Attributtyps bestimmt und geöffnet.
	 *
	 * @param item
	 *            das zu bearbeitende Objekt
	 * @param index
	 *            Nummer des Attributs innerhalb item, welches zu bearbeiten ist.
	 * @return der Pfad des Attributs innerhalb item, welches bearbeitet wurde.
	 *         Dieser wurde potentiell modifiziert (z.B. von .Leer auf .0).
	 */
	private String openDialog(final ParameterTableEditorItem item, final int index) {
		final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
		if ((index < 0) || (index >= plainDataValues.size())) {
			return null;
		}
		Object oldValue = null, newValue = null;
		final PlainDataValue editedValue = plainDataValues.get(index);
		final AttributeType attributeType = editedValue.getAttributeType();
		if (attributeType instanceof StringAttributeType) {
			final Object[] values = openTextDialog(editedValue);
			oldValue = values[0];
			newValue = values[1];
		} else if (attributeType instanceof IntegerAttributeType) {
			final Object[] values = openIntegerDialog(editedValue);
			oldValue = values[0];
			newValue = values[1];
		} else if (attributeType instanceof TimeAttributeType) {
			final Object[] values = openTimeDialog(editedValue);
			oldValue = values[0];
			newValue = values[1];
		} else if (attributeType instanceof ReferenceAttributeType) {
			final Object[] values = openReferenceDialog(editedValue);
			oldValue = values[0];
			newValue = values[1];
		} else if (attributeType instanceof DoubleAttributeType) {
			final Object[] values = openDoubleDialog(editedValue);
			oldValue = values[0];
			newValue = values[1];
		}
		if ((oldValue != null) && (newValue != null) && !oldValue.equals(newValue)) {
			item.setDirty(true);
			if (null != modifyListener) {
				modifyListener.valueModified(item, index, oldValue, newValue, attributeType);
			}
		}
		return editedValue.getPath();
	}

	/**
	 * Diese Methode behandelt den Doppelklick auf erster Ebene, indem überprüft
	 * wird, ob das selektierte Item bearbeitbar ist. Dies ist gegeben, wenn es
	 * gültig ist oder das erste ungültige Item in einem Feld.
	 *
	 * @param cursor
	 *            der Tabellencursor, über den man Zeile und Spalte bekommt.
	 */
	private void handleDoubleClick(final TableCursor cursor) {
		final int index = cursor.getColumn() - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
		if (index < 0) {
			// Man könnte hier einen Info-Dialog öffnen
			return;
		}
		final ParameterTableEditorItem item = (ParameterTableEditorItem) cursor.getRow().getData();
		final PlainDataValue editedValue = item.getPlainDataValues().get(index);
		boolean canEdit = editedValue.isValid();
		if (!canEdit) {
			final int firstUndefinedIndex = item.getFirstUndefinedIndexOfArray(index);
			final ArrayNameAndIndex ari = ParameterTableEditorItem.getArrayIndexFromPath(editedValue.getPath());
			if (ari.getIndex() == firstUndefinedIndex) {
				if (-2 == firstUndefinedIndex) {
					final AttributeSet parentAttributeSet = editedValue.getParentAttributeSet();
					if (null == parentAttributeSet) {
						canEdit = true;
					} else {
						canEdit = parentAttributeSet.getAttributes().size() == 1;
						if (!canEdit) {
							final ArrayInTableInfo aiti = new ArrayInTableInfo(item, index, true);
							if (aiti.getNumMembers() > 0) {
								canEdit = true;
							}
						}
					}
				} else {
					canEdit = true;
				}
			}
		}
		if (!canEdit) {
			return;
		}
		TableParamEditorDoubleClickListener.DEBUG
				.finer("Öffne Dialog für item " + item + ", Effektive Spalte " + index);
		final Table table = (Table) cursor.getData(TableParamEditorFormPage.CURSORDATA_KEY_TABLE);
		final TableColumn column = table.getColumn(cursor.getColumn());
		final String newPath = openDialog(item, index);
		if ((null != newPath) && !newPath.equals(column.getText())) {
			column.setText(newPath);
		}
	}

	@Override
	public void doubleClick(final DoubleClickEvent event) {
		final IStructuredSelection selection = (IStructuredSelection) event.getSelection();
		if ((selection != null) && !selection.isEmpty()) {
			final Object selected = selection.getFirstElement();
			if (event.getViewer() instanceof TableViewer) {
				final Table table = ((TableViewer) event.getViewer()).getTable();
				final TableCursor cursor = (TableCursor) table.getData(TableParamEditorFormPage.TABLEDATA_KEY_CURSOR);
				if (selected instanceof ParameterTableEditorItem) {
					handleDoubleClick(cursor);
				}
			}
		}
	}

	@Override
	public void mouseDoubleClick(final MouseEvent e) {
		// Kommt nie
	}

	@Override
	public void mouseDown(final MouseEvent e) {
		TableParamEditorDoubleClickListener.DEBUG.finest("mouse down: " + e);
		final TableCursor cursor = (TableCursor) e.widget;
		if (1 == e.button) {
			handleDoubleClick(cursor);
		} else if (3 == e.button) {
			final int effCol = cursor.getColumn() - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
			final ParameterTableEditorItem item = (ParameterTableEditorItem) cursor.getRow().getData();
			if ((effCol >= 0) && (effCol < item.getPlainDataValues().size())) {
				final PlainDataValue editedValue = item.getPlainDataValues().get(effCol);
				if (null != modifyListener) {
					modifyListener.showContextMenu(editedValue, item, effCol);
				}
			}
		}
	}

	@Override
	public void mouseUp(final MouseEvent e) {
		// uninteressant
	}

}
