/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.provider;

import java.util.List;

/**
 * Die Klasse verkapselt Informationen über ein Feld innerhalb eines
 * aufgefächerten Datensatzes.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public class ArrayInTableInfo {

	/**
	 * Der erste Index innerhalb des dem Konstruktor übergebenen Items, welches Teil
	 * des zu betrachtenden Feldes ist. Dieses wird über den dem Konstruktor
	 * übergebenen index und dem daraus bestimmten PlainDataValue ermittelt.
	 */
	private int firstElementIndex = -1;

	/**
	 * Der letzte Index innerhalb des dem Konstruktor übergebenen Items, welches
	 * Teil des zu betrachtenden Feldes ist. Dieses wird über den dem Konstruktor
	 * übergebenen index und dem daraus bestimmten PlainDataValue ermittelt.
	 */
	private int lastElementIndex = -1;

	/**
	 * Die Anzahl von Mitglieder im Feld. Dieses wird über den dem Konstruktor
	 * übergebenen index und dem daraus bestimmten PlainDataValue ermittelt.
	 */
	private int numMembers = -1;

	/**
	 * Konstruktor führt die Berechnung gleich durch.
	 *
	 * @param item
	 *            Das Item, auf das sich die Info bezieht.
	 * @param index
	 *            Der Index, welcher das zu betrschtende PlainDataValue innerhalb
	 *            item identifiziert.
	 * @param ignoreEmptyArray
	 *            Führt die Berechnung auch aus, wenn das Feld leer ist. Notwendig,
	 *            um leere, aber noch vorhandene Folgeelemente zu entdecken.
	 */
	public ArrayInTableInfo(final ParameterTableEditorItem item, final int index, final boolean ignoreEmptyArray) {
		final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
		final ArrayNameAndIndex ari = ParameterTableEditorItem
				.getArrayIndexFromPath(plainDataValues.get(index).getPath());
		if (ari.getIndex() >= 0 || ignoreEmptyArray) {
			int loop;
			final String relevantArrayName = ari.getArrayName();
			for (loop = 0; loop < plainDataValues.size(); ++loop) {
				if (plainDataValues.get(loop).getPath().startsWith(relevantArrayName)) {
					if (-1 == firstElementIndex) {
						firstElementIndex = loop;
					}
					lastElementIndex = loop;
				} else {
					if (-1 != firstElementIndex) {
						break;
					}
				}
			}
			// firstElementIndex kann hier nicht -1 sein!
			final PlainDataValue lastValue = plainDataValues.get(lastElementIndex);
			for (final ArrayNameAndIndex lastArn : ParameterTableEditorItem
					.getAllArrayComponentsFromPath(lastValue.getPath())) {
				if (lastArn.getArrayName().equals(relevantArrayName)) {
					numMembers = lastArn.getIndex() + 1;
				}
			}
			if (numMembers < 0 && ignoreEmptyArray) {
				final PlainDataValue relevantValue = plainDataValues.get(index);
				if (null == relevantValue.getParentAttributeSet()) {
					numMembers = 1;
				} else {
					numMembers = (lastElementIndex - firstElementIndex + 1)
							/ relevantValue.getParentAttributeSet().getAttributes().size();
				}
			}
		} else {
			firstElementIndex = ari.getIndex();
			lastElementIndex = ari.getIndex();
			numMembers = ari.getIndex();
		}
	}

	/**
	 * Liefert den ersten Index innerhalb des dem Konstruktor übergebenen Items,
	 * welches Teil des zu betrachtenden Feldes ist. Dieses wird über den dem
	 * Konstruktor übergebenen index und dem daraus bestimmten PlainDataValue
	 * ermittelt.
	 *
	 * @return der erste Index oder -1, falls gar kein Feld vorliegt oder -2, wenn
	 *         dieses leer ist.
	 */
	public int getFirstElementIndex() {
		return firstElementIndex;
	}

	/**
	 * Liefert den letzten Index innerhalb des dem Konstruktor übergebenen Items,
	 * welches Teil des zu betrachtenden Feldes ist. Dieses wird über den dem
	 * Konstruktor übergebenen index und dem daraus bestimmten PlainDataValue
	 * ermittelt.
	 *
	 * @return der letzte Index oder -1, falls gar kein Feld vorliegt oder -2, wenn
	 *         dieses leer ist.
	 */
	public int getLastElementIndex() {
		return lastElementIndex;
	}

	/**
	 * Liefert die Anzahl von Mitgliedern im Feld. Dieses wird über den dem
	 * Konstruktor übergebenen index und dem daraus bestimmten PlainDataValue
	 * ermittelt.
	 *
	 * @return die Anzahl oder -1, falls gar kein Feld vorliegt oder -2, wenn dieses
	 *         leer ist.
	 */
	public int getNumMembers() {
		return numMembers;
	}

}
