/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.provider;

import java.util.List;

import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;

import de.bsvrz.buv.plugin.param.ParamSharedImage;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;

/**
 * Label- und Colorprovider für Tabellen, die viele
 * {@link ParameterTableEditorItem}s anzeigen können. In Spalte 0 wird das
 * Objekt selbst angezeigt.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ParamTableEditorLabelProvider implements ITableLabelProvider, ITableColorProvider {

	/**
	 * das Inputobjekt des zugrunde liegenden Editors.
	 */
	private final ParameterEditorInput editorInput;

	/**
	 * Read-Only Flag. Wenn es gesetzt ist, so gibt es keine Einfärbung.
	 */
	private final boolean readOnly;

	/**
	 * Konstruktor übernimmt nur Werte.
	 *
	 * @param editorInput
	 *            das Inputobjekt des zugrunde liegenden Editors
	 * @param readOnly
	 *            Read-Only Flag. Wenn es gesetzt ist, so gibt es keine Einfärbung.
	 */
	public ParamTableEditorLabelProvider(final ParameterEditorInput editorInput, final boolean readOnly) {
		this.editorInput = editorInput;
		this.readOnly = readOnly;
	}

	@Override
	public Image getColumnImage(final Object element, final int columnIndex) {
		if (!(element instanceof ParameterTableEditorItem)) {
			return null;
		}
		Image image = null;
		final int effIdx = columnIndex - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
		final ParameterTableEditorItem item = (ParameterTableEditorItem) element;
		final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
		if ((effIdx >= 0) && (effIdx < plainDataValues.size())) {
			final PlainDataValue plainValue = plainDataValues.get(effIdx);
			if (!plainValue.isDefined() && !plainValue.hasUserValue()) {
				image = ParamSharedImage.ERROR_NULL.getImage();
			}
		}
		return image;
	}

	@Override
	public String getColumnText(final Object element, final int columnIndex) {
		String text = "unbekannt";
		if (!(element instanceof ParameterTableEditorItem)) {
			return text;
		}
		final ParameterTableEditorItem item = (ParameterTableEditorItem) element;
		if (0 == columnIndex) {
			final String objektName = item.getParameter().getObjekt().toString();
			if (item.isDirty()) {
				text = "*" + objektName;
			} else {
				text = objektName;
			}
		} else {
			final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
			final int effIdx = columnIndex - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
			if (effIdx < plainDataValues.size()) {
				text = plainDataValues.get(effIdx).getValue();
			}
		}
		return text;
	}

	@Override
	public void addListener(final ILabelProviderListener listener) {
		// Keine Listener
	}

	@Override
	public void dispose() {
		// Keine Ressourcen freizugeben
	}

	@Override
	public boolean isLabelProperty(final Object element, final String property) {
		// Keine Labelproperties
		return false;
	}

	@Override
	public void removeListener(final ILabelProviderListener listener) {
		// Keine Listener
	}

	@Override
	public Color getBackground(final Object element, final int columnIndex) {
		if (!(element instanceof ParameterTableEditorItem)) {
			return null;
		}
		Color background = null;
		final ParameterTableEditorItem item = (ParameterTableEditorItem) element;
		if (!readOnly) {
			final List<PlainDataValue> plainDataValues = item.getPlainDataValues();
			final int effIdx = columnIndex - ParameterTableEditorItem.NUM_COLUMNS_BEFORE;
			if ((0 <= effIdx) && (effIdx < plainDataValues.size())) {
				final PlainDataValue plainDataValue = plainDataValues.get(effIdx);
				background = editorInput.getVorgabeColor(item.getParameter().getInfo(), plainDataValue.getPath(),
						plainDataValue.getValueText());
			}
		}
		return background;
	}

	@Override
	public Color getForeground(final Object element, final int columnIndex) {
		// Keien Vordergrundfärbung
		return null;
	}

}
