/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.provider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IAdaptable;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.table.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.AttributeSet;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Verkapselt ein Feld von {@link PlainDataValue} für jedes atomare Attribut
 * (=Spalte) in einer Tabelle.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ParameterTableEditorItem implements IAdaptable {

	/**
	 * Anzahl Spalten vor der ersten Attributspalte.
	 */
	public static final int NUM_COLUMNS_BEFORE = 1;

	/**
	 * Trennzeichen zwischen den einzelnen Komponenten zum Pfad eines Attributs.
	 */
	public static final String PATH_SEPERATOR = ".";

	/**
	 * Daf-Logger für Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Flag zum Abschalten der Logausgaben. Aus Performancegründen eingeführt, im
	 * Release auf false setzen.
	 */
	private static final boolean DO_DEBUG = false;

	/**
	 * Die Liste der Einzelattribute. Beim Instanziieren mit einem DAF Data-Objekt
	 * wird die Liste initial gefüllt, und bei
	 * {@link #catchUpBiDirectional(ParameterTableEditorItem)} eventuell mit
	 * Platzhalter-Objekten erweitert.
	 */
	private final List<PlainDataValue> plainDataValues = new ArrayList<>();

	/**
	 * Arbeitskopie des gesamten Datensatzes des verkapselten Parameters.
	 */
	private final Data workingCopy;

	/**
	 * Das Daten-Backend-Objekt, welches wir repräsentieren.
	 */
	private final Parameter parameter;

	/**
	 * Mit diesem Flag kann die Anzeige der drei Spalten Urlasser, Ursache und
	 * Veranlasser ein- bzw. ausgeschaltet werden.
	 */
	private final boolean showUrlasserInfos;

	/**
	 * Dirty-Flag für Editor-Zwecke. Das Flag wird nirgends automatisiert gesetzt
	 * oder gelöscht aufgrund irgendwelcher Bedingungen, sondern nur explizit von
	 * außen durch unsere Benutzer.
	 */
	private boolean dirty;

	/**
	 * Konstruktor weist finale Felder zu bzw. erzeugt sie, nd führt Prüfungen durch
	 * (mit {@link Assert}).
	 *
	 * @param parameter
	 *            der verkapselte Parameter
	 * @param showUrlasserInfos
	 *            true, wenn die Urlasser-ATL auch mit geführt werden soll, sonst
	 *            false
	 * @param runImmediately
	 *            Sofort die Erzeugung der internen Liste mit PlainDataValues
	 *            starten
	 */
	public ParameterTableEditorItem(final Parameter parameter, final boolean showUrlasserInfos,
			final boolean runImmediately) {
		this.parameter = parameter;
		Assert.isNotNull(parameter);
		Assert.isTrue(RahmenwerkService.getService().getObjektFactory().isVerbunden(),
				Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		if (null == parameter.getData()) {
			workingCopy = RahmenwerkService.getService().getObjektFactory().getDav().createData(parameter.getAtg());
		} else {
			workingCopy = parameter.getData().createModifiableCopy();
		}
		this.showUrlasserInfos = showUrlasserInfos;
		if (runImmediately) {
			run();
		}
	}

	/**
	 * Rekursive Methode zum Füllen der {@link #plainDataValues} aus einem DAF
	 * Data-Objekt. Abbruchkriterium ist data.isPlain(), bei isList oder isArray
	 * geht es in die Rekursion.
	 *
	 * Das SuppressWarnings ist Ok, es ist der auf älteren Java-Versionen
	 * entwickelten DAF-API geschuldet.
	 *
	 * @param parentData
	 *            Das übergeordnete Attributset, entweder die Attributgruppe oder
	 *            eine Attributliste auf irgendeiner Ebene.
	 * @param data
	 *            das DAF-Data-Objekt, aus dem heraus ein neues
	 *            {@link PlainDataValue} erzeugt werden soll.
	 * @param path
	 *            Bisheriger, aktueller Pfad, an den das neue Objekt angehangen
	 *            wird.
	 * @param index
	 *            der Index, an dem das nächste Objekt in {@link #plainDataValues}
	 *            eingehangen werden soll.
	 * @return der nächste Index, an dem ein (eventuelles) nächstes Objekt
	 *         eingehangen werden soll.
	 *
	 */
	private int bestimmePfadUndIndex(final AttributeSet parentData, final Data data, final String path,
			final int index) {
		int idx = index;
		String nextBasePath;
		if (path.length() > 0) {
			nextBasePath = path + ParameterTableEditorItem.PATH_SEPERATOR + data.getName();
		} else {
			nextBasePath = data.getName();
		}
		if (data.isList()) {
			final Iterator<Data> listIter = data.iterator();
			while (listIter.hasNext()) {
				idx = bestimmePfadUndIndex((AttributeSet) data.getAttributeType(), listIter.next(), nextBasePath, idx);
			}
		} else if (data.isArray()) {
			AttributeSet parentAttributeSet = null;
			final AttributeType a = data.getAttributeType();
			AttributeType relevantAttributeType = a;
			if (a instanceof AttributeSet) {
				parentAttributeSet = (AttributeSet) a;
				if (ParameterTableEditorItem.DO_DEBUG) {
					ParameterTableEditorItem.DEBUG.finer("Feld mit nicht-atomaren" + " Subattributen gefunden",
							a.getName());
				}
				relevantAttributeType = parentAttributeSet.getAttributes().get(0).getAttributeType();
			}
			if (data.asArray().getLength() <= 0) {
				final PlainDataValue arrayKopf = new PlainDataValue(null,
						nextBasePath + ParameterTableEditorItem.PATH_SEPERATOR + Zeichenketten.EMPTY_INDICATOR);
				plainDataValues.add(idx, arrayKopf);
				++idx;
				arrayKopf.setAttributeType(relevantAttributeType);
				arrayKopf.setParentAttributeSet(parentAttributeSet);
				arrayKopf.setAttributeName(data.getName());
				if (null != parentAttributeSet) {
					for (final Attribute attribut : parentAttributeSet.getAttributes()) {
						arrayKopf.addFeldAttribut(attribut);
					}
				}
			} else {
				int itemLoop;
				for (itemLoop = 0; itemLoop < data.asArray().getLength(); ++itemLoop) {
					idx = bestimmePfadUndIndex(parentAttributeSet, data.asArray().getItem(itemLoop), nextBasePath, idx);
				}
			}
		} else if (data.isPlain()) {
			final PlainDataValue element = new PlainDataValue(data, nextBasePath);
			element.setParentAttributeSet(parentData);
			plainDataValues.add(idx, element);
			++idx;
		}
		return idx;
	}

	/**
	 * (Neu-)Initialisierung des Objektes aus der privaten workingCopy.
	 *
	 * Das SuppressWarnings ist Ok, es ist der auf älteren Java-Versionen
	 * entwickelten DAF-API geschuldet.
	 */
	public void run() {
		plainDataValues.clear();
		final String path = "";
		final Iterator<Data> iterator = workingCopy.iterator();
		int index = 0;
		while (iterator.hasNext()) {
			final Data next = iterator.next();
			if (showUrlasserInfos || !"atl.urlasser".equals(next.getAttributeType().getPid())) {
				index = bestimmePfadUndIndex(parameter.getAtg(), next, path, index);
			}
		}
	}

	/**
	 * Bestimmung des Index eines (Einzel-)Attributs, repräsentiert durch seinen
	 * Pfad, innerhalb eines Feldes.
	 *
	 * @param path
	 *            der Pfad. Es kann sowohl ein echt atomares Einzelattribut sein
	 *            oder auch ein Pfad zu einer komplexen Substruktur.
	 * @return Ein Objekt, welches gleich mehrere Informationen bereitstellt: Den
	 *         Restpfad (d.i. der Attributname bei atomaren Einzelattributen), den
	 *         gesuchten Index und den vebleibenden Pfad. Der gesuchte Index ist -2
	 *         für leere Felder und -1, wenn der Pfad nicht Teil eines Feldes ist.
	 *         Der verbleibende Pfad kann wiederum Aufrufparameter für diese Methode
	 *         sein, um bei verschachtelten Feldern bis zur obersten Ebene gelangen
	 *         zu können.
	 */
	public static ArrayNameAndIndex getArrayIndexFromPath(final String path) {
		final ArrayNameAndIndex result = new ArrayNameAndIndex();
		result.setArrayName(path);
		result.setRestPfad("");
		final String[] components = path.split("\\.");
		int backLoop;
		int arrayNameLoop;
		int restNameLoop;
		if (Zeichenketten.EMPTY_INDICATOR.equals(components[components.length - 1])) {
			result.setIndex(-2);
			result.setArrayName("");
			for (arrayNameLoop = 0; arrayNameLoop < (components.length - 1); ++arrayNameLoop) {
				if (0 != arrayNameLoop) {
					result.setArrayName(result.getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR);
				}
				result.setArrayName(result.getArrayName() + components[arrayNameLoop]);
			}
			return result;
		}
		for (backLoop = components.length - 1; backLoop >= 0; --backLoop) {
			int charLoop;
			boolean isNumber = true;
			if (!Zeichenketten.EMPTY_INDICATOR.equals(components[backLoop])) {
				for (charLoop = 0; charLoop < components[backLoop].length(); ++charLoop) {
					if (!Character.isDigit(components[backLoop].charAt(charLoop))) {
						isNumber = false;
						break;
					}
				}
			}
			if (isNumber) {
				if (!Zeichenketten.EMPTY_INDICATOR.equals(components[backLoop])) {
					result.setIndex(Integer.parseInt(components[backLoop]));
				} else {
					result.setIndex(-2);
				}
				result.setArrayName("");
				for (arrayNameLoop = 0; arrayNameLoop < backLoop; ++arrayNameLoop) {
					if (0 != arrayNameLoop) {
						result.setArrayName(result.getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR);
					}
					result.setArrayName(result.getArrayName() + components[arrayNameLoop]);
				}
				for (restNameLoop = backLoop + 1; restNameLoop < components.length; ++restNameLoop) {
					if (restNameLoop != (backLoop + 1)) {
						result.setRestPfad(result.getRestPfad() + ParameterTableEditorItem.PATH_SEPERATOR);
					}
					result.setRestPfad(result.getRestPfad() + components[restNameLoop]);
				}
				break;
			}
		}
		return result;
	}

	/**
	 * Durchsucht einen Pfad von hinten nach vorn nach Feldern und speichert die
	 * Komponenten in einer Liste.
	 *
	 * @param path
	 *            der zu durchsuchende Pfad.
	 * @return die Liste. Enthält immer mindestens ein Element.
	 */
	public static List<ArrayNameAndIndex> getAllArrayComponentsFromPath(final String path) {
		final List<ArrayNameAndIndex> indices = new ArrayList<>();
		int index;
		String curPath = path;
		do {
			final ArrayNameAndIndex ari = ParameterTableEditorItem.getArrayIndexFromPath(curPath);
			index = ari.getIndex();
			if (-1 != index) {
				indices.add(ari);
				curPath = ari.getArrayName();
			}
		} while (index != -1);
		return indices;
	}

	/**
	 * Erzeugt eine modifizierbare Kopie dieses Items. Die Kopie enthält auch die
	 * vom User modifizierten / hinzugefügten Werte sowie den ursprünglichen
	 * dirty-Status.
	 *
	 * @return die Kopie
	 */
	public ParameterTableEditorItem createModifiableCopy() {
		final Parameter p = new Parameter(getParameter().getInfo(),
				new DataWithTime(createData(), System.currentTimeMillis()));
		final ParameterTableEditorItem copy = new ParameterTableEditorItem(p, showUrlasserInfos, false);
		copy.setDirty(isDirty());
		return copy;
	}

	/**
	 * Rekursive Methode zum Auffüllen von Feldern.
	 *
	 * @param attributeSet
	 *            das zu bearbeitende Attributset. Beim initialen Einsprung die ATG
	 *            selbst, dann jede beliebige Attributliste innerhalb dieser.
	 * @param d1
	 *            Vergleichsobjekt 1. Sein Attributtyp muss attributeSet
	 *            entsprechen.
	 * @param d2
	 *            Vergleichsobjekt 2. Sein Attributtyp muss attributeSet
	 *            entsprechen.
	 */
	private void doCatchUpByDataRekursiv(final AttributeSet attributeSet, final Data d1, final Data d2) {
		for (final Attribute a : attributeSet.getAttributes()) {
			if (a.isArray()) {
				final Array shorterArray;
				final Array array1 = d1.getArray(a.getName());
				final Array array2 = d2.getArray(a.getName());
				final int len1 = array1.getLength();
				final int len2 = array2.getLength();
				final int sollLength = Math.max(len1, len2);
				if (len1 != len2) {
					if (len1 > len2) {
						shorterArray = array2;
					} else {
						shorterArray = array1;
					}
					shorterArray.setLength(sollLength);
				}
				if (ParameterTableEditorItem.DO_DEBUG) {
					ParameterTableEditorItem.DEBUG
							.finest("Array-Attribut: " + a + " Längen: " + len1 + " <--> " + len2);
				}
				if (a.getAttributeType() instanceof AttributeSet) {
					int loop;
					for (loop = 0; loop < sollLength; ++loop) {
						final AttributeSet attributeSetImArray = (AttributeSet) a.getAttributeType();
						final Data d1ImArray = array1.getItem(loop);
						final Data d2ImArray = array2.getItem(loop);
						doCatchUpByDataRekursiv(attributeSetImArray, d1ImArray, d2ImArray);
					}
				}
			} else {
				if (ParameterTableEditorItem.DO_DEBUG) {
					ParameterTableEditorItem.DEBUG.finest("Attribut: " + a + " ist kein Feld");
				}
			}
		}
	}

	/**
	 * Diese Methode vergleicht die verkapselten Data-Objekte. Bei Unterschieden
	 * wird das fehlende Subattribut im jeweils anderen Objekt eingefügt.
	 *
	 * @param otherSpaltenBestimmer
	 *            das andere Objekt
	 */
	public void catchUpByData(final ParameterTableEditorItem otherSpaltenBestimmer) {
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.finest("Vorher Meine Working Copy: " + workingCopy);
			ParameterTableEditorItem.DEBUG
					.finest("Vorher Andere Working Copy: " + otherSpaltenBestimmer.getWorkingCopy());
		}
		doCatchUpByDataRekursiv(parameter.getAtg(), workingCopy, otherSpaltenBestimmer.getWorkingCopy());
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.finest("Nachher Meine Working Copy: " + workingCopy);
			ParameterTableEditorItem.DEBUG
					.finest("Nachher Andere Working Copy: " + otherSpaltenBestimmer.getWorkingCopy());
		}
	}

	/**
	 * Diese Methode vergleicht die verkapselten Pfade. Bei Unterschieden wird das
	 * fehlende Objekt im jeweils anderen Objekt eingefügt.
	 *
	 * @deprecated Der hier verwendete Algorithmus ist fehlerbehaftet und durch den
	 *             neuen Algorithmus
	 *             {@link #catchUpByData(ParameterTableEditorItem)} ersetzt worden.
	 *             Ich will den Code aber noch nicht löschen, im Fall ich muss doch
	 *             noch darauf zurückgreifen.
	 *
	 * @param otherSpaltenBestimmer
	 *            das andere Objekt
	 */
	@Deprecated
	public void catchUpBiDirectional(final ParameterTableEditorItem otherSpaltenBestimmer) {
		boolean go = true;
		final List<PlainDataValue> otherPlainDataValues = otherSpaltenBestimmer.getPlainDataValues();
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.config("Catch-up entry: we/they ", new Object[] { plainDataValues.size(),
					otherPlainDataValues.size(), plainDataValues, otherPlainDataValues });
		}
		int pfadLoop = 0;
		int lastEqualPath = 0;
		while (go) {
			go = false;
			final int maxCount = Math.min(plainDataValues.size(), otherPlainDataValues.size());
			for (pfadLoop = lastEqualPath; pfadLoop < maxCount; ++pfadLoop) {
				final String path = plainDataValues.get(pfadLoop).getPath();
				final String otherPath = otherPlainDataValues.get(pfadLoop).getPath();
				if (ParameterTableEditorItem.DO_DEBUG) {
					ParameterTableEditorItem.DEBUG
							.finest("pfadLoop=" + pfadLoop + "\nwe:  " + path + "\nthey:" + otherPath);
				}
				if (path.equals(otherPath)) {
					lastEqualPath = pfadLoop;
					continue;
				}
				final List<ArrayNameAndIndex> ourArrayComponents = ParameterTableEditorItem
						.getAllArrayComponentsFromPath(path);
				final List<ArrayNameAndIndex> theirArrayComponents = ParameterTableEditorItem
						.getAllArrayComponentsFromPath(otherPath);
				final ArrayNameAndIndex ourArrayNameAndIndex = ourArrayComponents.get(0);
				final int index = ourArrayNameAndIndex.getIndex();
				final ArrayNameAndIndex theirArrayNameAndIndex = theirArrayComponents.get(0);
				final int otherIndex = theirArrayNameAndIndex.getIndex();
				if (ParameterTableEditorItem.DO_DEBUG) {
					ParameterTableEditorItem.DEBUG.finer("DISKREPANZ: Our Array=" + ourArrayNameAndIndex.getArrayName()
							+ ", index=" + index + "\nTheir Array=" + theirArrayNameAndIndex.getArrayName()
							+ ", otherIndex=" + otherIndex);
				}
				if (ourArrayNameAndIndex.getArrayName().equals(theirArrayNameAndIndex.getArrayName())) {
					if (ParameterTableEditorItem.DO_DEBUG) {
						ParameterTableEditorItem.DEBUG.finer("Gleiche Felder, aber ungleiche Indices");
					}
					if ((-1 == index) && (-1 == otherIndex)) {
						ParameterTableEditorItem.DEBUG.finer("###########################");
						continue;
					} else if ((-2 == index) && (otherIndex >= 0)) {
						// Wir sind ein leeres Array, die anderen nicht
						fillEmptyArrayFirstElement(pfadLoop, path, plainDataValues, otherPlainDataValues);
						go = true;
						break;
					} else if ((0 <= index) && (-2 == otherIndex)) {
						// Die anderen sind ein leeres Array, wir nicht
						fillEmptyArrayFirstElement(pfadLoop, otherPath, otherPlainDataValues, plainDataValues);
						go = true;
						break;
					} else if ((index > otherIndex) && (otherIndex != -1)) {
						// Wir haben mehr Elemente als other
						final PlainDataValue filler = new PlainDataValue(null, path);
						otherPlainDataValues.add(pfadLoop, filler);
						if (ParameterTableEditorItem.DO_DEBUG) {
							ParameterTableEditorItem.DEBUG.finer("Hinzugefügt bei denen: " + filler);
						}
						go = true;
						break;
					} else if ((otherIndex > index) && (index != -1)) {
						// Die anderen haben mehr Elemente als wir
						final PlainDataValue filler = new PlainDataValue(null, otherPath);
						plainDataValues.add(pfadLoop, filler);
						if (ParameterTableEditorItem.DO_DEBUG) {
							ParameterTableEditorItem.DEBUG.finer("Hinzugefügt bei uns: " + filler);
						}
						go = true;
						break;
					} else {
						ParameterTableEditorItem.DEBUG.finer(">>>>>>>>>>>>>>>>>>>>>>>>>");
					}
				} else {
					if (ParameterTableEditorItem.DO_DEBUG) {
						ParameterTableEditorItem.DEBUG.finer("Ungleiche Felder");
					}
					handleUngleicheFelder(pfadLoop, ourArrayNameAndIndex, theirArrayNameAndIndex, otherPlainDataValues);
					go = true;
					break;
				}

			}
			if (ParameterTableEditorItem.DO_DEBUG) {
				ParameterTableEditorItem.DEBUG.config("Intermediate Catch-up status: we/they ", new Object[] {
						plainDataValues.size(), otherPlainDataValues.size(), plainDataValues, otherPlainDataValues });
			}
			if (!go && (pfadLoop >= maxCount) && (plainDataValues.size() != otherPlainDataValues.size())) {
				go = handleFinalPadding(otherPlainDataValues);
			}
			if (go) {
				ParameterTableEditorItem.DEBUG
						.finest("Zwischenstand:\nWir:" + this + "\n\nDie: " + otherSpaltenBestimmer);
			}
		} // while (go)
		if (plainDataValues.size() != otherPlainDataValues.size()) {
			ParameterTableEditorItem.DEBUG.warning("Final Catch-up failure",
					new Object[] { plainDataValues, otherPlainDataValues });
		}
	}

	/**
	 * Spezialbehandlung, wenn das lezte Attribut einer Attributgruppe ein variabel
	 * langes Feld ist. Damit werden soviele Platzhalterobjekte eingefügt, wie
	 * notwendig.
	 *
	 * @param otherPlainDataValues
	 *            Liste mit den anderen Werten, gegen die unsere eigenen
	 *            {@link #plainDataValues} abgeglichen werden sollen.
	 * @return true - Felder wurdn erfolgreich abgeglichen. false - Abgleich nicht
	 *         möglich, weil ungleiche Felder oder Abgleich fehlgeschlagen.
	 */
	private boolean handleFinalPadding(final List<PlainDataValue> otherPlainDataValues) {
		final List<PlainDataValue> smallerList, biggerList, paddedValues;
		if (plainDataValues.size() < otherPlainDataValues.size()) {
			smallerList = plainDataValues;
			biggerList = otherPlainDataValues;
		} else {
			smallerList = otherPlainDataValues;
			biggerList = plainDataValues;
		}
		paddedValues = new ArrayList<>();
		final PlainDataValue lastCommonValue = smallerList.get(smallerList.size() - 1);
		final ArrayNameAndIndex lastCommonAri = ParameterTableEditorItem
				.getArrayIndexFromPath(lastCommonValue.getPath());
		final PlainDataValue firstRestValue = biggerList.get(smallerList.size());
		final ArrayNameAndIndex firstRestAri = ParameterTableEditorItem.getArrayIndexFromPath(firstRestValue.getPath());
		if (!lastCommonAri.getArrayName().equals(firstRestAri.getArrayName())) {
			if (ParameterTableEditorItem.DO_DEBUG) {
				ParameterTableEditorItem.DEBUG
						.fine("Versuch, final Padding zu erzielen," + " aber Felder sind nicht kompatibel");
			}
			return false;
		}
		if (null == lastCommonValue.getParentAttributeSet()) {
			int restLoop;
			for (restLoop = smallerList.size(); restLoop < biggerList.size(); ++restLoop) {
				final PlainDataValue restValue = biggerList.get(restLoop);
				final ArrayNameAndIndex restAri = ParameterTableEditorItem.getArrayIndexFromPath(restValue.getPath());
				if (lastCommonAri.getArrayName().equals(restAri.getArrayName())) {
					final StringBuilder b = new StringBuilder(restAri.getArrayName());
					b.append(ParameterTableEditorItem.PATH_SEPERATOR);
					b.append((lastCommonAri.getIndex() + 1 + restLoop) - smallerList.size());
					final PlainDataValue padValue = new PlainDataValue(null, b.toString());
					padValue.setAttributeType(lastCommonValue.getAttributeType());
					padValue.setParentAttributeSet(lastCommonValue.getParentAttributeSet());
					paddedValues.add(padValue);
				}
			}
		} else {
			final StringBuilder b = new StringBuilder(lastCommonAri.getArrayName());
			b.append(ParameterTableEditorItem.PATH_SEPERATOR);
			b.append(lastCommonAri.getIndex() + 1);
			paddedValues.addAll(ParameterTableEditorItem.createPlainDataValues(b.toString(),
					lastCommonValue.getParentAttributeSet()));
		}
		smallerList.addAll(paddedValues);
		return plainDataValues.size() != otherPlainDataValues.size();
	}

	/**
	 * Standardbehandlung für zueinandergehörige, aber ungleich lange Felder.
	 *
	 * @param pfadLoop
	 *            Aktueller Schleifenindex in unserem Feld.
	 * @param ourArrayNameAndIndex
	 *            Aufgelöster Pfad in unserem Feld.
	 * @param theirArrayNameAndIndex
	 *            Aufgelöster Pfad im anderen Feld.
	 * @param otherPlainDataValues
	 *            Liste mit den anderen Werten, gegen die unsere eigenen
	 *            {@link #plainDataValues} abgeglichen werden sollen.
	 */
	private void handleUngleicheFelder(final int pfadLoop, final ArrayNameAndIndex ourArrayNameAndIndex,
			final ArrayNameAndIndex theirArrayNameAndIndex, final List<PlainDataValue> otherPlainDataValues) {
		List<PlainDataValue> plainDataValuesMissingElements;
		List<PlainDataValue> plainDataValuesProvidingElements;
		ArrayNameAndIndex targetDetails;
		if ((ourArrayNameAndIndex.getIndex() > theirArrayNameAndIndex.getIndex())
				&& (theirArrayNameAndIndex.getIndex() >= 0)) {
			// bei denen fehlen Elemente
			plainDataValuesMissingElements = otherPlainDataValues;
			plainDataValuesProvidingElements = plainDataValues;
			targetDetails = ourArrayNameAndIndex;
		} else if ((ourArrayNameAndIndex.getIndex() >= 0)
				&& (ourArrayNameAndIndex.getIndex() < theirArrayNameAndIndex.getIndex())) {
			// bei uns fehlen Elemente
			plainDataValuesMissingElements = plainDataValues;
			plainDataValuesProvidingElements = otherPlainDataValues;
			targetDetails = theirArrayNameAndIndex;
		} else {
			// Bei uns ab pfadLoop + 1 suchen, ob anderes Array schon folgt -
			// dann sind wir der Provider
			int beiUnsLoop = 0;
			// int beiDenenLoop = 0;
			boolean foundByUs = false;
			for (beiUnsLoop = pfadLoop + 1; beiUnsLoop < plainDataValues.size(); ++beiUnsLoop) {
				if (plainDataValues.get(beiUnsLoop).getPath().startsWith(theirArrayNameAndIndex.getArrayName())) {
					foundByUs = true;
					break;
				}
			}
			if (foundByUs) {
				plainDataValuesMissingElements = otherPlainDataValues;
				plainDataValuesProvidingElements = plainDataValues;
				targetDetails = ourArrayNameAndIndex;
			} else {
				plainDataValuesMissingElements = plainDataValues;
				plainDataValuesProvidingElements = otherPlainDataValues;
				targetDetails = theirArrayNameAndIndex;
			}

			// String[] compareStrings = new String[] {
			// ourArrayNameAndIndex.arrayName,
			// theirArrayNameAndIndex.arrayName };
			// int compareStringLoop;
			// for (compareStringLoop = 0; compareStringLoop <
			// compareStrings.length
			// && beiUnsLoop == beiDenenLoop; ++compareStringLoop) {
			// for (beiUnsLoop = 0; beiUnsLoop < plainDataValues.size();
			// ++beiUnsLoop) {
			// if (plainDataValues.get(beiUnsLoop).getPath().startsWith(
			// compareStrings[compareStringLoop])) {
			// break;
			// }
			// }
			// for (beiDenenLoop = 0; beiDenenLoop < otherPlainDataValues
			// .size(); ++beiDenenLoop) {
			// if (otherPlainDataValues.get(beiDenenLoop).getPath()
			// .startsWith(compareStrings[compareStringLoop])) {
			// break;
			// }
			// }
			// }
			// debug.fine("beiUnsLoop=" + beiUnsLoop + ", beiDenenLoop="
			// + beiDenenLoop);
			// if (beiUnsLoop > beiDenenLoop) {
			// plainDataValuesMissingElements = plainDataValues;
			// plainDataValuesProvidingElements = otherPlainDataValues;
			// targetDetails = theirArrayNameAndIndex;
			// } else if (beiUnsLoop < beiDenenLoop) {
			// plainDataValuesMissingElements = otherPlainDataValues;
			// plainDataValuesProvidingElements = plainDataValues;
			// targetDetails = ourArrayNameAndIndex;
			// } else {
			// int beiUnsLen = ourArrayNameAndIndex.arrayName.length();
			// int beiDenenLen = theirArrayNameAndIndex.arrayName.length();
			// if (beiUnsLen > beiDenenLen) {
			// plainDataValuesMissingElements = plainDataValues;
			// plainDataValuesProvidingElements = otherPlainDataValues;
			// targetDetails = theirArrayNameAndIndex;
			// } else if (beiUnsLen < beiDenenLen) {
			// plainDataValuesMissingElements = otherPlainDataValues;
			// plainDataValuesProvidingElements = plainDataValues;
			// targetDetails = ourArrayNameAndIndex;
			// } else {
			// throw new IllegalStateException(
			// "Positionsbestimmung fehlgeschlagen!!!");
			// }
			// }
		}
		fillMissingArrayMember(pfadLoop, plainDataValuesMissingElements, plainDataValuesProvidingElements,
				targetDetails);

	}

	/**
	 * Diese Methode führt das Padding mit Plazhalter-Objekten wirklich durch.
	 *
	 * @param pfadLoop
	 *            Index für den Start der Einfügeoperation.
	 * @param plainDataValuesMissingElements
	 *            Feld, bei denen Elemente fehlen
	 * @param plainDataValuesProvidingElements
	 *            Feld, welches diese Elemente hat.
	 * @param targetDetails
	 *            Informationen zum Finden des EInsprungpunktes. Die sind wichtig,
	 *            wenn die einzufügenden Elemente keine atomaren Einzelattribute,
	 *            sondern eine Attributliste sind, die ja mehrere Spalten benötigen.
	 */
	private void fillMissingArrayMember(final int pfadLoop, final List<PlainDataValue> plainDataValuesMissingElements,
			final List<PlainDataValue> plainDataValuesProvidingElements, final ArrayNameAndIndex targetDetails) {
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.finer("fillMissingArrayMember",
					new Object[] { "pfadLoop / missing / provoding / target", pfadLoop, plainDataValuesMissingElements,
							plainDataValuesProvidingElements, targetDetails });
		}
		int providingLoop;
		final List<PlainDataValue> newValues = new ArrayList<>();
		for (providingLoop = pfadLoop; providingLoop < plainDataValuesProvidingElements.size(); ++providingLoop) {
			final PlainDataValue providingValue = plainDataValuesProvidingElements.get(providingLoop);
			if (providingValue.getPath().startsWith(targetDetails.getArrayName())) {
				final PlainDataValue newValue = new PlainDataValue(null, providingValue.getPath());
				newValue.setAttributeType(providingValue.getAttributeType());
				newValue.setParentAttributeSet(providingValue.getParentAttributeSet());
				newValues.add(newValue);
			} else {
				break;
			}
		}
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.finer("fillMissingArrayMember: Füge " + newValues.size() + " Elemente ein",
					newValues);
		}
		int missingLoop;
		final List<Integer> indicesToBeRemoved = new ArrayList<>();
		final List<PlainDataValue> leerItemsToBeRemoved = new ArrayList<>();
		for (missingLoop = pfadLoop; missingLoop < plainDataValuesMissingElements.size(); ++missingLoop) {
			final PlainDataValue testValue = plainDataValuesMissingElements.get(missingLoop);
			if (testValue.getPath().endsWith(Zeichenketten.EMPTY_INDICATOR)) {
				final String pathOhneEmpty = testValue.getPath().substring(0,
						testValue.getPath().length() - Zeichenketten.EMPTY_INDICATOR.length() - 1);
				int newValuesLoop;
				for (newValuesLoop = 0; newValuesLoop < newValues.size(); ++newValuesLoop) {
					final PlainDataValue piePiece = newValues.get(newValuesLoop);
					if (piePiece.getPath().startsWith(pathOhneEmpty) && !indicesToBeRemoved.contains(missingLoop)) {
						indicesToBeRemoved.add(missingLoop);
						leerItemsToBeRemoved.add(testValue);
					}
				}
			}
		}
		if (ParameterTableEditorItem.DO_DEBUG) {
			ParameterTableEditorItem.DEBUG.finer(
					"fillMissingArrayMember: Entferne " + leerItemsToBeRemoved.size() + " Elemente",
					new Object[] { leerItemsToBeRemoved, indicesToBeRemoved });
		}
		plainDataValuesMissingElements.removeAll(leerItemsToBeRemoved);
		plainDataValuesMissingElements.addAll(pfadLoop, newValues);
	}

	/**
	 * Spezialbehandlung, wenn in ein bisher leeres Feld (welches durch eine
	 * einzelne Spalte repräsentiert wird) ein erstes Element eingefügt wird. Dabei
	 * wird beachtet, ob es sich um ein atomares Einzelattribut bzw. Attributset mit
	 * nur 1 Element handelt (in dem Fall wird die Spalte nur umkonfiguriert) oder
	 * ob eine Attributset aus n Elementen eingefügt werden soll - in dem Fall
	 * müssen n-1 Platzhalterobjekte eingefügt werden.
	 *
	 * @param pfadLoop
	 *            Suchindex in unseren {@link #plainDataValues}
	 * @param path
	 *            Pfad zum Attribut
	 * @param emptyArray
	 *            das Feld, welches an Index pfadLoop den Platzhalterobjekt
	 *            beinhaltet.
	 * @param filledArray
	 *            Das bereits gefüllte Feld.
	 */
	private void fillEmptyArrayFirstElement(final int pfadLoop, final String path,
			final List<PlainDataValue> emptyArray, final List<PlainDataValue> filledArray) {
		final PlainDataValue emptyArrayValue = emptyArray.get(pfadLoop);
		final List<Attribute> arrayAttributes = emptyArrayValue.getFeldAttribute();
		int attributLoop;
		for (attributLoop = 1; attributLoop < arrayAttributes.size(); ++attributLoop) {
			final Attribute attribute = arrayAttributes.get(attributLoop);
			final String attributName = attribute.getName();
			String neuerPfad = path.replace(Zeichenketten.EMPTY_INDICATOR, "0")
					+ ParameterTableEditorItem.PATH_SEPERATOR + attributName;
			if (attribute.isArray()) {
				neuerPfad += ParameterTableEditorItem.PATH_SEPERATOR + Zeichenketten.EMPTY_INDICATOR;
			}
			final AttributeType at = emptyArrayValue.getAttributeType();
			List<Attribute> geradeEntdeckteArrayAttribute = null;
			if (at instanceof AttributeListDefinition) {
				final AttributeListDefinition ald = (AttributeListDefinition) at;
				final Attribute attr = ald.getAttribute(attributName);
				if (null == attr) {
					ParameterTableEditorItem.DEBUG.warning("Oops");
				} else {
					if (attr.isArray()) {
						geradeEntdeckteArrayAttribute = new ArrayList<>();
						for (final Attribute arrayAttribut : ald.getAttributes()) {
							geradeEntdeckteArrayAttribute.add(arrayAttribut);
						}
					}
				}
			}
			final PlainDataValue filler = new PlainDataValue(null, neuerPfad);
			filler.setAttributeType(at);
			if (null != geradeEntdeckteArrayAttribute) {
				for (final Attribute geradeEntdecktesAttribut : geradeEntdeckteArrayAttribute) {
					filler.addFeldAttribut(geradeEntdecktesAttribut);
				}
			}
			emptyArray.add(pfadLoop + attributLoop, filler);
			if (ParameterTableEditorItem.DO_DEBUG) {
				ParameterTableEditorItem.DEBUG.fine("Sub-Attribut hinzugefügt" + filler);
			}
		}
		// String emptyPfadNeu = emptyArrayValue.getPath().replace(
		// EMPTY_INDICATOR, "0");
		// // HINWEIS: Später mal if (isStructuredArray())
		// if (arrayAttributes.size() > 1) {
		// emptyPfadNeu += PATH_SEPERATOR + arrayAttributes.get(0);
		// }
		// if (DO_DEBUG) {
		// DEBUG.finer("Vorgängerpfad wird modifiziert", new Object[] {
		// emptyPfadNeu, emptyArrayValue.getPath() });
		// }
		// emptyArrayValue.setPath(emptyPfadNeu);
		final PlainDataValue filledPlainDataValue = filledArray.get(pfadLoop);
		emptyArrayValue.setPath(filledPlainDataValue.getPath());
	}

	/**
	 * Schreibt die Daten eines gültigen {@link PlainDataValue} in ein
	 * DAF-Data-Objekt. Bei ungültigen PlainDataValues (also solche, die weder ein
	 * DAF-Data-Objekt noch einen Uservalue haben und im Editor als <Undefiniert>
	 * erscheinen, passiert gar nichts.
	 *
	 * @param data
	 *            das Ziel-DAF-Data-Objekt.
	 * @param restPfad
	 *            Noch verbleibende Komponenten, um zu erkennen, ob ein atomares
	 *            Attribut zu schreiben oder eine komplexe Struktur vorliegt und
	 *            demzufolge in die Rekursion zu gehen ist.
	 * @param value
	 *            der zu schreibende Wert.
	 */
	private void writeDataItemRekursiv(final Data data, final String restPfad, final PlainDataValue value) {
		if (!value.isValid()) {
			return;
		}
		final String[] splitPath = restPfad.split("\\.");
		final Data item;
		if (data.isPlain()) {
			item = data;
		} else {
			item = data.getItem(splitPath[0]);
		}
		if (splitPath.length == 1) {
			final AttributeType attributeType = value.getAttributeType();
			if (attributeType instanceof StringAttributeType) {
				item.asTextValue().setText(value.getValue());
			} else if (attributeType instanceof IntegerAttributeType) {
				item.asTextValue().setText(value.getValue());
			} else if (attributeType instanceof TimeAttributeType) {
				final Long timeValue = Long.valueOf(value.getPid());
				item.asTimeValue().setMillis(timeValue);
			} else if (attributeType instanceof DoubleAttributeType) {
				item.asTextValue().setText(value.getValue());
			} else if (attributeType instanceof ReferenceAttributeType) {
				item.asTextValue().setText(value.getPid());
			}
			return;
		}
		final StringBuilder restPath = new StringBuilder(value.getPath().length());
		int restLoop;
		int restLoopStart = 1;
		if (item.isArray()) {
			ParameterTableEditorItem.DEBUG.finer("Array-Item", item.getAttributeType());
			++restLoopStart;
		}
		for (restLoop = restLoopStart; restLoop < splitPath.length; ++restLoop) {
			if (restLoop > restLoopStart) {
				restPath.append(ParameterTableEditorItem.PATH_SEPERATOR);
			}
			restPath.append(splitPath[restLoop]);
		}
		if (item.isList()) {
			writeDataItemRekursiv(item, restPath.toString(), value);
		} else if (item.isArray()) {
			if (Zeichenketten.EMPTY_INDICATOR.equals(splitPath[1])) {
				item.asArray().setLength(0);
				// value has no meaningful value here
				return;
			}
			// Integer idx = item.asArray().getLength();
			final Integer idx = Integer.parseInt(splitPath[1]);
			item.asArray().setLength(idx + 1);
			writeDataItemRekursiv(item.asArray().getItem(idx), restPath.toString(), value);
		}
	}

	/**
	 * Die Methode bestimmt den ersten Feldindex, der ein undefiniertes Element
	 * enthält. Zunächst wird Beginn und Ende des Feldes bestimmt. Danach wird über
	 * diese Subliste iteriert und bei Auftauchen des ersten undefinierten
	 * PlainDataValue wird abgebrochen - der laufende Index + 1 ist dann das
	 * Ergebnis.
	 *
	 * @param index
	 *            Index eines PlainDataValue innerhalb unserer items. Muss >= 0 und
	 *            < Anzahl items sein, sonst {@link ArrayIndexOutOfBoundsException}.
	 * @return der Arrayindex. Falls das Feld leer ist: -2, wenn der über index
	 *         referenzierte Wert gar nicht Teil eines Feldes ist: -1.
	 */
	public int getFirstUndefinedIndexOfArray(final int index) {
		int loop;
		final ArrayInTableInfo aiti = new ArrayInTableInfo(this, index, false);
		if (aiti.getFirstElementIndex() < 0) {
			return aiti.getFirstElementIndex();
		}
		final ArrayNameAndIndex relevantAri = ParameterTableEditorItem
				.getArrayIndexFromPath(plainDataValues.get(index).getPath());
		final String relevantArrayName = relevantAri.getArrayName();

		// die flache Subliste von firstElementIndex bis lastElementIndex
		// inclusive nach Feldindex strukturieren
		final List<List<PlainDataValue>> indexedLists = new ArrayList<>(aiti.getNumMembers());
		int indexLoop;
		for (indexLoop = 0; indexLoop <= aiti.getNumMembers(); ++indexLoop) {
			indexedLists.add(new ArrayList<PlainDataValue>());
		}
		for (loop = aiti.getFirstElementIndex(); loop <= aiti.getLastElementIndex(); ++loop) {
			final PlainDataValue value = plainDataValues.get(loop);
			for (final ArrayNameAndIndex arn : ParameterTableEditorItem
					.getAllArrayComponentsFromPath(value.getPath())) {
				if (arn.getArrayName().equals(relevantArrayName)) {
					int arnidx = arn.getIndex();
					if (arnidx < 0) {
						ParameterTableEditorItem.DEBUG.warning("Negativer arnidx", value);
						arnidx = 0;
					}
					indexedLists.get(arnidx).add(value);
				}
			}
		}

		// Jetzt über die strukturierte Liste iterieren. Damit wird
		// sichergestellt, dass bei Feldern von Attributlisten der gesuchte
		// Index immer richtig bestimmt wird, egal, welches Element in der Liste
		// ungültig ist.
		int result = aiti.getNumMembers();
		for (indexLoop = 0; indexLoop < aiti.getNumMembers(); ++indexLoop) {
			final List<PlainDataValue> currentList = indexedLists.get(indexLoop);
			boolean currentListIsValid = true;
			for (loop = 0; loop < currentList.size(); ++loop) {
				final PlainDataValue value = currentList.get(loop);
				if (!value.isValid()) {
					currentListIsValid = false;
					break;
				}
			}
			if (!currentListIsValid) {
				result = indexLoop;
				break;
			}
		}
		return result;
	}

	/**
	 * Einsprungmethode für das Erzeugen eines DAF-Data-Objektes aus den
	 * PlainDataValues. Diese erzeugt das notwendige Data-Objekt und geht dann in
	 * die Rekursion mit writeDataItemRekursiv(Data, String, PlainDataValue).
	 *
	 * @return das DAF-Data-Objekt.
	 */
	public Data createData() {
		final ObjektFactory f = RahmenwerkService.getService().getObjektFactory();
		Assert.isTrue(f.isVerbunden(), Zeichenketten.PLUGIN_PARAM_BEZEICHNER_KEINE_DAVVERBINDUNG);
		final Data data = f.getDav().createData(parameter.getAtg());
		int spalte;
		ParameterTableEditorItem.DEBUG.finest("Ich selbst", this);
		for (spalte = 0; spalte < plainDataValues.size(); ++spalte) {
			final PlainDataValue value = plainDataValues.get(spalte);
			writeDataItemRekursiv(data, value.getPath(), value);
		}

		return data;
	}

	/**
	 * Extrahiert aus unseren plainDataValues eine Subliste mit allen Werten, die zu
	 * einer bestimmten Substruktur gehören.
	 *
	 * @param ari
	 *            Informationen über das zu benutzende Feld.
	 * @param startIndex
	 *            Startindex für die Suche in unseren plainDataValues.
	 * @param plain
	 *            true - Suche nach atomaren Elementen in Feldern (letztes Element
	 *            ist der Index selbst). false - Suche nach komplexen Strukturen.
	 *            Dieses Flag muss vom Aufrufer in Abhängigkeit des Attributtyps
	 *            selbst bestimmt werden.
	 * @return die Subliste. Stets ungleich null, aber möglicherweise leer.
	 */
	public List<PlainDataValue> getValuesOfList(final ArrayNameAndIndex ari, final int startIndex,
			final boolean plain) {
		final List<PlainDataValue> result = new ArrayList<>(plainDataValues.size());
		final String searchString;
		if (plain) {
			searchString = ari.getArrayName();
		} else {
			searchString = ari.getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR + ari.getIndex();
		}
		int loop;
		for (loop = startIndex; loop < plainDataValues.size(); ++loop) {
			final PlainDataValue candidate = plainDataValues.get(loop);
			if (candidate.getPath().startsWith(searchString)) {
				result.add(candidate);
			}
		}
		return result;
	}

	/**
	 * Durchsucht eine Liste von Werten nach einem bestimmten Pfad und bestimmt
	 * dessen Index.
	 *
	 * @param values
	 *            die zu durchsuchende Liste
	 * @param elementPath
	 *            der zu findende Pfad
	 * @return der Index. Falls Pfad nicht gefunden wurde, values.size().
	 */
	public static int bestimmePositionInnerhalbFeld(final List<PlainDataValue> values, final String elementPath) {
		int pos;
		for (pos = 0; pos < values.size(); ++pos) {
			if (values.get(pos).getPath().equals(elementPath)) {
				break;
			}
		}
		return pos;
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof ParameterTableEditorItem) {
			return ((ParameterTableEditorItem) obj).getParameter().getObjekt().equals(parameter.getObjekt());
		}
		return super.equals(obj);
	}

	@Override
	public int hashCode() {
		return parameter.getObjekt().hashCode();
	}

	/**
	 * Liefert die Liste der Einzelattribute. Beim Instanziieren mit einem DAF
	 * Data-Objekt wird die Liste initial gefüllt, und bei
	 * {@link #catchUpBiDirectional(ParameterTableEditorItem)} eventuell mit
	 * Platzhalter-Objekten erweitert.
	 *
	 * @return die Liste.
	 */
	public List<PlainDataValue> getPlainDataValues() {
		return plainDataValues;
	}

	/**
	 * Liefert das Daten-Backend-Objekt, welches wir repräsentieren.
	 *
	 * @return der Parameter. Stets ungleich null.
	 */
	public Parameter getParameter() {
		return parameter;
	}

	/**
	 * Bestimmt das Dirty-Flag für Editor-Zwecke. Das Flag wird nirgends
	 * automatisiert gesetzt oder gelöscht aufgrund irgendwelcher Bedingungen,
	 * sondern nur explizit von außen durch unsere Benutzer.
	 *
	 * @return true - Es gab Veränderungen.
	 */
	public boolean isDirty() {
		return dirty;
	}

	/**
	 * Setzt das Dirty-Flag für Editor-Zwecke. Das Flag wird nirgends automatisiert
	 * gesetzt oder gelöscht aufgrund irgendwelcher Bedingungen, sondern nur
	 * explizit von außen durch unsere Benutzer.
	 *
	 * @param dirty
	 *            true - Markiere Veränderungen
	 */
	public void setDirty(final boolean dirty) {
		this.dirty = dirty;
	}

	/**
	 * Erzeugt neue Platzhalter-Objekte, wenn Feldelemente hinzukommen.
	 *
	 * @param prefix
	 *            Pfad-Prefix
	 * @param attributeSet
	 *            Attributset, um die Anzahl von zu erzeugenden Items zu bestimmen.
	 * @return die Liste mit den neuen Werten.
	 */
	public static List<PlainDataValue> createPlainDataValues(final String prefix, final AttributeSet attributeSet) {
		final List<PlainDataValue> values = new ArrayList<>();
		for (final Attribute a : attributeSet.getAttributes()) {
			final AttributeType type = a.getAttributeType();
			int faktor = 1;
			if (a.isArray()) {
				if (!a.isCountVariable()) {
					faktor = a.getMaxCount();
				} else {
					final PlainDataValue value = new PlainDataValue(null,
							prefix + ParameterTableEditorItem.PATH_SEPERATOR + a.getName()
									+ ParameterTableEditorItem.PATH_SEPERATOR + Zeichenketten.EMPTY_INDICATOR);
					if (type instanceof AttributeSet) {
						value.setParentAttributeSet((AttributeSet) type);
						value.setAttributeName(a.getName());
					}
					value.setAttributeType(type);

					values.add(value);
					continue;
				}
			}
			if (type instanceof AttributeSet) {
				int loop;
				for (loop = 0; loop < faktor; ++loop) {
					String neuesPrefix = prefix + ParameterTableEditorItem.PATH_SEPERATOR + a.getName();
					if (a.isArray()) {
						neuesPrefix += ParameterTableEditorItem.PATH_SEPERATOR + loop;
					}
					values.addAll(ParameterTableEditorItem.createPlainDataValues(neuesPrefix, (AttributeSet) type));
				}
			} else {
				final PlainDataValue value = new PlainDataValue(null,
						prefix + ParameterTableEditorItem.PATH_SEPERATOR + a.getName());
				value.setAttributeType(type);
				value.setParentAttributeSet(attributeSet);
				values.add(value);
			}
		}
		return values;
	}

	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder(plainDataValues.size() * 50);
		b.append("Objekt: ");
		b.append(parameter.getObjekt().toString());
		b.append('\n');
		int loop;
		for (loop = 0; loop < plainDataValues.size(); ++loop) {
			b.append(loop);
			b.append(": ");
			b.append(plainDataValues.get(loop).toString());
			b.append('\n');
		}
		return b.toString();
	}

	/**
	 * Liefert die Arbeitskopie des verkapselten DAF-Datenobjektes. Dieses wird oft
	 * einige undefinierte Attribute enthalten.
	 *
	 * @return die Kopie.
	 */
	Data getWorkingCopy() {
		return workingCopy;
	}

	/**
	 * {@inheritDoc}
	 *
	 * Das Suppress Warnings ist ok, da die Schnittstelle durch Eclipse vorgegeben
	 * wird.
	 */
	@Override
	public Object getAdapter(final Class adapter) {
		Object adapted = null;
		if (Parameter.class.equals(adapter)) {
			adapted = parameter;
		} else if (SystemObject.class.equals(adapter)) {
			adapted = parameter.getObjekt();
		}
		return adapted;
	}

}
