/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.table.provider;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.eclipse.core.runtime.Assert;

import com.bitctrl.math.SkalierungHelper;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.DataValueTextArrayMemberCombinator;
import de.bsvrz.buv.plugin.param.editors.table.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeSet;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * Repräsentiert ein atomares Attribut, welches innerhalb einer Zelle
 * dargestellt wird.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class PlainDataValue {

	/**
	 * Formatierer für absoluten Zeitstempel, sekundengenau.
	 */
	private static final DateFormat DATE_FORMAT_SEKUNDENGENAU_ABSOLUT = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");

	/**
	 * Formatierer für absoluten Zeitstempel, millisekundengenau.
	 */
	private static final DateFormat DATE_FORMAT_MILLISEKUNDENGENAU_ABSOLUT = new SimpleDateFormat(
			"dd.MM.yyyy HH:mm:ss,SSS");

	/**
	 * Kopiert von
	 * {@link de.bsvrz.dav.daf.communication.dataRepresentation.AbstractData.RelativeTimeValue#getValueText()}
	 * .
	 *
	 * @param timeValue
	 *            relative Zeitangabe in Millisekunden
	 * @param accuracy
	 *            Millisekundengenaue Ausgabe oder nicht. (Siehe
	 *            TimeAttributeType.MILLISECONDS)
	 *
	 * @return textliche Repräsentation
	 */
	private static String formatRelativeTimeLikeDaV(final long timeValue, final byte accuracy) {
		long val = timeValue;
		final StringBuffer text = new StringBuffer();
		// Beispiel=
		// "234 Tage 12 Stunden 34 Minuten 33 Sekunden 443 Millisekunden"
		final int millis = (int) (val % 1000);
		val /= 1000;
		final int seconds = (int) (val % 60);
		val /= 60;
		final int minutes = (int) (val % 60);
		val /= 60;
		final int hours = (int) (val % 24);
		val /= 24;
		final long days = val;
		if (days != 0) {
			if (days == 1) {
				text.append("1 Tag ");
			} else if (days == -1) {
				text.append("-1 Tag ");
			} else {
				text.append(days).append(" Tage ");
			}
		}
		if (hours != 0) {
			if (hours == 1) {
				text.append("1 Stunde ");
			} else if (hours == -1) {
				text.append("-1 Stunde ");
			} else {
				text.append(hours).append(" Stunden ");
			}
		}
		if (minutes != 0) {
			if (minutes == 1) {
				text.append("1 Minute ");
			} else if (minutes == -1) {
				text.append("-1 Minute ");
			} else {
				text.append(minutes).append(" Minuten ");
			}
		}
		if ((seconds != 0) || ((days == 0) && (hours == 0) && (minutes == 0) && (millis == 0))) {
			if (seconds == 1) {
				text.append("1 Sekunde ");
			} else if (seconds == -1) {
				text.append("-1 Sekunde ");
			} else {
				text.append(seconds).append(" Sekunden ");
			}
		}
		if ((millis != 0) && (accuracy == TimeAttributeType.MILLISECONDS)) {
			if (millis == 1) {
				text.append("1 Millisekunde ");
			} else if (millis == -1) {
				text.append("-1 Millisekunde ");
			} else {
				text.append(millis).append(" Millisekunden ");
			}
		}
		text.setLength(text.length() - 1);
		return text.toString();
	}

	/**
	 * Formatierer für absoluten Zeitstempel, bekommt eine der statischen
	 * Formatierer zugewiesen oder bleibt null, wenn eine relative Zeitdauer
	 * verkapselt wird.
	 */
	private DateFormat dateFormat;

	/**
	 * Ursprüngliche Daten aus dem Data-Object, wenn wir aus einem solchen heraus
	 * konstruiert werden. Kann später nicht hinzugefügt werden. Wenn wir als
	 * Lückenfüller konstruiert werden, ist dieser Member null.
	 */
	private Data dafData;

	/**
	 * Der Wert als Text, wie er zugewiesen wurde (z.B. über den Editor).
	 */
	private String userValue;

	/**
	 * Der absolute Pfad zu uns innerhalb der Attributgruppe. Die einzelnen Ebenen
	 * werden durch Punkt getrennt, die Feldindices numerisch angegeben (ohne []).
	 */
	private String path;

	/**
	 * Die Feldattribute als DAF Objekte.
	 */
	private List<Attribute> feldAttribute;

	/**
	 * Der Attributtyp. Wenn wir mit einem richtigen {@link Data}-Objekt konstruiert
	 * werden, können wir den Attributtyp selber bestimmen. Für Lückenfüller-Objekte
	 * muss er nach dem Erzeugen von außen gesetzt werden.
	 */
	private AttributeType attributeType;

	/**
	 * Die PID des Wertes. Diese wird für Vergleichsoperationen verwendet. Der
	 * String hat unterschiedliche Bedeutung in Abhängigkeit vom Attributtyp:
	 * <ul>
	 * <li>Referenz: PID=Pid des referenzierten Objekts</li>
	 * <li>Text: Der Text selbst</li>
	 * <li>Integervalue: Der unskalierte Wert</li>
	 * <li>Time: Long-Wert Millisekunden als String</li>
	 * <li>Doublevalue: Der unskalierte Wert</li>
	 * </ul>
	 */
	private String pid = "0";

	/**
	 * Übergeordnetes Attributset, in dem wir enthalten sind. Dies kann eine
	 * Atributliste oder auch die Attributgruppe selbst sein.
	 */
	private AttributeSet parentAttributeSet;

	/**
	 * Name des verkapselten Attributs.
	 */
	private String attributeName;

	/**
	 * Erzeugt das Objekt und führt Prüfungen durch (mit {@link Assert}).
	 *
	 * @param dafData
	 *            DaV-Datenobjekt. Kann <code>null</code> sein. Wenn ungleich null,
	 *            muss es plain sein
	 * @param path
	 *            Pfad zum Datum. Darf nicht null und nicht leer sein.
	 */
	public PlainDataValue(final Data dafData, final String path) {
		this.dafData = dafData;
		this.path = path;
		Assert.isNotNull(path);
		Assert.isTrue(path.length() > 0);
		if (null != dafData) {
			attributeType = dafData.getAttributeType();
			attributeName = dafData.getName();
			if (attributeType instanceof ReferenceAttributeType) {
				final SystemObject referencedSystemObject = dafData.asReferenceValue().getSystemObject();
				if (null == referencedSystemObject) {
					this.pid = "0";
				} else {
					this.pid = referencedSystemObject.getPid();
				}
			} else if (attributeType instanceof TimeAttributeType) {
				this.pid = String.valueOf(dafData.asTimeValue().getMillis());
				if (TimeAttributeType.MILLISECONDS == ((TimeAttributeType) attributeType).getAccuracy()) {
					if (!((TimeAttributeType) attributeType).isRelative()) {
						dateFormat = PlainDataValue.DATE_FORMAT_MILLISEKUNDENGENAU_ABSOLUT;
					}
				} else {
					if (!((TimeAttributeType) attributeType).isRelative()) {
						dateFormat = PlainDataValue.DATE_FORMAT_SEKUNDENGENAU_ABSOLUT;
					}
				}
			} else if (attributeType instanceof IntegerAttributeType) {
				this.pid = String.valueOf(dafData.asUnscaledValue().longValue());
			} else if (attributeType instanceof StringAttributeType) {
				this.pid = dafData.asTextValue().getValueText();
			}
		} else {
			setAttributeNameFromPath();
		}
	}

	/**
	 * Bestimmt den Attributnamen aus dem Pfad dadurch, dass der letzte Teilstring
	 * nach dem letzten Punkt als Attributname angenommen wird.
	 */
	public void setAttributeNameFromPath() {
		final String[] splitPath = path.split("\\.");
		this.attributeName = splitPath[splitPath.length - 1];
	}

	/**
	 * Hinzufügen eines Attributs, wenn wir ein Feld darstellen.
	 *
	 * @param attribute
	 *            Hinzuzufügendes Feldattribut.
	 */
	public void addFeldAttribut(final Attribute attribute) {
		if (null == attribute) {
			return;
		}
		if (null == feldAttribute) {
			feldAttribute = new ArrayList<>();
		}
		if (!feldAttribute.contains(attribute)) {
			feldAttribute.add(attribute);
		}
	}

	/**
	 * Liefert den aktuellen Wert. Wenn es einen Userwert gibt, diesen, ansonsten
	 * den ursprünglichen Data-Wert, wenn der auch <code>null</code> oder ungültig
	 * ist, dann
	 * {@link DataValueTextArrayMemberCombinator#IDENTIFIER_NICHT_DEFINIERT}.
	 *
	 * @return der Wert als Text
	 */
	public String getValue() {
		String value = DataValueTextArrayMemberCombinator.IDENTIFIER_NICHT_DEFINIERT;
		if (hasUserValue()) {
			value = getUserValue();
		} else if (isDefined()) {
			if (attributeType instanceof ReferenceAttributeType) {
				final SystemObject referencedSystemObject = dafData.asReferenceValue().getSystemObject();
				if (null != referencedSystemObject) {
					value = referencedSystemObject.toString();
				}
			} else {
				value = dafData.valueToString();
			}
		}
		return value;
	}

	/**
	 * Bestimmt, ob wir einen echten DAF-Datensatzteil verkörpern. Siehe auch Member
	 * dafData.
	 *
	 * Wenn unser Member dafData null ist, ist das Ergebnis einfach false.
	 * Kompliziert wird es, wenn er ungleich null ist. Dann müssen wir prüfen, ob er
	 * auch gültig (d.h. gefüllt) ist. Da aber {Data#isValid()} nur bei kompletten
	 * Datensätzen funktioniert und nicht bei einzelnen Subattributen, holen wird
	 * den valueToString() und vergleichen mit <Undefiniert>. Riskant, geht aber
	 * nicht anders. Bei Referenzattributen holen wir das SystemObject und
	 * vergleichen mit null.
	 *
	 * @return true - Ja, DAF-Datensatzteil vorhanden.
	 */
	public boolean isDefined() {
		boolean result = false;
		if (null != dafData) {
			if (attributeType instanceof ReferenceAttributeType) {
				final SystemObject referencedSystemObject = dafData.asReferenceValue().getSystemObject();
				result = null != referencedSystemObject;
			} else {
				final String value = dafData.valueToString();
				result = !DataValueTextArrayMemberCombinator.IDENTIFIER_NICHT_DEFINIERT.equals(value);
			}
		}
		return result;
	}

	/**
	 * Bestimmt, ob ein Benutzerwert zugewiesen wurde (z.B. im Editor).
	 *
	 * @return Ja, Benutzerwert wurde zugewiesen.
	 */
	public boolean hasUserValue() {
		return null != userValue;
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof PlainDataValue) {
			return path.equals(((PlainDataValue) obj).getPath());
		}
		return super.equals(obj);
	}

	@Override
	public int hashCode() {
		return path.hashCode();
	}

	@Override
	public String toString() {
		String result = path + ":Nutzerwert=";
		if (null == userValue) {
			result += "undefiniert";
		} else {
			result += userValue;
		}
		result += ",Datawert=";
		if (null == dafData) {
			result += "undefiniert";
		} else {
			result += dafData.valueToString();
		}
		result += ", Attributtyp=";
		if (null == attributeType) {
			result += "undefiniert";
		} else {
			result += attributeType.toString();
		}
		result += ", Attributname=";
		result += attributeName;
		return result;
	}

	/**
	 * Liefert den absoluten Pfad zu uns innerhalb der Attributgruppe. Die einzelnen
	 * Ebenen werden durch Punkt getrennt, die Feldindices numerisch angegeben (ohne
	 * []).
	 *
	 * @return der Pfad
	 */
	public String getPath() {
		return path;
	}

	/**
	 * Liefert den Wert als Text, wie er zugewiesen wurde (z.B. über den Editor)
	 *
	 * @return der Wert. Kann null sein, deshalb vorher mit {@link #hasUserValue()}
	 *         prüfen.
	 */
	public String getUserValue() {
		return userValue;
	}

	/**
	 * Setzt den absoluten Pfad zu uns innerhalb der Attributgruppe. Die einzelnen
	 * Ebenen werden durch Punkt getrennt, die Feldindices numerisch angegeben (ohne
	 * []).
	 *
	 * @param path
	 *            der neue Pfad.
	 */
	public void setPath(final String path) {
		this.path = path;
	}

	/**
	 * Liefert den Attributtyp. Wenn wir mit einem richtigen {@link Data}-Objekt
	 * konstruiert werden, können wir den Attributtyp selber bestimmen. Für
	 * Lückenfüller-Objekte muss er nach dem Erzeugen von außen gesetzt werden.
	 *
	 * @return der Attributtyp. Kann null sein.
	 */
	public AttributeType getAttributeType() {
		return attributeType;
	}

	/**
	 * Setzt den Attributtyp. Wenn wir mit einem richtigen {@link Data}-Objekt
	 * konstruiert werden, können wir den Attributtyp selber bestimmen. Für
	 * Lückenfüller-Objekte muss er nach dem Erzeugen von außen gesetzt werden.
	 *
	 * @param attributeType
	 *            der Attributtyp.
	 */
	public void setAttributeType(final AttributeType attributeType) {
		this.attributeType = attributeType;
	}

	/**
	 * Liefert die Pid unsres Wertes. Der String hat unterschiedliche Bedeutung in
	 * Abhängigkeit vom Attributtyp:
	 * <ul>
	 * <li>Referenz: PID=Pid des referenzierten Objekts</li>
	 * <li>Text: Der Text selbst</li>
	 * <li>Integervalue: Der unskalierte Wert</li>
	 * <li>Time: Long-Wert Millisekunden als String</li>
	 * <li>Doublevalue: Der unskalierte Wert</li>
	 * </ul>
	 *
	 * @return die Pid.
	 */
	public String getPid() {
		return pid;
	}

	/**
	 * Liefert den Wert als Text ohne jede Zusätze (wie z.B. Maßeineiten). Im
	 * Zweifelsfall die Pid
	 *
	 * @see de.bsvrz.buv.plugin.param.editors.ParameterEditorInput#getPlainValueFromPlainData(Data)
	 *
	 * @return der Text
	 */
	public String getValueText() {
		String result = pid;
		if (attributeType instanceof TimeAttributeType) {
			try {
				final long timeValue = Long.parseLong(pid);
				if (((TimeAttributeType) attributeType).isRelative()) {
					result = PlainDataValue.formatRelativeTimeLikeDaV(timeValue,
							((TimeAttributeType) attributeType).getAccuracy());
				} else {
					result = dateFormat.format(new Date(timeValue));
				}
			} catch (final NumberFormatException e) {
				throw new IllegalArgumentException("Die pid " + pid + " ist keine Gültige Zeitangabe", e);
			}
		}
		return result;
	}

	/**
	 * Setzt die Pid unseres Wertes und auch, abhängig vom Attributtyp, den
	 * userValue und den Pfad. Der String hat unterschiedliche Bedeutung in
	 * Abhängigkeit vom Attributtyp:
	 * <ul>
	 * <li>Referenz: PID=Pid des referenzierten Objekts</li>
	 * <li>Text: Der Text selbst</li>
	 * <li>Integervalue: Der unskalierte Wert</li>
	 * <li>Time: Long-Wert Millisekunden als String</li>
	 * <li>Doublevalue: Der unskalierte Wert</li>
	 * </ul>
	 *
	 * @param pid
	 *            die neue Pid.
	 */
	public void setPid(final String pid) {
		if (attributeType instanceof ReferenceAttributeType) {
			final ObjektFactory f = RahmenwerkService.getService().getObjektFactory();
			final SystemObject o = f.getDav().getDataModel().getObject(pid);
			if (null == o) {
				throw new IllegalArgumentException("Die pid " + pid + " referenziert kein SystemObject");
			}
			this.userValue = o.toString();
		} else if (attributeType instanceof TimeAttributeType) {
			try {
				final long timeValue = Long.parseLong(pid);
				if (((TimeAttributeType) attributeType).isRelative()) {
					userValue = PlainDataValue.formatRelativeTimeLikeDaV(timeValue,
							((TimeAttributeType) attributeType).getAccuracy());
				} else {
					userValue = dateFormat.format(new Date(timeValue)) + " Uhr";
				}
			} catch (final NumberFormatException e) {
				throw new IllegalArgumentException("Die pid " + pid + " ist keine Gültige Zeitangabe", e);
			}
		} else if (attributeType instanceof IntegerAttributeType) {
			try {
				final long longValue = Long.parseLong(pid);
				final IntegerAttributeType typ = (IntegerAttributeType) attributeType;
				boolean wertGefunden = false;
				if (null != typ.getStates()) {
					for (final IntegerValueState state : typ.getStates()) {
						if (state.getValue() == longValue) {
							userValue = state.getName();
							wertGefunden = true;
							break;
						}
					}
				}
				if (!wertGefunden) {
					if (null != typ.getRange()) {
						if (1.0 == typ.getRange().getConversionFactor()) {
							userValue = pid;
						} else {
							userValue = SkalierungHelper.getInstanz()
									.format(longValue, typ.getRange().getConversionFactor()).replace('.', ',');
						}
						final String unit = typ.getRange().getUnit();
						if ((null != unit) && (unit.length() > 0)) {
							userValue += " " + unit;
						}
					} else {
						userValue = pid;
					}
				}
			} catch (final NumberFormatException e) {
				throw new IllegalArgumentException("Die pid " + pid + " ist kein gültiger Integerwert", e);
			}
		} else if (attributeType instanceof StringAttributeType) {
			userValue = pid;
		}
		this.pid = pid;
		if (path.contains(Zeichenketten.EMPTY_INDICATOR)) {
			String arrayName = path.replace(Zeichenketten.EMPTY_INDICATOR, "0");
			if ((null != parentAttributeSet) && (null != attributeType) && !path.endsWith(attributeName)) {
				arrayName += ParameterTableEditorItem.PATH_SEPERATOR;
				arrayName += attributeName;
			}
			setPath(arrayName);
		}
	}

	/**
	 * Liefert das übergeordnete Attributset, in dem wir enthalten sind. Dies kann
	 * eine Attributliste oder auch die Attributgruppe selbst sein.
	 *
	 * @return das Set.
	 */
	public AttributeSet getParentAttributeSet() {
		return parentAttributeSet;
	}

	/**
	 * Setzt das übergeordnete Attributset, in dem wir enthalten sind. Dies kann
	 * eine Attributliste oder auch die Attributgruppe selbst sein.
	 *
	 * @param parentAttributeSet
	 *            das Set.
	 */
	public void setParentAttributeSet(final AttributeSet parentAttributeSet) {
		this.parentAttributeSet = parentAttributeSet;
	}

	/**
	 * Bestimmt, ob wir einen gültigen Wert beinhalten, also einen, den man in ein
	 * echtes {@link Data}-Objekt reinschreiben kann, um es erfolgreich zu
	 * versenden.
	 *
	 * @return true - Ja, wir beinhalten versendbare Daten.
	 */
	public boolean isValid() {
		boolean result = isDefined() || (userValue != null);
		if (!result && path.contains(Zeichenketten.EMPTY_INDICATOR)) {
			if (null == parentAttributeSet) {
				result = true;
			} else {
				result = parentAttributeSet.getAttributes().size() == 1;
			}
		}
		return result;
	}

	/**
	 * Markiert DIE DATEN des Objekts als ungültig.
	 */
	public void invalidate() {
		dafData = null;
		userValue = null;
		pid = "0";
		final ArrayNameAndIndex ari = ParameterTableEditorItem.getArrayIndexFromPath(path);
		if (0 == ari.getIndex()) {
			setPath(ari.getArrayName() + ParameterTableEditorItem.PATH_SEPERATOR + Zeichenketten.EMPTY_INDICATOR
					+ ParameterTableEditorItem.PATH_SEPERATOR + ari.getRestPfad());
		}
	}

	/**
	 * Liefert den Namen des verkapselten Attributs.
	 *
	 * @return der Name.
	 */
	public String getAttributeName() {
		return attributeName;
	}

	/**
	 * Setzt den Namen des verkapselten Attributs.
	 *
	 * @param attributeName
	 *            der Name.
	 */
	public void setAttributeName(final String attributeName) {
		this.attributeName = attributeName;
	}

	/**
	 * Liefert die Feldattribute als DAF Objekte.
	 *
	 * @return die Feldattribute. Kann <code>null</code> sein.
	 */
	public List<Attribute> getFeldAttribute() {
		return feldAttribute;
	}
}
