/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.standard;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.editor.IFormPage;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorSaveMode;
import de.bsvrz.buv.plugin.param.editors.ParameterSaveResultEnum;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver;
import de.bsvrz.buv.plugin.param.editors.ParameterSaver.ParameterSaveUrsache;
import de.bsvrz.buv.plugin.param.editors.helper.ParaEditComposite;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;

/**
 * FormPart, der eine Menge von {@link ParaEditComposite}s enthält und auf
 * dessen Datenbestand zurückgreift.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public class ParaEditCompositeFormPart extends AbstractFormPart {

	/**
	 * Die Menge aller erzeugten Composites. Die Größe entspricht der Anzahl
	 * Parameter, die vom Nutzer zum Bearbeiten ausgewählt wurden.
	 */
	private final ParaEditComposite[] paraEditComposites;

	/**
	 * true - Kein Editieren und damit kein Speichern möglich.
	 */
	private final boolean readOnly;

	/**
	 * Konstruktor speichert nur das übergebene Feld und prüft, ob es nicht
	 * <code>null</code> ist.
	 *
	 * @param paraEditComposites
	 *            das Feld von Editoren
	 * @param readOnly
	 *            true, wenn ein Speichern der Werte nicht erwünscht ist
	 */
	public ParaEditCompositeFormPart(final ParaEditComposite[] paraEditComposites, final boolean readOnly) {
		super();
		this.paraEditComposites = paraEditComposites;
		this.readOnly = readOnly;
		Assert.isNotNull(paraEditComposites);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(final boolean onSave) {
		if (!onSave) {
			return;
		}
		if (readOnly) {
			// Should have no effect since we became never dirty but...
			super.commit(onSave);
			return;
		}

		final StandardParamEditorFormPage page = (StandardParamEditorFormPage) getManagedForm().getContainer();
		final ParameterEditorSaveMode saveMode = page.getParameterEditorInput().getSaveMode();
		Parameter[] editedParameters = null;
		int loop;
		switch (saveMode) {
		case ALL:
			editedParameters = new Parameter[paraEditComposites.length];
			for (loop = 0; loop < paraEditComposites.length; ++loop) {
				final ParaEditComposite pec = paraEditComposites[loop];
				editedParameters[loop] = pec.getEditedParameter();
			}
			break;
		case ONLY_MODIFIED:
			final List<Parameter> parameterToSave = new ArrayList<>(paraEditComposites.length);
			for (loop = 0; loop < paraEditComposites.length; ++loop) {
				final ParaEditComposite pec = paraEditComposites[loop];
				if (pec.isDirty()) {
					parameterToSave.add(pec.getEditedParameter());
				}
			}
			editedParameters = parameterToSave.toArray(new Parameter[parameterToSave.size()]);
			break;
		default:
			break;
		}
		Assert.isNotNull(editedParameters,
				Zeichenketten.PLUGIN_PARAM_HINWEIS_ZU_SPEICHERNDE_PARAMETER_NICHT_BESTIMMBAR + saveMode);
		final ParameterSaveResultEnum saveResult = ParameterSaver.saveParameters(editedParameters, ParameterSaver
				.buildDefaultUrsache(editedParameters, editedParameters, ParameterSaveUrsache.PARAMETRIERT));
		switch (saveResult) {
		case PARAMETER_SAVE_FAILURE:
			/* Make Eclipse display an error dialog */
			throw new IllegalStateException(ParameterSaver.getLastError());
		case PARAMETER_SAVE_CANCEL:
			// Do nothing
			break;
		case PARAMETER_SAVE_SUCCESS:
			super.commit(onSave);
			getManagedForm().dirtyStateChanged();
			for (loop = 0; loop < paraEditComposites.length; ++loop) {
				final ParaEditComposite pec = paraEditComposites[loop];
				pec.markSaved();
			}
			break;
		default:
			/* Make Eclipse display an error dialog */
			throw new IllegalStateException("Resultat von Speichern nicht bestimmbar");
		}
	}

	@Override
	public void markStale() {
		if (!readOnly) {
			return;
		}
		final IFormPage formPage = (IFormPage) getManagedForm().getContainer();
		if (1 == formPage.getIndex()) {
			final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
			try {
				editorInput.sollParameterAbfragen();
				super.markStale();
				int loop;
				final Parameter[] neueSollParameter = editorInput.getSollParameters();
				for (loop = 0; loop < paraEditComposites.length; ++loop) {
					paraEditComposites[loop].showData(new Parameter[] { neueSollParameter[loop] }, false);
					/* Mantis#1511 */
					StandardParamEditorPlugin.getDefault().getLogger()
							.warning("Parameter " + neueSollParameter[loop].getInfo()
									+ Zeichenketten.PLUGIN_PARAM_HINWEIS_PARAM_PARAM_PRUEFEN);
				}
			} catch (final ParameterClientException e) {
				StandardParamEditorPlugin.getDefault().getLogger().error("Fehler bei Aktualisierung der Soll-Parameter",
						e);
			}
		}
		if (3 == formPage.getIndex()) {
			final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
			editorInput.istParameterAbfragen();
			super.markStale();
			int loop;
			final Parameter[] neueIstParameter = editorInput.getIstParameters();
			for (loop = 0; loop < paraEditComposites.length; ++loop) {
				paraEditComposites[loop].showData(new Parameter[] { neueIstParameter[loop] }, false);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.AbstractFormPart#markDirty()
	 */
	@Override
	public void markDirty() {
		if (readOnly) {
			return;
		}
		final IFormPage formPage = (IFormPage) getManagedForm().getContainer();
		final ParameterEditorInput editorInput = (ParameterEditorInput) formPage.getEditorInput();
		for (final Integer index : editorInput.getSelectedParameters()) {
			final ParaEditComposite tableItem = paraEditComposites[index];
			if (null != tableItem) {
				tableItem.fireExternalModificationEvent();
			}
		}
		super.markDirty();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.AbstractFormPart#refresh()
	 */
	@Override
	public void refresh() {
		final StandardParamEditorFormPage formPage = (StandardParamEditorFormPage) getManagedForm().getContainer();
		final ParameterEditorInput pei = (ParameterEditorInput) formPage.getEditorInput();
		Parameter[] parameters = null;
		switch (formPage.getIndex()) {
		case 0:
			parameters = pei.getParameters();
			break;
		case 1:
			parameters = pei.getSollParameters();
			break;
		case 2:
			parameters = pei.getDefaultParameters();
			break;
		case 3:
			parameters = pei.getIstParameters();
			break;
		default:
			break;
		}
		if (null == parameters) {
			throw new IllegalArgumentException(
					"Index " + formPage.getIndex() + " wird nicht unterstützt.\n" + "Gültige Indices sind von 0 bis 3");
		}
		int loop;
		for (loop = 0; loop < paraEditComposites.length; ++loop) {
			if (pei.getSelectedParameters().contains(loop)) {
				paraEditComposites[loop].showData(new Parameter[] { parameters[loop] }, true);
			}
		}
		// Globalen Dirty-Status bestimmen und ggf. publizieren.
		boolean globalDirtyState = false;
		for (loop = 0; loop < paraEditComposites.length; ++loop) {
			if (paraEditComposites[loop].isDirty()) {
				globalDirtyState = true;
				break;
			}
		}
		if (getManagedForm().isDirty() != globalDirtyState) {
			// dirty = false; geht nicht da private!
			super.refresh();
			getManagedForm().dirtyStateChanged();
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.AbstractFormPart#dispose()
	 */
	@Override
	public void dispose() {
		for (final ParaEditComposite pec : paraEditComposites) {
			pec.dispose();
		}
	}

	/**
	 * Liefert das Feld mit den Composites, die wir verwalten.
	 *
	 * @return das Feld
	 */
	public ParaEditComposite[] getParaEditComposites() {
		return paraEditComposites;
	}

}
