/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.standard;

import java.util.List;

import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.nebula.paperclips.core.border.LineBorder;
import org.eclipse.nebula.paperclips.core.grid.CellBackgroundProvider;
import org.eclipse.nebula.paperclips.core.grid.DefaultGridLook;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.nebula.paperclips.core.text.TextPrint;
import org.eclipse.nebula.paperclips.core.text.TextStyle;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

import de.bsvrz.buv.plugin.param.editors.AbstractParameterFormPage;
import de.bsvrz.buv.plugin.param.editors.ParameterEditor;
import de.bsvrz.buv.plugin.param.editors.ParameterEditorInput;
import de.bsvrz.buv.plugin.param.editors.helper.IParaEditCompositeContainer;
import de.bsvrz.buv.plugin.param.editors.helper.MyTreeObject;
import de.bsvrz.buv.plugin.param.editors.helper.ParaEditComposite;
import de.bsvrz.buv.plugin.param.editors.helper.TableTreeLabelProvider;
import de.bsvrz.buv.plugin.param.editors.standard.actions.ExportTreeEditorToCSVorHTMLAction;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Die (einzige) Seite im Standard-Editor für die Parameter.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public class StandardParamEditorFormPage extends AbstractParameterFormPage
		implements IParaEditCompositeContainer, RwPrintable {
	/**
	 * Debug-Logger für alle StandardParamEditorFormPage-Instanzen. Ein Logger pro
	 * Instanz lohnt nicht.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Die Beschriftung der möglichen Tab-Reiter. Daraus ergibt sich nicht nur die
	 * Beschriftung für Reiter n, sondern auch, welche Indices gültig sind, nämlich
	 * 0 <= n < {@link #VALID_INDICES}.length.
	 */
	public static final String[] VALID_INDICES = { "Vorgabe", "Soll", "Default", "Ist" };

	/**
	 * Konstruktor reicht nur durch.
	 *
	 * @param editor
	 *            the parent editor
	 * @param id
	 *            the unique identifier
	 * @param index
	 *            der Index der Seite innerhalb des Multipage-Editors.
	 */
	public StandardParamEditorFormPage(final FormEditor editor, final String id, final int index) {
		super(editor, id, StandardParamEditorFormPage.VALID_INDICES[index]);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see com.bitctrl.lib.eclipse.editors.FormPageWithBorders#createFormContent
	 * (org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createFormContent(final IManagedForm managedForm) {
		final ScrolledForm form = managedForm.getForm();
		final FormToolkit toolkit = managedForm.getToolkit();

		final TableWrapLayout layout = new TableWrapLayout();
		form.getBody().setLayout(layout);
		layout.numColumns = 2;
		updateFormText();
		final ParameterEditorInput pei = (ParameterEditorInput) getEditorInput();
		final Parameter[] parameters = pei.getParameters();
		final Parameter[] sollParameters = pei.getSollParameters();
		final Parameter[] defaultParameters = pei.getDefaultParameters();
		final Parameter[] istParameters = pei.getIstParameters();
		final ParaEditComposite[] paraEditComposites = new ParaEditComposite[parameters.length];
		int loop = 0;
		final boolean readOnly = getIndex() > ParameterEditor.EDITED_INDEX;
		for (final Parameter p : parameters) {
			final Parameter[] results = new Parameter[1];
			/*
			 * HINWEIS Überprüfe, ob die Indices korrespondieren. Notfalls iterieren und
			 * SystemObjects vergleichen.
			 */
			final Parameter korrespondierenderSollParameter = sollParameters[loop];
			if (ParameterEditor.EDITED_INDEX == getIndex()) {
				results[0] = p;
			} else if (1 == getIndex()) {
				results[0] = korrespondierenderSollParameter;
			} else if (2 == getIndex()) {
				results[0] = defaultParameters[loop];
			} else if (3 == getIndex()) {
				results[0] = istParameters[loop];
			}
			paraEditComposites[loop++] = erzeugeAbschnittDaten(form, toolkit, results, readOnly);
		}
		final ParaEditCompositeFormPart formPart = new ParaEditCompositeFormPart(paraEditComposites, readOnly);
		for (final ParaEditComposite pec : paraEditComposites) {
			pec.setFormPart(formPart);
		}
		managedForm.addPart(formPart);
		final IToolBarManager toolBarManager = form.getToolBarManager();
		final ExportTreeEditorToCSVorHTMLAction exportAction = new ExportTreeEditorToCSVorHTMLAction(formPart);
		exportAction.setTitleText(((ParameterEditor) getEditor()).getDefaultHeader(true));
		toolBarManager.add(exportAction);
		toolBarManager.update(true);
		setExportAction(exportAction);

		super.createFormContent(managedForm);
	}

	/**
	 * Erzeugt eine Form Sektion zum Editieren eines Parameters. Hauptinhalt ist das
	 * {@link ParaEditComposite}.
	 *
	 * @param form
	 *            das Mutter-Form
	 * @param toolkit
	 *            das Toolkit zum Erzeugen neuer Steuerelemente
	 * @param results
	 *            Der zu bearbeitenden Parameter
	 * @param readOnly
	 *            true - Kein Editieren möglich.
	 * @return das erzeugte Composite.
	 */
	private ParaEditComposite erzeugeAbschnittDaten(final ScrolledForm form, final FormToolkit toolkit,
			final Parameter[] results, final boolean readOnly) {
		final Section sektion = toolkit.createSection(form.getBody(), Section.DESCRIPTION
				| ExpandableComposite.TITLE_BAR | ExpandableComposite.TWISTIE | ExpandableComposite.EXPANDED);
		final TableWrapData td = new TableWrapData(TableWrapData.FILL_GRAB);
		td.colspan = 2;
		sektion.setLayoutData(td);

		sektion.setText("Objekt: " + results[0].getObjekt().toString());
		sektion.setDescription(
				"In diesem Abschnitt erfolgt das Editieren des Objektes " + results[0].getObjekt().toString());
		sektion.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				form.reflow(true);
			}
		});

		return new ParaEditComposite(toolkit, sektion, results, null, this, readOnly);
	}

	/**
	 * Aktualisiert den Überschrifts-Text des Form mit dem Textwert aus dem
	 * EditorInput-Objekt.
	 */
	public void updateFormText() {
		final IManagedForm managedForm = getManagedForm();
		final ScrolledForm form = managedForm.getForm();
		final ParameterEditorInput pei = (ParameterEditorInput) getEditorInput();
		final StringBuilder b = new StringBuilder();
		final Parameter[] parameters = pei.getParameters();
		b.append(parameters.length);
		b.append(" Objekt");
		if (parameters.length > 1) {
			b.append('e');
		}
		b.append(" in Bearbeitung");
		form.setText(b.toString());
	}

	@Override
	public void dirtyStateChanged(final ParaEditComposite pec, final boolean status) {
		if (status) {
			getEditor().editorDirtyStateChanged();
		}
	}

	@Override
	public void valueModified(final ParaEditComposite pec, final MyTreeObject myTreeObject, final Object oldValue,
			final Object newValue, final AttributeType attributeType) {
		ParaEditCompositeFormPart formPart = null;
		for (final IFormPart part0 : getManagedForm().getParts()) {
			if (part0 instanceof ParaEditCompositeFormPart) {
				formPart = (ParaEditCompositeFormPart) part0;
				break;
			}
		}
		if ((formPart != null) && (myTreeObject.getData() instanceof Data)) {
			handleAttributePropagation(pec, myTreeObject, newValue, attributeType, formPart);
		}
	}

	/**
	 * Behandelt das Setzen in einem Parameter geänderter Werte in allen anderen.
	 * Das wird aber nur gemacht, wenn der gewählte Propagation Mode auf ALL steht.
	 *
	 * @param pec
	 *            das geänderte Composite.
	 * @param myTreeObject
	 *            das geänderte Attribut
	 * @param newValue
	 *            der neue Wert.
	 * @param attributeType
	 *            der Attributtyp.
	 * @param formPart
	 *            das Formpart. Hier sind alle Composites gespeichert, so dass wir
	 *            iterieren können.
	 */
	private void handleAttributePropagation(final ParaEditComposite pec, final MyTreeObject myTreeObject,
			final Object newValue, final AttributeType attributeType, final ParaEditCompositeFormPart formPart) {
		switch (getParameterEditorInput().getAttributePropagationMode()) {
		case ONLY_EDITED:
			return;
		case EDITED_AND_SELECTION:
			throw new UnsupportedOperationException("Attributpropagierung auf Auswahl wird nicht unterstützt");
		case ALL:
			break;
		default:
			break;
		}
		final String path = myTreeObject.getFullPath();
		for (final ParaEditComposite currentPec : formPart.getParaEditComposites()) {
			if (currentPec == pec) {
				continue;
			}
			final MyTreeObject matchedMyTreeObject = currentPec.getMyTreeObjectForPath(path);
			if (matchedMyTreeObject.getData() instanceof Data) {
				final Data matchData = (Data) matchedMyTreeObject.getData();
				if (attributeType instanceof IntegerAttributeType) {
					matchData.asUnscaledValue().set((Long) newValue);
				} else if (attributeType instanceof DoubleAttributeType) {
					matchData.asUnscaledValue().set((Double) newValue);
				} else if (attributeType instanceof StringAttributeType) {
					matchData.asTextValue().setText((String) newValue);
				} else if (attributeType instanceof TimeAttributeType) {
					matchData.asUnscaledValue().set((Long) newValue);
				} else if (attributeType instanceof ReferenceAttributeType) {
					matchData.asReferenceValue().setSystemObject((SystemObject) newValue);
				}
				currentPec.fireExternalModificationEvent();
				StandardParamEditorFormPage.DEBUG.finer("Objekt modifiziert:", matchedMyTreeObject);
			}
		}
	}

	@Override
	public ParameterEditorInput getParameterEditorInput() {
		return (ParameterEditorInput) getEditorInput();
	}

	/**
	 * Liefert ein Paperclips-Grid des enthaltenen Treeviewers.
	 *
	 * @param pec
	 *            das ParaEditComposite.
	 * @return das Grid.
	 */
	private GridPrint getGridPrint(final ParaEditComposite pec) {
		final TreeViewer viewer = pec.getTreeViewer();
		final Tree tree = viewer.getTree();
		final TableTreeLabelProvider labelAndColorProvider = (TableTreeLabelProvider) viewer.getLabelProvider();
		final FontData systemFontData = tree.getDisplay().getSystemFont().getFontData()[0];
		final List<MyTreeObject> flatList = pec.getMediatorDataObject().getFlatList();

		final GridPrint tabelle = new GridPrint();
		final DefaultGridLook look = new DefaultGridLook();
		look.setHeaderBackground(new RGB(200, 200, 200));
		look.setBodyBackgroundProvider(new CellBackgroundProvider() {

			private final RGB white = new RGB(255, 255, 255);

			@Override
			public RGB getCellBackground(final int row, final int column, final int colspan) {
				final MyTreeObject item = flatList.get(row);
				final Color bColor = labelAndColorProvider.getBackground(item, column);
				if (null == bColor) {
					return white;
				}
				return bColor.getRGB();
			}
		});

		TextStyle defaultTextStyle = new TextStyle();
		final TextStyle headerStyle = defaultTextStyle.font(systemFontData.getName(), systemFontData.getHeight(),
				SWT.BOLD);
		defaultTextStyle = defaultTextStyle.font(systemFontData.getName(), systemFontData.getHeight(), SWT.NORMAL);

		look.setCellBorder(new LineBorder(new RGB(0, 0, 0)));
		tabelle.setLook(look);

		int loop;
		final int numColumns = tree.getColumns().length;
		for (loop = 0; loop < numColumns; ++loop) {
			final TreeColumn column = tree.getColumns()[loop];
			final GridColumn gridColumn = new GridColumn(SWT.LEFT, (2 + (loop * 3)) * 95, 1 + (loop * 2));
			tabelle.addColumn(gridColumn);
			tabelle.addHeader(headerStyle.create(column.getText()));
		}

		final List<Integer> ebeneList = pec.getMediatorDataObject().getEbeneList();
		final int numItems = flatList.size();
		for (loop = 0; loop < numItems; ++loop) {
			final MyTreeObject myTreeObject = flatList.get(loop);
			final int ebene = ebeneList.get(loop);
			addTreeItem(numColumns, tabelle, defaultTextStyle, myTreeObject, ebene, labelAndColorProvider);
		}
		return tabelle;
	}

	/**
	 * Fügt der zu druckenden Tabelle eine Zeile aus dem Baum hinzu. Dabei wird die
	 * logische Ebene, auf der sich diese Zeile innerhalb des Baumes befindet, durch
	 * Einrückung beachtet.
	 *
	 * @param numColumns
	 *            Anzahl Spalten
	 * @param tabelle
	 *            die zu druckende Tabelle
	 * @param defaultTextStyle
	 *            der Textstil für die einzelnen Zellen
	 * @param myTreeObject
	 *            das Modellobjekt aus dem Viewer
	 * @param ebene
	 *            die logische Ebene
	 * @param labelAndColorProvider
	 *            der Label- und Farbprovider des Viewers
	 */
	private void addTreeItem(final int numColumns, final GridPrint tabelle, final TextStyle defaultTextStyle,
			final MyTreeObject myTreeObject, final int ebene, final TableTreeLabelProvider labelAndColorProvider) {
		for (int colIdx = 0; colIdx < numColumns; ++colIdx) {
			final String colText = labelAndColorProvider.getColumnText(myTreeObject, colIdx);
			final Image image = labelAndColorProvider.getColumnImage(myTreeObject, colIdx);
			final StringBuilder realColText = new StringBuilder();
			if ((0 == colIdx) && (myTreeObject.getData() instanceof Data)) {
				final Data myTreeObjectData = (Data) myTreeObject.getData();
				if (myTreeObjectData.isArray()) {
					if (myTreeObject.hasChildren()) {
						realColText.append("+ ");
					} else {
						realColText.append("- ");
					}
				}
			}
			final GridPrint gridForTable;
			final GridPrint gridWithLeftPadding = new GridPrint();
			if (0 == colIdx) {
				final DefaultGridLook combinedLook = new DefaultGridLook();
				combinedLook.setCellPadding(10 * ebene, 0, 3, 0);
				gridWithLeftPadding.setLook(combinedLook);
			}
			gridWithLeftPadding.addColumn(
					new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
			if (image != null) {
				gridForTable = new GridPrint();
				gridForTable.addColumn(
						new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
				gridForTable.addColumn(
						new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
				gridWithLeftPadding.add(SWT.LEFT, SWT.CENTER, new ImagePrint(image.getImageData()));
				gridForTable.add(gridWithLeftPadding);
				final TextPrint tPrint = defaultTextStyle.create(colText);
				gridForTable.add(SWT.LEFT, SWT.CENTER, tPrint);
			} else {
				realColText.append(colText);
				final String realColTextStr = realColText.toString();
				final TextPrint tPrint = defaultTextStyle.create(realColTextStr);
				gridWithLeftPadding.add(SWT.LEFT, SWT.CENTER, tPrint);
				gridForTable = gridWithLeftPadding;
			}
			tabelle.add(gridForTable);
		}
	}

	@Override
	public PagePrint getDruckAuftrag() {
		final String header = ((ParameterEditor) getEditor()).getDefaultHeader(false);
		ParaEditCompositeFormPart formPart = null;
		for (final IFormPart part0 : getManagedForm().getParts()) {
			if (part0 instanceof ParaEditCompositeFormPart) {
				formPart = (ParaEditCompositeFormPart) part0;
				break;
			}
		}
		final FontData systemFontData = Display.getDefault().getSystemFont().getFontData()[0];
		final GridPrint mainGrid = new GridPrint();
		mainGrid.addColumn(
				new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
		mainGrid.add(new TextPrint(header,
				new FontData(systemFontData.getName(), systemFontData.getHeight() + 2, SWT.BOLD)));
		final DefaultGridLook mainLook = new DefaultGridLook();
		mainLook.setCellSpacing(0, systemFontData.getHeight());
		mainGrid.setLook(mainLook);
		final TextStyle objectTextStyle = new TextStyle()
				.font(new FontData(systemFontData.getName(), systemFontData.getHeight() + 1, SWT.BOLD));
		if (formPart != null) {
			for (final ParaEditComposite pec : formPart.getParaEditComposites()) {
				final GridPrint pecGrid = new GridPrint();
				pecGrid.addColumn(
						new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
				final DefaultGridLook pecLook = new DefaultGridLook();
				pecLook.setCellSpacing(0, systemFontData.getHeight() / 4);
				pecGrid.setLook(pecLook);
				final SystemObject o = pec.getEditedParameter().getObjekt();
				final String objectTitle = "Objekt: " + o.toString();
				pecGrid.add(objectTextStyle.create(objectTitle));
				final GridPrint tabelle = getGridPrint(pec);
				pecGrid.add(tabelle);
				mainGrid.add(pecGrid);
			}
		}

		return new PagePrint(new BigPrint(mainGrid));
	}

	@Override
	public String getTitel() {
		// The default provided by ParameterEditor is sufficient
		return null;
	}
}
