/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.standard.actions;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;

import com.bitctrl.lib.eclipse.actions.AbstractCopyTableAction;

import de.bsvrz.buv.plugin.param.editors.helper.MyDataObject;
import de.bsvrz.buv.plugin.param.editors.helper.MyTreeObject;
import de.bsvrz.buv.plugin.param.editors.helper.ParaEditComposite;
import de.bsvrz.buv.plugin.param.editors.helper.TableTreeLabelProvider;
import de.bsvrz.buv.plugin.param.editors.standard.ParaEditCompositeFormPart;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Action zum Exportieren der Baumeditoren nach CSV oder HTML.
 *
 * Vorlage aus Streckenprofil, credits to Christian Hösel.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class ExportTreeEditorToCSVorHTMLAction extends AbstractCopyTableAction {

	/**
	 * Konstante für den Zeileumbruch.
	 */
	private static final String LINEBREAK = "\r\n";

	/**
	 * Der Titeltext für den Export.
	 */
	private static String title = "";

	/**
	 * Konstante für den HTML Header.
	 */
	private static final String HTML_HEADER_START = "<html xmlns=\"http://www.w3.org/1999/xhtml\"><head><meta http-equiv=\"content-type\" content=\"text/html;charset=ISO-8859-1\"/><title>";

	/**
	 * Konstante für das Ende des HTML Headers und den Body.
	 */
	private static final String HTML_HEADER_ENDE = "</title></head><body>";

	/**
	 * Konstante für das Ende des Body.
	 */
	private static final String HTML_FOOTER = "</body></html>";

	/**
	 * Die Datenquelle für den Export.
	 */
	private final ParaEditCompositeFormPart treeEditorFormPart;

	/**
	 * Erzeugt eines neue Aktion für den Export eines
	 * {@link ParaEditCompositeFormPart}.
	 *
	 * @param treeEditorFormPart
	 *            der zu exportierende FormPart
	 */
	public ExportTreeEditorToCSVorHTMLAction(final ParaEditCompositeFormPart treeEditorFormPart) {
		this.treeEditorFormPart = treeEditorFormPart;
		Assert.isNotNull(treeEditorFormPart);
		Assert.isTrue(treeEditorFormPart.getParaEditComposites().length > 0);
		setToolTipText("Exportiert den Baumeditor nach CSV oder HTML");
	}

	@Override
	public void run() {
		final ParaEditComposite[] pecs = treeEditorFormPart.getParaEditComposites();
		final Control control = pecs[0].getTreeViewer().getControl();
		if (control != null && !control.isDisposed()) {
			exportToFile(control.getDisplay());
		}
	}

	/**
	 * Setzt den Titeltext, der im Header des Exports angezeigt wird.
	 *
	 * @param neuerTitle
	 *            der Titel.
	 */
	public void setTitleText(final String neuerTitle) {
		title = neuerTitle != null ? neuerTitle : "";
	}

	/**
	 * Tabellenkopf. Soll nur einmal erscheinen.
	 *
	 * @param main
	 *            das Exporttag
	 * @return der Header
	 */
	private String getHeader(final MainTag main) {
		final ParaEditComposite pec = treeEditorFormPart.getParaEditComposites()[0];
		final StringBuffer text = new StringBuffer();
		final TreeViewer treeViewer = pec.getTreeViewer();
		final Tree tree = treeViewer.getTree();
		final TreeColumn[] columns = tree.getColumns();
		for (final TreeColumn column : columns) {
			text.append(main.head.row.data.getBegin());
			final String content = column.getText();
			text.append(main.mask(content));
			text.append(main.head.row.data.getEnd());
		}
		return text.toString();
	}

	/**
	 * Liefert den Inhalt eines Objektes aus dem Tree.
	 *
	 * @param main
	 *            das Main-Tag
	 * @param pec
	 *            das zu exportierende Composite
	 * @return der Inhalt
	 */
	private String getContent(final MainTag main, final ParaEditComposite pec) {
		final StringBuffer text = new StringBuffer();
		final TreeViewer treeViewer = pec.getTreeViewer();
		final TableTreeLabelProvider labelAndColorProvider = (TableTreeLabelProvider) treeViewer.getLabelProvider();
		final Tree tree = treeViewer.getTree();
		final TreeColumn[] columns = tree.getColumns();
		final MyDataObject mediator = pec.getMediatorDataObject();
		final List<MyTreeObject> flatList = mediator.getFlatList();
		int loop;
		final int numColumns = columns.length;
		final List<Integer> ebeneList = pec.getMediatorDataObject().getEbeneList();
		final int numItems = flatList.size();
		for (loop = 0; loop < numItems; ++loop) {
			text.append(main.body.row.getBegin());
			final MyTreeObject myTreeObject = flatList.get(loop);
			final int ebene = ebeneList.get(loop);
			for (int colIdx = 0; colIdx < numColumns; ++colIdx) {
				text.append(main.body.row.data.getBegin());
				final String content = labelAndColorProvider.getColumnText(myTreeObject, colIdx);
				if (0 == colIdx) {
					int ebeneLoop = ebene;
					while (ebeneLoop-- > 0) {
						text.append("  ");
					}
					if (myTreeObject.hasChildren()) {
						text.append("Knoten:");
					} else {
						text.append("Blatt:");
					}
				}
				text.append(main.mask(content));
				text.append(main.body.row.data.getEnd());
			}
			text.append(main.body.row.getEnd());
		}

		return text.toString();
	}

	@Override
	protected String getTextForSelection(final MainTag main) {
		final ParaEditComposite[] pecs = treeEditorFormPart.getParaEditComposites();
		final StringBuffer text = new StringBuffer();

		text.append(main.getBegin());
		text.append(main.head.getBegin());
		text.append(main.head.row.getBegin());
		text.append(getHeader(main));
		text.append(main.head.row.getEnd());
		text.append(main.head.getEnd());

		text.append(main.body.getBegin());
		for (final ParaEditComposite pec : pecs) {
			// Zeile: Objekt-Pid;Objekt Name
			final SystemObject systemObject = pec.getEditedParameter().getObjekt();
			text.append(main.body.row.getBegin());
			text.append(main.body.row.data.getBegin());
			text.append(main.mask("Objekt:" + systemObject.getPidOrId()));
			text.append(main.body.row.data.getEnd());
			text.append(main.body.row.data.getBegin());
			text.append(main.mask(systemObject.getNameOrPidOrId()));
			text.append(main.body.row.data.getEnd());
			text.append(main.body.row.getEnd());

			// Zeile(n) der TreeTable
			text.append(getContent(main, pec));
		}
		text.append(main.body.getEnd());
		text.append(main.getEnd());

		return text.toString();
	}

	/**
	 * Exportiert die Verkehrsdaten ein eine Datei.
	 *
	 * @param display
	 *            Display, mit dessen Shell, der {@link FileDialog} geöffnet wird.
	 */
	protected void exportToFile(final Display display) {
		final FileDialog fd = new FileDialog(display.getActiveShell(), SWT.SAVE);
		fd.setFilterExtensions(new String[] { ".csv", ".html" });
		final String newFilePath = fd.open();
		if (null == newFilePath) {
			return;
		}
		try {
			if (!checkFileExists(display, newFilePath)) {
				return;
			}
			final File f = new File(newFilePath);
			final FileOutputStream fo = new FileOutputStream(f);
			final OutputStreamWriter ouw = new OutputStreamWriter(fo, "ISO-8859-1");
			String text = "";
			if (newFilePath.endsWith("csv".toLowerCase())) {
				text = getTextForSelection(CSV);
				ouw.write("#" + title + LINEBREAK + text);
			} else if (newFilePath.endsWith("html".toLowerCase())) {
				text = getTextForSelection(HTML);
				ouw.write(HTML_HEADER_START + title + HTML_HEADER_ENDE + text + HTML_FOOTER);
			} else {
				text = getTextForSelection(TEXT);
				ouw.write(title + LINEBREAK + text);
			}

			ouw.close();
			fo.close();
		} catch (final FileNotFoundException e) {
			MessageDialog.openError(display.getActiveShell(), "Fehler beim Export", e.getLocalizedMessage());
		} catch (final IOException e) {
			MessageDialog.openError(display.getActiveShell(), "Fehler beim Export", e.getLocalizedMessage());
		}
	}

	/**
	 * Überprüft, ob die zu schreibende Datei bereits existiert. Wenn ja, wird der
	 * Nutzer gefragt, ob sie überschrieben werden darf.
	 *
	 * @param display
	 *            Display für den eventuellen Messagedialog
	 * @param newFilePath
	 *            Pfad zur Datei
	 * @return true - Schreiben erlaubt
	 */
	private boolean checkFileExists(final Display display, final String newFilePath) {
		boolean ok = true;
		final File f = new File(newFilePath);
		if (f.exists()) {
			// überschreiben??
			ok = MessageDialog.openQuestion(display.getActiveShell(), "Die Datei " + newFilePath + " existiert bereits",
					"Die Datei " + newFilePath + " existiert bereits. Möchten Sie diese überschreiben?");
			if (ok) {
				f.delete();
			}
		}
		return ok;
	}
}
