/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.imex.wizards;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * Abstrakte Repräsentation eines Knotens im Baum zur Darstellung zu
 * exportierender Datensätze im Parameter-Export-Assistenten.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
abstract class CheckableNode {

	/**
	 * das Objekt, das implizit oder explizit durch diesen Knoten repräsentiert
	 * wird.
	 */
	private final Object value;

	/** der übergeordnete Knoten. */
	private CheckableNode parent;

	/** die Menge der Unterknoten. */
	private final Set<CheckableNode> children = new HashSet<>();

	/** der direkte Markierungsstatus des Knotens. */
	private boolean checkState;

	/**
	 * Konstruktor, erstellt eine Instanz des Knotens, der das übergebene Objekt
	 * repräsentiert.
	 *
	 * @param value
	 *            das Objekt
	 */
	protected CheckableNode(final Object value) {
		this.value = value;
	}

	/**
	 * fügt dem Knoten einen Unterknoten hinzu und setzt dessen Parent-Referenz.
	 *
	 * @param child
	 *            der Unterknoten
	 */
	protected void addChild(final CheckableNode child) {
		if (child != null) {
			if (children.add(child)) {
				child.setParent(this);
			}
		}
	}

	/**
	 * setzt den direkten Markierungsstatus, seiner Kinder und seiner Eltern auf den
	 * angegebenen Wert. Die tatsächliche Markierung eines Knotens kann in
	 * Abhängigkeit der Unterknoten davon abweichen!
	 *
	 * @param state
	 *            der zu setzende Status
	 */
	public void markRekursiv(final boolean state) {
		checkState = state;
		for (final CheckableNode child : children) {
			child.markRekursiv(state);
		}

		CheckableNode checkParent = getParent();
		while (checkParent != null) {
			checkParent.checkState = state;
			checkParent = checkParent.getParent();
		}
	}

	/**
	 * liefert alle markierten Knoten, die Instanzen der angegebenen Klasse sind,
	 * ausgehend vom aktuellen Knoten und rekursiv von allen Unterknoten.
	 *
	 * @param clazz
	 *            die Klasse der gesuchten Knoten
	 * @return die Menge der ermittelten Knoten
	 */
	public Collection<CheckableNode> getCheckedNodesOfType(final Class<? extends CheckableNode> clazz) {

		final Collection<CheckableNode> result = new ArrayList<>();
		if ((clazz == null) || this.getClass().equals(clazz)) {
			if (isChecked()) {
				result.add(this);
			}
		}

		for (final CheckableNode child : children) {
			result.addAll(child.getCheckedNodesOfType(clazz));
		}

		return result;
	}

	/**
	 * liefert die Unterknoten dieses Knotens.
	 *
	 * @return die Knoten
	 */
	protected final Set<CheckableNode> getChildren() {
		return children;
	}

	/**
	 * liefert den übergeordneten Knoten.
	 *
	 * @return den Knoten
	 */
	public final CheckableNode getParent() {
		return parent;
	}

	/**
	 * liefert den Wert (das repräsentierte Objekt) des Knotens.
	 *
	 * @return den Wert
	 */
	protected Object getValue() {
		return value;
	}

	/**
	 * liefert den logischen Markierungszustand, der vom direkten abweichen kann!
	 *
	 * @return den Zustand
	 */
	public boolean isChecked() {
		boolean result = false;
		for (final CheckableNode child : children) {
			if (child.isChecked()) {
				result = true;
				break;
			}
		}
		return result;
	}

	/**
	 * liefert den direkten Markierungszustand des Knotens.
	 *
	 * @return den Zustand
	 */
	protected final boolean isCheckState() {
		return checkState;
	}

	/**
	 * liefert den logischen Grauzustand des Knotens. Ein Knoten wird als GRAU
	 * markiert, wenn mindestens einer seiner Unterknoten markiert ist und
	 * mindestens einer nicht.
	 *
	 * @return den Zustand
	 */
	public boolean isGrayed() {
		boolean result = false;
		boolean oneChecked = false;
		for (final CheckableNode child : children) {
			if (child.isGrayed()) {
				result = true;
				break;
			}
			if (child.isChecked()) {
				oneChecked = true;
				result = false;
			} else if (oneChecked) {
				result = true;
				break;
			}
		}
		return result;
	}

	/**
	 * entfernt den angegebenen Unterknoten und setzt dessen Parent-Referenz zurück.
	 *
	 * @param child
	 *            der Unterknoten
	 */
	protected void removeChild(final CheckableNode child) {
		if (children.remove(child)) {
			child.setParent(null);
		}
	}

	/**
	 * setzt den übergeordneten Knoten.
	 *
	 * @param parent
	 *            der Knoten
	 */
	private void setParent(final CheckableNode parent) {
		this.parent = parent;
	}

	@Override
	public String toString() {
		String result = null;
		if (value == null) {
			result = "Keine Daten";
		} else {
			result = value.toString();
		}
		return result;
	}

	/**
	 * setzt den Markierungsstatus, wenn der Knoten das übergebene Objekt
	 * repräsentiert.
	 *
	 * @param object
	 *            das Objekt
	 */
	protected void initSelectionFor(final Object object) {
		if (object.equals(getValue())) {
			markRekursiv(true);
		} else {
			for (final CheckableNode child : children) {
				child.initSelectionFor(object);
			}
		}
	}

	/**
	 * liefert den ersten Knoten, dessen Wert mit dem übergebenen Objekt
	 * übereinstimmt.
	 *
	 * @param search
	 *            der Wert des gesuchten Knotens
	 * @return den gefundenen Knoten oder <code>null</code>
	 */
	protected CheckableNode findNode(final Object search) {
		CheckableNode result = null;
		if (search.equals(getValue())) {
			result = this;
		} else {
			for (final CheckableNode child : getChildren()) {
				result = child.findNode(search);
				if (result != null) {
					break;
				}
			}
		}
		return result;
	}
}
