/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.imex.wizards;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Assistentenseite zur Auswahl zu exportierender Parameterdaten.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public final class ParameterExportSelectionPage extends WizardPage {

	/**
	 * der Viewer zur Darstellung der potentiell zu exportierenden
	 * Parameterdatensätze.
	 */
	private CheckboxTreeViewer viewer;

	/** Wurzelobjekt des Auswahlbaums. */
	private DavNode davNode;

	/** Name der Datei, in die der Export erfolgen soll. */
	private String destName;

	/** Konstruktor. */
	public ParameterExportSelectionPage() {
		super("Datensatzauswahl", "Parameterdaten exportieren", null);
	}

	@Override
	public void createControl(final Composite parent) {

		setMessage("Wählen Sie die Typen, Objekte oder Datensätze aus, die exportiert werden sollen!");

		final Composite panel = new Composite(parent, SWT.NONE);
		GridLayoutFactory.swtDefaults().numColumns(2).applyTo(panel);

		Label label = new Label(panel, SWT.NONE);
		label.setText("Datensatzauswahl:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		viewer = new CheckboxTreeViewer(panel);
		viewer.setComparator(new ViewerComparator());
		final ParameterExportNodeContentProvider contentProvider = new ParameterExportNodeContentProvider();
		viewer.setContentProvider(contentProvider);
		viewer.setCheckStateProvider(contentProvider);
		viewer.addCheckStateListener(new ICheckStateListener() {
			@Override
			public void checkStateChanged(final CheckStateChangedEvent event) {
				((CheckableNode) event.getElement()).markRekursiv(event.getChecked());
				((Viewer) event.getCheckable()).refresh();
			}
		});
		davNode = new DavNode();
		viewer.setInput(davNode);

		GridDataFactory.fillDefaults().grab(true, true).span(2, 1).hint(300, 500).applyTo(viewer.getTree());

		label = new Label(panel, SWT.NONE);
		label.setText("Ziel:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		final Text destNameField = new Text(panel, SWT.BORDER);
		destNameField.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(final ModifyEvent e) {
				setDestName(destNameField.getText().trim());
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).applyTo(destNameField);

		final Button fileSelBtn = new Button(panel, SWT.PUSH);
		fileSelBtn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final FileDialog dialog = new FileDialog(getShell());
				final String result = dialog.open();
				if (result != null) {
					destNameField.setText(result);
					setDestName(result);
				}
			}
		});
		fileSelBtn.setText("Suchen");
		GridDataFactory.swtDefaults().applyTo(fileSelBtn);

		initializeSelection();
		viewer.refresh();

		setControl(panel);
	}

	/**
	 * liefert den Name der Datei, in die der Export erfolgen soll.
	 *
	 * @return den Name
	 */
	public String getDestName() {
		return destName;
	}

	/**
	 * liefert die zu exportierenden Datensätze als Map von Systemobjekten mit einer
	 * zugeordneten Liste von zu exportierenden Parameterdatensätzen.
	 *
	 * @return die Definition der Exportdaten
	 */
	public Map<SystemObject, Set<AttributeGroup>> getExportSelection() {
		final Map<SystemObject, Set<AttributeGroup>> result = new HashMap<>();

		for (final Object obj : davNode.getCheckedNodesOfType(ParameterSetNode.class)) {
			final SystemObject object = ((ParameterSetNode) obj).getSystemObject();
			final AttributeGroup atg = ((ParameterSetNode) obj).getValue();

			if ((atg != null) && (object != null)) {
				Set<AttributeGroup> set = result.get(object);
				if (set == null) {
					set = new HashSet<>();
					result.put(object, set);
				}
				set.add(atg);
			}
		}

		return result;
	}

	/**
	 * initialisiert die Selektion aus der aktuellen Auswahl der aktuellen
	 * Workbench.
	 */
	private void initializeSelection() {
		if (davNode != null) {
			final ISelection selection = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getSelectionService()
					.getSelection();
			if (selection instanceof IStructuredSelection) {
				for (final Object object : ((IStructuredSelection) selection).toArray()) {
					davNode.addInitialSelectionFor(object);
				}
			}
		}
	}

	/**
	 * setzt den Name für die Datei, in die der Export erfolgen soll.
	 *
	 * @param destName
	 *            der Name
	 */
	void setDestName(final String destName) {
		this.destName = destName;
	}
}
