/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.imex.wizards;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import de.bsvrz.buv.plugin.param.imex.ParameterImex;
import de.bsvrz.puk.param.lib.ParameterInfo;

/**
 * Dialog zur Anzeige von Fehlern beim Import von Parametern.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class ParameterImportResultDialog extends Dialog {

	/** Labelprovider für die Fehlertabelle. */
	static class ErrorLabelProvider extends LabelProvider implements ITableLabelProvider {

		@Override
		public Image getColumnImage(final Object element, final int columnIndex) {
			return null;
		}

		@Override
		public String getColumnText(final Object element, final int columnIndex) {
			String result = null;

			if (element instanceof Entry<?, ?>) {
				final Object key = ((Entry<?, ?>) element).getKey();
				final Object value = ((Entry<?, ?>) element).getValue();

				if ((key instanceof ParameterInfo) && (value instanceof String)) {
					switch (columnIndex) {
					case 0:
						result = ((ParameterInfo) key).getObjekt().toString();
						break;
					case 1:
						result = ((ParameterInfo) key).getAtg().toString();
						break;
					case 2:
						result = (String) value;
						break;
					default:
						break;
					}
				}
			}

			if (result == null) {
				result = "";
			}

			return result;
		}

	}

	/** die Liste der anzuzeigenden Fehler. */
	private final Map<ParameterInfo, String> errors;

	/**
	 * Konstruktor, erzeugt einen Dialog, um die übergebenen Fehler anzuzeigen.
	 *
	 * @param parent das übergeordnete Element
	 * @param errors die anzuzeigenden Fehler
	 */
	protected ParameterImportResultDialog(final Shell parent, final Map<ParameterInfo, String> errors) {
		super(parent);
		this.errors = errors;
	}

	@Override
	protected boolean isResizable() {
		return true;
	}

	@Override
	protected void createButtonsForButtonBar(final Composite parent) {
		createButton(parent, IDialogConstants.CANCEL_ID, "Ausgabe", false);
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
	}

	/** der Abbruch-Button dient hier zur Ausgabe der Liste von Fehlern. */
	@Override
	protected void cancelPressed() {
		final FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
		final String fileName = dialog.open();
		if (fileName != null) {
			exportErrors(fileName);
		}
	}

	/**
	 * exportiert die Fehlermeldungen in die Datei mit dem übergebenen Dateinamen.
	 *
	 * @param fileName der Name der Zieldatei
	 */
	private void exportErrors(final String fileName) {
		try (PrintWriter output = new PrintWriter(
				new OutputStreamWriter(new FileOutputStream(fileName), Charset.defaultCharset()))) {
			for (final Entry<ParameterInfo, String> error : errors.entrySet()) {
				output.print(error.getKey().getObjekt().toString());
				output.print(';');
				output.print(error.getKey().getAtg().toString());
				output.print(';');
				output.println(error.getValue());
			}
		} catch (final FileNotFoundException e) {
			ParameterImex.zeigeFehler(e);
		}
	}

	@Override
	protected Control createDialogArea(final Composite parent) {

		getShell().setText("Parameter-Import");

		final Composite panel = new Composite(parent, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(panel);
		GridLayoutFactory.swtDefaults().applyTo(panel);

		final Label label = new Label(panel, SWT.NONE);
		label.setText("Fehler beim Import: ");

		final Composite tablePanel = new Composite(panel, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).hint(500, 300).applyTo(tablePanel);

		final TableColumnLayout tabLayout = new TableColumnLayout();

		final TableViewer viewer = new TableViewer(tablePanel);
		final Table table = viewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		TableColumn column = new TableColumn(table, SWT.NONE);
		column.setText("Objekt");
		tabLayout.setColumnData(column, new ColumnWeightData(1));

		column = new TableColumn(table, SWT.NONE);
		column.setText("Attributgruppe");
		tabLayout.setColumnData(column, new ColumnWeightData(1));

		column = new TableColumn(table, SWT.NONE);
		column.setText("Fehler");
		tabLayout.setColumnData(column, new ColumnWeightData(2));

		tablePanel.setLayout(tabLayout);

		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setComparator(new ViewerComparator());
		viewer.setLabelProvider(new ErrorLabelProvider());
		viewer.setInput(errors.entrySet().toArray());

		return panel;
	}
}
