/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.imex.wizards;

import java.io.EOFException;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.SelectionDialog;

import de.bsvrz.buv.plugin.param.imex.ParameterImex;
import de.bsvrz.buv.plugin.param.imex.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.bitctrl.eclipse.dialogs.SystemObjectSelectionDialog;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.SerializableParameter;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Assistentenseite zur Zuordnung importierbarer Parameterdatensätze zu
 * Zielobjekten.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class ParameterImportSelectionPage extends WizardPage {

	/**
	 * die Menge der verfügbaren Datensätze mit den zugeordneten Zielobjekten.
	 */
	private final Map<Parameter, Set<SystemObject>> parameterSources = new HashMap<>();

	/** der Viewer für Darstellung der verfügbaren Datensätze. */
	private TableViewer parameterViewer;

	/** der Viewer für die Liste der zugeordneten Zielobjekte für den Import. */
	private TableViewer detailsViewer;

	/** ein Element zur Anzeige von Parameterdaten. */
	private Text parameterText;

	/** Standardkonstruktor. */
	public ParameterImportSelectionPage() {
		super("Datensatzauswahl", "Parameterdaten importieren", null);
	}

	@Override
	public void createControl(final Composite parent) {
		setMessage("Wählen Sie die zu importierenden Daten aus!");

		final Composite panel = new Composite(parent, SWT.NONE);
		GridLayoutFactory.swtDefaults().numColumns(2).equalWidth(false).applyTo(panel);

		Label label = new Label(panel, SWT.NONE);
		label.setText("Quelldatei:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		final Text sourceNameText = new Text(panel, SWT.BORDER | SWT.READ_ONLY);
		sourceNameText.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(final ModifyEvent e) {
				updateParameterSources(sourceNameText.getText().trim());
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).applyTo(sourceNameText);

		final Button fileSelBtn = new Button(panel, SWT.PUSH);
		fileSelBtn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final FileDialog dialog = new FileDialog(getShell());
				final String result = dialog.open();
				if (result != null) {
					sourceNameText.setText(result);
					updateParameterSources(result);
				}
			}

		});
		fileSelBtn.setText("Suchen");
		GridDataFactory.fillDefaults().applyTo(fileSelBtn);

		label = new Label(panel, SWT.NONE);
		label.setText("Datenauswahl:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		final Composite tablePanel = new Composite(panel, SWT.NONE);
		GridLayoutFactory.swtDefaults().applyTo(tablePanel);
		GridDataFactory.fillDefaults().hint(400, 250).grab(true, true).applyTo(tablePanel);
		final TableColumnLayout tabLayout = new TableColumnLayout();

		final Table table = new Table(tablePanel, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		TableColumn column = new TableColumn(table, SWT.NONE);
		column.setText("Objekt");
		tabLayout.setColumnData(column, new ColumnWeightData(1));

		column = new TableColumn(table, SWT.NONE);
		column.setText("Attributgruppe");
		tabLayout.setColumnData(column, new ColumnWeightData(1));

		tablePanel.setLayout(tabLayout);

		parameterViewer = new TableViewer(table);
		getParameterViewer().setComparator(new ViewerComparator());
		getParameterViewer().setContentProvider(new ArrayContentProvider());
		getParameterViewer().setLabelProvider(new ParameterImportSourceLabelProvider());
		getParameterViewer().addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(final SelectionChangedEvent event) {
				updateDetails();
			}
		});

		final Composite parameterButtons = new Composite(panel, SWT.NONE);
		GridLayoutFactory.swtDefaults().applyTo(parameterButtons);
		GridDataFactory.fillDefaults().grab(false, true).applyTo(parameterButtons);

		Button button = new Button(parameterButtons, SWT.PUSH);
		button.setText("Alles löschen");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				for (final Entry<Parameter, Set<SystemObject>> entry : getParameterSources().entrySet()) {
					entry.getValue().clear();
					getParameterViewer().refresh(entry);
					updateDetails();
				}
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		button = new Button(parameterButtons, SWT.PUSH);
		button.setText("Alles Standard");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				for (final Entry<Parameter, Set<SystemObject>> entry : getParameterSources().entrySet()) {
					entry.getValue().clear();
					entry.getValue().add(entry.getKey().getObjekt());
					getParameterViewer().refresh(entry);
				}
				updateDetails();
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		button = new Button(parameterButtons, SWT.PUSH);
		button.setText("Löschen");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				for (final Entry<Parameter, Set<SystemObject>> entry : getSelectedAssignements()) {
					entry.getValue().clear();
					getParameterViewer().refresh(entry);
				}
				updateDetails();
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		button = new Button(parameterButtons, SWT.PUSH);
		button.setText("Standard");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				for (final Entry<Parameter, Set<SystemObject>> entry : getSelectedAssignements()) {
					entry.getValue().clear();
					entry.getValue().add(entry.getKey().getObjekt());
					getParameterViewer().refresh(entry);
				}
				updateDetails();
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		label = new Label(panel, SWT.NONE);
		label.setText("Zuweisungen:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		detailsViewer = new TableViewer(panel, SWT.BORDER);
		getDetailsViewer().setComparator(new ViewerComparator());
		getDetailsViewer().setContentProvider(new ArrayContentProvider());
		GridDataFactory.fillDefaults().hint(400, 100).grab(true, false).applyTo(getDetailsViewer().getControl());

		final Composite assignButtons = new Composite(panel, SWT.NONE);
		GridLayoutFactory.swtDefaults().applyTo(assignButtons);
		GridDataFactory.fillDefaults().grab(false, true).applyTo(assignButtons);

		button = new Button(assignButtons, SWT.PUSH);
		button.setText("Hinzufügen");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Set<SystemObjectType> types = new HashSet<>();
				for (final Entry<Parameter, Set<SystemObject>> entry : getSelectedAssignements()) {
					final SystemObject objekt = entry.getKey().getObjekt();
					if (objekt instanceof SystemObjectType) {
						types.add((SystemObjectType) objekt);
					} else {
						types.add(entry.getKey().getTyp());
					}
				}

				if (types.size() == 1) {
					final SelectionDialog dialog = SystemObjectSelectionDialog.createListSelectionDialog(getShell(),
							(SystemObjectType) types.toArray()[0], SWT.MULTI);
					if (dialog.open() == Window.OK) {

						for (final Entry<Parameter, Set<SystemObject>> entry : getSelectedAssignements()) {
							for (final Object object : dialog.getResult()) {
								entry.getValue().add((SystemObject) object);
							}
							getParameterViewer().refresh(entry);
						}
						updateDetails();
					}
				}
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		button = new Button(assignButtons, SWT.PUSH);
		button.setText("Entfernen");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				for (final Entry<Parameter, Set<SystemObject>> entry : getSelectedAssignements()) {
					for (final Object object : ((IStructuredSelection) getDetailsViewer().getSelection()).toList()) {
						entry.getValue().remove(object);
					}
					getParameterViewer().refresh(entry);
				}
				updateDetails();
			}
		});
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(button);

		label = new Label(panel, SWT.NONE);
		label.setText("Parameterdaten:");
		GridDataFactory.fillDefaults().span(2, 1).applyTo(label);

		parameterText = new Text(panel, SWT.MULTI | SWT.BORDER | SWT.READ_ONLY | SWT.WRAP);
		GridDataFactory.fillDefaults().span(2, 1).hint(SWT.DEFAULT, 80).applyTo(parameterText);

		setControl(panel);
	}

	/**
	 * liefert die Menge der importierbaren Datensätze und den jeweiligen
	 * Zielobjekten.
	 *
	 * @return die Menge
	 */
	public final Map<Parameter, Set<SystemObject>> getParameterSources() {
		return parameterSources;
	}

	/**
	 * liefert die Menge der aktuell ausgewählten Parameterdatensätze inklusive der
	 * zugeordneten Zielobjekte.
	 *
	 * @return die Menge
	 */
	List<Entry<Parameter, Set<SystemObject>>> getSelectedAssignements() {
		final IStructuredSelection selection = (IStructuredSelection) getParameterViewer().getSelection();
		return selection.toList();
	}

	/**
	 * aktualisiert die Anzeige der zugeordneten Objekte für den Import eines
	 * Datensatzes und die Darstellung des Datenstzinhaltes.
	 *
	 */
	void updateDetails() {
		final List<Entry<Parameter, Set<SystemObject>>> assignements = getSelectedAssignements();
		parameterText.setText("");

		boolean showAssignments = true;

		final Collection<SystemObject> assigned = new ArrayList<>();
		if (assignements.size() == 1) {
			final Entry<Parameter, Set<SystemObject>> element = assignements.get(0);
			assigned.addAll(element.getValue());
			final Data data = element.getKey().getData();
			if (data == null) {
				parameterText.setText("Datensatz wird gelöscht, Vererbung tritt in Kraft!");
			} else {
				parameterText.setText(data.toString());
			}
		} else if (assignements.size() > 1) {
			boolean init = false;
			for (final Entry<Parameter, Set<SystemObject>> element : assignements) {
				final Set<SystemObject> nextList = element.getValue();
				if (!init) {
					assigned.addAll(nextList);
					init = true;
				} else {
					if ((assigned.size() != nextList.size()) || (!assigned.containsAll(nextList))) {
						getDetailsViewer().setInput(new String[] { "<mehrere Zuordnungen>" });
						showAssignments = false;
						break;
					}
				}
			}
			parameterText.setText("Keine Vorschau bei Mehrfachauswahl möglich!");
		}
		if (showAssignments) {
			getDetailsViewer().setInput(assigned.toArray());
		}
	}

	/**
	 * aktualisiert die für den Import zur Verfügung stehenden Datensätze aus der
	 * mit dem übergebenen Name beschriebenen Datei.
	 *
	 * @param sourceName
	 *            der Dateiname für den Import
	 */
	void updateParameterSources(final String sourceName) {
		parameterSources.clear();
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (rahmenwerk.isOnline()) {
			ObjectInputStream stream = null;
			try {
				stream = new ObjectInputStream(new FileInputStream(sourceName));

				final ClientDavInterface dav = rahmenwerk.getDavVerbindung();
				Object inputObject = null;
				do {
					try {
						inputObject = stream.readObject();
						if (inputObject instanceof SerializableParameter) {
							final Parameter parameter = ((SerializableParameter) inputObject).erzeugeParameter(dav);
							parameterSources.put(parameter, new HashSet<SystemObject>());
						}
					} catch (final EOFException e) {
						inputObject = null;
					}
				} while (inputObject != null);
			} catch (final FileNotFoundException e) {
				ParameterImex.zeigeFehler(e);
			} catch (final IOException e) {
				ParameterImex.zeigeFehler(e);
			} catch (final ClassNotFoundException e) {
				ParameterImex.zeigeFehler(e);
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (final IOException e) {
						// kann ignoriert werden
						Debug.getLogger().finest(e.getLocalizedMessage(), e);
					}
				}
			}
		}
		getParameterViewer().setInput(parameterSources.entrySet().toArray());
	}

	/**
	 * liefert den Viewer zur Darstellung der importierbaren Datensätze.
	 *
	 * @return den Viewer
	 */
	TableViewer getParameterViewer() {
		return parameterViewer;
	}

	/**
	 * liefert den Viewer zur Darstellung der Zielobjekte des Imports von
	 * Datensätzen.
	 *
	 * @return den Viewer
	 */
	public TableViewer getDetailsViewer() {
		return detailsViewer;
	}
}
