/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.param.imex.wizards;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.imex.ParamImExFunktion;
import de.bsvrz.buv.plugin.param.imex.ParameterImex;
import de.bsvrz.buv.plugin.param.imex.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;

/**
 * Assistent zum Import von exportierten Parameterdatensätzen.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class ParameterImportWizard extends Wizard implements IImportWizard {

	/** die Seite zur Zuordnung zu importierender Datensätze zu Zielobjekten. */
	private final transient ParameterImportSelectionPage selectionPage = new ParameterImportSelectionPage();

	@Override
	public void addPages() {
		super.addPages();
		addPage(selectionPage);
	}

	@Override
	public void init(final IWorkbench workbench, final IStructuredSelection selection) {
		// Keine Initialisierungsfunktion notwendig
	}

	@Override
	public boolean performFinish() {

		if (!ParamImExFunktion.IMPORTIEREN.isFreigegeben()) {
			ParameterImex.zeigeFehler("Sie haben keine Berechtigung zum Importieren von Parametern!");
			return false;
		}

		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();

		if (!rahmenwerk.isOnline()) {
			ParameterImex.zeigeFehler("Import nicht möglich, es besteht keine Datenverteilerverbindung!");
			return false;
		}

		final Job job = new Job("Parameter importieren") {

			private final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				final Map<ParameterInfo, String> errors = new HashMap<>();

				for (final Map.Entry<Parameter, Set<SystemObject>> source : selectionPage.getParameterSources()
						.entrySet()) {

					final Parameter parameter = source.getKey();
					final Collection<SystemObject> objects = source.getValue();

					if ((objects != null) && (!objects.isEmpty())) {
						final Collection<Parameter> auftraege = new ArrayList<>();
						for (final SystemObject object : objects) {
							ParameterInfo info = null;
							if (object instanceof SystemObjectType) {
								info = new ParameterInfo(object, (SystemObjectType) object, parameter.getAtg());
							} else {
								info = new ParameterInfo(object, parameter.getAtg());
							}
							auftraege.add(new Parameter(info, parameter.getDataWithTime()));
						}

						try {
							errors.putAll(MethodenBibliothek.getParameterManager(rahmenwerk.getDavVerbindung())
									.setParameterChecked(auftraege.toArray(new Parameter[auftraege.size()])));
						} catch (final ParameterClientException e) {
							ParameterImex.zeigeFehler(e);
						}
					}
				}

				Display.getDefault().asyncExec(new Runnable() {

					@Override
					public void run() {
						if (errors.size() == 0) {
							MessageDialog.openInformation(
									PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Parameterimport",
									"Die Parameter wurden erfolgreich importiert!");
						} else {
							new ParameterImportResultDialog(shell, errors).open();
						}
					}
				});

				return Status.OK_STATUS;
			}
		};

		job.schedule();
		return true;
	}
}
