/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.bitctrl.beans.BeanUtils;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.sys.funclib.bitctrl.modell.AenderbareMenge;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewsimulationglobal.konfigurationsdaten.KdSimulationsEigenschaften;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewsimulationglobal.objekte.Simulation;
import de.bsvrz.sys.funclib.bitctrl.modell.util.KappichModellUtil;

/**
 * Backend für die Teilansicht rechts oben in Abb 5-41. Es ist ein Bean,
 * konzeptionell aber ein Singleton. Da wir im Aktivator die einzige Instanz
 * anlegen, ist dies aber gegeben.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ArtDesParametersatzesCache implements IArtDesParametersatzesProvider {

	/**
	 * die aktuell ausgewählte Art des Parametersatzes als Text.
	 */
	private String artDesParametersatzes = IArtDesParametersatzesProvider.ART_ORIGINAL;

	/**
	 * die tatsächlich ausgewählte (aktive) Simulationsvariante.
	 */
	private short selectedSimulationsVariante = DataDescription.NO_SIMULATION_VARIANT_SET;

	/**
	 * Die üblichen Millisekunden seit 1.1.1970, Besonderheit: -1 heißt: aktueller
	 * Wert.
	 */
	private long zeitpunkt = IArtDesParametersatzesProvider.ZEITPUNKT_AKTUELL;

	/**
	 * Flag zur Anzeige des Archivstatus.
	 */
	private boolean archivAvailable;

	/**
	 * Kann zum feuern von {@link java.beans.PropertyChangeEvent}s verwendet werden.
	 */
	private final PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

	@Override
	public short getSelectedSimulationsVariante() {
		return selectedSimulationsVariante;
	}

	@Override
	public String getSimulationsVarianteString(final short simulationsVariante) {
		switch (simulationsVariante) {
		case DataDescription.NO_SIMULATION_VARIANT_SET:
			return "Automatisch";
		case 0:
			return "Original (Echtanlage)";
		default:
			final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
			String userName = "";
			if (factory.isVerbunden()) {
				final AenderbareMenge<Simulation> simulationen = KappichModellUtil.getAOE(factory).getSimulationen();
				for (final Simulation simulation : simulationen) {
					final KdSimulationsEigenschaften.Daten kdSimulationsEigenschaften = simulation
							.getKdSimulationsEigenschaften().getDatum();
					if (kdSimulationsEigenschaften == null) {
						ParamPlugin.getDefault().getLogger().error("Konfigurationsdaten für Simulation fehlen!");
					} else {
						if (kdSimulationsEigenschaften.getSimulationsVariante().getValue() == simulationsVariante) {
							userName = ": " + simulation.getName();
							break;
						}
					}
				}
			}
			return "Simulation " + simulationsVariante + userName;
		}
	}

	@Override
	public List<Short> getValidSimlationsVarianten() {
		final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
		final List<Short> validSimulationsVarianten = new ArrayList<>();
		validSimulationsVarianten.add(DataDescription.NO_SIMULATION_VARIANT_SET);
		validSimulationsVarianten.add((short) 0);

		if (factory.isVerbunden()) {
			final AenderbareMenge<Simulation> simulationen = KappichModellUtil.getAOE(factory).getSimulationen();
			for (final Simulation simulation : simulationen) {
				final KdSimulationsEigenschaften.Daten kdSimulationsEigenschaften = simulation
						.getKdSimulationsEigenschaften().getDatum();
				if (kdSimulationsEigenschaften == null) {
					ParamPlugin.getDefault().getLogger().error("Konfigurationsdaten für Simulation fehlen!");
				} else {
					validSimulationsVarianten.add(kdSimulationsEigenschaften.getSimulationsVariante().getValue());
				}
			}
		}
		Collections.sort(validSimulationsVarianten);
		return validSimulationsVarianten;
	}

	@Override
	public void setSelectedSimulationsVariante(final short selectedSimulationsVariante) {
		final boolean valid = getValidSimlationsVarianten().contains(selectedSimulationsVariante);
		if (valid) {
			if (this.selectedSimulationsVariante != selectedSimulationsVariante) {
				final short oldValue = this.selectedSimulationsVariante;
				this.selectedSimulationsVariante = selectedSimulationsVariante;
				propertyChangeSupport.firePropertyChange(
						IArtDesParametersatzesProvider.PROP_SELECTEDSIMULATIONSVARIANTE, oldValue,
						this.selectedSimulationsVariante);
			}
		} else {
			ParamPlugin.getDefault().getLogger()
					.error("Simulationsvariante " + selectedSimulationsVariante
							+ " ist nicht definiert, vorheriger Wert (" + this.selectedSimulationsVariante
							+ ") wird beibehalten.");
		}
	}

	@Override
	public long getZeitpunkt() {
		return zeitpunkt;
	}

	@Override
	public void setZeitpunkt(final long zeitpunkt) {
		final long now = RahmenwerkService.getService().getObjektFactory().getDav().getTime();
		if (zeitpunkt > now) {
			ParamPlugin.getDefault().getLogger().warning("Zeitpunkt liegt in der Zukunft, setze AKTUELL");
			final long oldValue = this.zeitpunkt;
			this.zeitpunkt = IArtDesParametersatzesProvider.ZEITPUNKT_AKTUELL;
			propertyChangeSupport.firePropertyChange(IArtDesParametersatzesProvider.PROP_ZEITPUNKT, oldValue,
					this.zeitpunkt);
		} else {
			if (this.zeitpunkt != zeitpunkt) {
				final long oldValue = this.zeitpunkt;
				this.zeitpunkt = zeitpunkt;
				propertyChangeSupport.firePropertyChange(IArtDesParametersatzesProvider.PROP_ZEITPUNKT, oldValue,
						this.zeitpunkt);
			}
		}
	}

	@Override
	public void addPropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public String toString() {
		return BeanUtils.toString(this);
	}

	@Override
	public String getArtDesParametersatzes() {
		return artDesParametersatzes;
	}

	@Override
	public void setArtDesParametersatzes(final String artDesParametersatzes) {
		if (IArtDesParametersatzesProvider.ART_ORIGINAL.equals(artDesParametersatzes)
				|| IArtDesParametersatzesProvider.ART_SIMULATION.equals(artDesParametersatzes)
				|| IArtDesParametersatzesProvider.ART_HISTORISCH.equals(artDesParametersatzes)) {
			if (!this.artDesParametersatzes.equals(artDesParametersatzes)) {
				final String oldValue = this.artDesParametersatzes;
				this.artDesParametersatzes = artDesParametersatzes;
				propertyChangeSupport.firePropertyChange(IArtDesParametersatzesProvider.PROP_ART_DES_PARAMETERSATZES,
						oldValue, artDesParametersatzes);
			}
		}

	}

	@Override
	public boolean isArchivAvailable() {
		return archivAvailable;
	}

	@Override
	public void setArchivAvailable(final boolean archivAvailable) {
		this.archivAvailable = archivAvailable;
	}
}
