/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.bitctrl.beans.BeanUtils;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.puk.param.lib.ParameterInfo;

/**
 * Klasse zum Zusammenfassen von je einem SystemObject, SystemObjectType,
 * AttributreGroup und Simulationsvariante. Im Prinzip sowas wie
 * {@link ParameterInfo}, allerdings mit der Möglichkeit, die Werte peu-a-peu zu
 * setzen. Um ein "echtes" {@link ParameterInfo}-Objekt zu bekommen, welches man
 * in die {@link de.bsvrz.puk.param.lib.MethodenBibliothek} füttern kann, steht
 * eine Methode zur Verfügung.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class ModifiableParameterInfo {

	/** das Systemobjekt, an dem die Parameter enthalten sein sollen. */
	private SystemObject objekt;

	/**
	 * eine Liste von Typen, an denen die ModifiableParameterInfo.atg definiert ist.
	 */
	private final List<SystemObjectType> typen = new ArrayList<>();

	/** die Attributgruppe, die den Parameterdatensatz beschreibt. */
	private AttributeGroup atg;

	/** die Simulationsvariante, für die ein Parameter ermittelt werden soll. */
	private short sim = DataDescription.NO_SIMULATION_VARIANT_SET;

	/**
	 * true, wenn dieses Objekt historische Parameter beschreibt. Die Methode
	 * {@link #toParameterInfos()} liefert in dem Fall eine IllegalStateException.
	 */
	private boolean historisch;

	/**
	 * Zeitstempel, hat nur Bedeutung bei historischen Parametern. Als besonderer
	 * Wert wird {@link IArtDesParametersatzesProvider#ZEITPUNKT_AKTUELL}
	 * akzeptiert.
	 */
	private long zeitpunkt;

	/**
	 * Liefert das Systemobjekt, an dem die Parameter enthalten sein sollen.
	 *
	 * @return das Objekt
	 */
	public SystemObject getObjekt() {
		return objekt;
	}

	/**
	 * Setzt das Systemobjekt, an dem die Parameter enthalten sein sollen.
	 *
	 * @param objekt
	 *            das Objekt
	 */
	public void setObjekt(final SystemObject objekt) {
		this.objekt = objekt;
	}

	/**
	 * Fügt der Liste von Typen, an denen die ModifiableParameterInfo.atg definiert
	 * ist, ein Element hinzu. Falls dieser Typ bereits enthalten ist, passiert
	 * nichts.
	 *
	 * @param type
	 *            der Typ.
	 */
	public void addTyp(final SystemObjectType type) {
		synchronized (typen) {
			if (!typen.contains(type)) {
				typen.add(type);
			}
		}
	}

	/**
	 * Entfernt aus der Liste von Typen, an denen die ModifiableParameterInfo.atg
	 * definiert ist, ein Element.
	 *
	 * @param type
	 *            der Typ.
	 */
	public void removeType(final SystemObjectType type) {
		synchronized (typen) {
			typen.remove(type);
		}
	}

	/**
	 * Liefert die Attributgruppe, die den Parameterdatensatz beschreibt.
	 *
	 * @return die Attributgruppe
	 */
	public AttributeGroup getAtg() {
		return atg;
	}

	/**
	 * Setzt die Attributgruppe, die den Parameterdatensatz beschreibt.
	 *
	 * @param atg
	 *            die Attributgruppe
	 */
	public void setAtg(final AttributeGroup atg) {
		this.atg = atg;
	}

	/**
	 * Liefert die Simulationsvariante, für die ein Parameter ermittelt werden soll.
	 *
	 * @return die Simulationsvariante
	 */
	public short getSim() {
		return sim;
	}

	/**
	 * Setzt die Simulationsvariante, für die ein Parameter ermittelt werden soll.
	 *
	 * @param sim
	 *            die Simulationsvariante
	 */
	public void setSim(final short sim) {
		if ((sim >= 0 && sim <= 999) || sim == DataDescription.NO_SIMULATION_VARIANT_SET) {
			this.sim = sim;
		} else {
			throw new IllegalArgumentException("Simulationsvariante " + sim + " ist ungültig!");
		}
	}

	/**
	 * Bestimmt, ob dieses Objekt historische Parameter beschreibt. Die Methode
	 * {@link #toParameterInfos()} liefert in dem Fall eine IllegalStateException.
	 *
	 * @return historisch?
	 */
	public boolean isHistorisch() {
		return historisch;
	}

	/**
	 * Setzt, ob dieses Objekt historische Parameter beschreibt. Die Methode
	 * {@link #toParameterInfos()} liefert in dem Fall eine IllegalStateException.
	 *
	 * @param historisch
	 *            Historisch-Flag
	 */
	public void setHistorisch(final boolean historisch) {
		this.historisch = historisch;
	}

	/**
	 * Liefert den Zeitstempel, hat nur Bedeutung bei historischen Parametern. Als
	 * besonderer Wert kann {@link IArtDesParametersatzesProvider#ZEITPUNKT_AKTUELL}
	 * geliefert werden.
	 *
	 * @return der Zeitstempel
	 */
	public long getZeitpunkt() {
		return zeitpunkt;
	}

	/**
	 * Setzt den Zeitstempel, hat nur Bedeutung bei historischen Parametern. Als
	 * besonderer Wert wird {@link IArtDesParametersatzesProvider#ZEITPUNKT_AKTUELL}
	 * akzeptiert.
	 *
	 * @param zeitpunkt
	 *            der Zeitstempel
	 */
	public void setZeitpunkt(final long zeitpunkt) {
		this.zeitpunkt = zeitpunkt;
	}

	/**
	 * Die Methode versucht, anhand des gesetzten objekt und atg den oder die Typen
	 * zu bestimmen, die für eine Konvertierung in {@link ParameterInfo} in Frage
	 * kommen.
	 *
	 * @return die Auswahl
	 */
	public ObjektAuswahlEnum autoFillTypes() {
		ObjektAuswahlEnum objektAuswahlArt = null;
		synchronized (typen) {
			if (null == atg) {
				throw new IllegalStateException("Attributgruppe muss angegeben sein");
			}
			if (null == objekt) {
				throw new IllegalStateException("Objekt muss angegeben sein");
			}
			final ParametertypenCache ptypcache = ParametertypenCache.getInstanz();
			typen.clear();
			final Set<SystemObjectType> parents = ptypcache.getParents(atg);
			SystemObjectType selectedType;
			if (objekt instanceof SystemObjectType) {
				selectedType = (SystemObjectType) objekt;
				if (selectedType.getAttributeGroups().contains(atg)) {
					// Fall 1: Es wurde ein DaV-Typ selektiert, der die ATG
					// tatsächlich enthält
					objektAuswahlArt = ObjektAuswahlEnum.TYP_ECHT;
				} else {
					// Fall 2: Es wurde ein DaV-Typ selektiert, der die ATG nur
					// indirekt über Hierarchien referenziert
					objektAuswahlArt = ObjektAuswahlEnum.TYP_VIRTUELL;
				}
			} else {
				selectedType = objekt.getType();
				if (selectedType.getAttributeGroups().contains(atg)) {
					// Fall 3: Es wurde ein DaV-Objekt selektiert, das die ATG
					// tatsächlich enthält
					objektAuswahlArt = ObjektAuswahlEnum.INSTANZ_ECHT;
				} else {
					// Fall 4: Es wurde ein DaV-Objekt selektiert, das die ATG
					// nur indirekt über Hierarchien referenziert
					objektAuswahlArt = ObjektAuswahlEnum.INSTANZ_VIRTUELL;
				}
			}
			switch (objektAuswahlArt) {
			case TYP_ECHT:
			case INSTANZ_ECHT:
				addTyp(selectedType);
				break;
			case TYP_VIRTUELL:
			case INSTANZ_VIRTUELL:
				final Set<SystemObjectType> possibleTypes = new HashSet<>();
				possibleTypes.addAll(parents);
				possibleTypes.retainAll(ptypcache.getDirectParents(atg));
				possibleTypes.retainAll(ptypcache.getMappedTypes(selectedType));
				for (final SystemObjectType t : possibleTypes) {
					addTyp(t);
				}
				break;
			default:
				// Protect ourselves from someone extending the
				// ObjektAuswahlArt enum with something we don't understand
				final String s = "Unbekannte Auswahlart " + objektAuswahlArt;
				throw new IllegalStateException(s);
			}
		}
		return objektAuswahlArt;
	}

	/**
	 * Erzeugen von neuen {@link ParameterInfo}-Objekten aus unseren Daten. Für
	 * jeden Typ wird ein eigenes Objekt erzeugt.
	 *
	 * Wirft eine IllegalStateException, wenn einer unserer Member null oder die
	 * Liste mit den Typen leer ist.
	 *
	 * @return das Objekt
	 */
	public ParameterInfo[] toParameterInfos() {
		ParameterInfo[] infos;
		synchronized (typen) {
			if (typen.isEmpty()) {
				throw new IllegalStateException("Mindestens ein Typ muss angegeben sein");
			}
			if (atg == null) {
				throw new IllegalStateException("Attributgruppe muss angegeben sein");
			}
			if (objekt == null) {
				throw new IllegalStateException("Objekt muss angegeben sein");
			}
			infos = new ParameterInfo[typen.size()];
			int loop;
			for (loop = 0; loop < typen.size(); ++loop) {
				infos[loop] = new ParameterInfo(objekt, typen.get(loop), atg, sim);
			}
		}
		return infos;
	}

	/**
	 * Erzeugen von neuen {@link ModifiableParameterInfo}-Objekten aus unseren
	 * Daten. Für jeden Typ wird ein eigenes Objekt erzeugt. Die Methode
	 * funktioniert auch mit historischen {@link ModifiableParameterInfo}s!
	 *
	 * Wirft eine IllegalStateException, wenn einer unserer Member null oder die
	 * Liste mit den Typen leer ist.
	 *
	 * @return das Objekt
	 */
	public ModifiableParameterInfo[] resolve() {
		ModifiableParameterInfo[] infos;
		synchronized (typen) {
			if (typen.isEmpty()) {
				throw new IllegalStateException("Mindestens ein Typ muss angegeben sein");
			}
			if (atg == null) {
				throw new IllegalStateException("Attributgruppe muss angegeben sein");
			}
			if (objekt == null) {
				throw new IllegalStateException("Objekt muss angegeben sein");
			}
			infos = new ModifiableParameterInfo[typen.size()];
			int loop;
			for (loop = 0; loop < typen.size(); ++loop) {
				final ModifiableParameterInfo mpi = new ModifiableParameterInfo();
				mpi.setAtg(getAtg());
				mpi.setHistorisch(isHistorisch());
				mpi.setObjekt(getObjekt());
				mpi.setSim(getSim());
				mpi.setZeitpunkt(getZeitpunkt());
				mpi.addTyp(typen.get(loop));
				infos[loop] = mpi;
			}
		}
		return infos;
	}

	/**
	 * Initialisiert das Objekt aus einem originalen ParameterInfo-Objekt aus der
	 * Methodenbibliothek. Bei Benutzung dieser Methode ist sichergestellt, dass ein
	 * nachfolgendes {@link #toParameterInfos()} ein Feld mit genau einem Element
	 * liefert.
	 *
	 * @param info
	 *            die originale ParameterInfo
	 */
	public void fromParameterInfo(final ParameterInfo info) {
		objekt = info.getObjekt();
		typen.clear();
		typen.add(info.getTyp());
		atg = info.getAtg();
		sim = info.getSim();
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		return BeanUtils.toString(this);
	}
}
