/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;

public final class ParamUtils {

	private ParamUtils() {
		// TODO Auto-generated constructor stub
	}

	/** Systemschriftart, fett. */
	private static Font boldSystemFont;

	/**
	 * Liefert die Systemschriftart, fett.
	 *
	 * @return Systemschriftart, fett.
	 */
	public static Font getBoldSystemFont() {
		if (ParamUtils.boldSystemFont == null) {
			final FontData fontData = PlatformUI.getWorkbench().getDisplay().getSystemFont().getFontData()[0];
			final FontData boldFd = new FontData(fontData.getName() + " Fett", fontData.getHeight(),
					fontData.getStyle() | SWT.BOLD);
			ParamUtils.boldSystemFont = new Font(PlatformUI.getWorkbench().getDisplay(), boldFd);
		}
		return ParamUtils.boldSystemFont;
	}

	/**
	 * Liefert einen Infotext für einen bestimmten Attributtyp. Dieser kann z.B. in
	 * einem Editor als Hilfe angeboten werden.
	 *
	 * @param attributeType
	 *            der Attributtyp. Kann <code>null</code> sein.
	 * @param kurz
	 *            true - einen kurzen Infotext erzeugen. false - Mehr Informationen
	 *            geben.
	 * @return der Text.
	 */
	public static String getInfotextForAttributeType(final AttributeType attributeType, final boolean kurz) {
		if (null == attributeType) {
			return "Keine weiteren Informationen verfügbar";
		}
		final StringBuilder b = new StringBuilder(200);
		b.append("Attributtyp: ");
		if (kurz) {
			b.append(attributeType.getNameOrPidOrId());
		} else {
			b.append(attributeType.toString());
		}
		if (attributeType instanceof IntegerAttributeType) {
			final IntegerAttributeType iat = (IntegerAttributeType) attributeType;
			b.append("\nGanzzahl. ");
			final List<IntegerValueState> states = iat.getStates();
			if ((null == states) || states.isEmpty()) {
				b.append("Keine Zustände definiert.");
			} else {
				final int numStates = states.size();
				b.append(numStates);
				if (1 == numStates) {
					b.append(" Zustand");
				} else {
					b.append(" Zustände");
				}
				b.append(" definiert: ");
				int stateLoop = 0;
				for (final IntegerValueState state : states) {
					if (stateLoop > 0) {
						b.append(", ");
					}
					b.append(state.getName());
					b.append(" (");
					b.append(state.getValue());
					b.append(")");
					++stateLoop;
					if (stateLoop > 3) {
						break;
					}
				}
			}
			b.append("\nWertebereich: ");
			final IntegerValueRange range = iat.getRange();
			long minimum, maximum;
			double faktor = 1.;
			String unit = null;
			if (null == range) {
				switch (iat.getByteCount()) {
				case IntegerAttributeType.BYTE:
					minimum = Byte.MIN_VALUE;
					maximum = Byte.MAX_VALUE;
					break;
				case IntegerAttributeType.INT:
					minimum = Integer.MIN_VALUE;
					maximum = Integer.MAX_VALUE;
					break;
				case IntegerAttributeType.SHORT:
					minimum = Short.MIN_VALUE;
					maximum = Short.MAX_VALUE;
					break;
				case IntegerAttributeType.LONG:
				default:
					minimum = Long.MIN_VALUE;
					maximum = Long.MAX_VALUE;
					break;
				}
			} else {
				faktor = range.getConversionFactor();
				minimum = range.getMinimum();
				maximum = range.getMaximum();
				unit = range.getUnit();
			}
			b.append(minimum);
			b.append(" - ");
			b.append(maximum);
			b.append("\nGenauigkeit: ");
			if (faktor > 1.) {
				b.append((long) faktor);
			} else {
				b.append(faktor);
			}
			if (null != unit) {
				b.append(" ");
				b.append(unit);
			}
		} else if (attributeType instanceof ReferenceAttributeType) {
			final ReferenceAttributeType rat = (ReferenceAttributeType) attributeType;
			b.append("\nObjektreferenz ");
			if (rat.isUndefinedAllowed()) {
				b.append("(optional), ");
			} else {
				b.append("(Pflicht), ");
			}
			final SystemObjectType rot = rat.getReferencedObjectType();
			b.append("Typ: ");
			if (null == rot) {
				b.append("Alle erlaubt");
			} else {
				if (kurz) {
					b.append(rot.getNameOrPidOrId());
				} else {
					b.append(rot.toString());
				}
			}
		} else if (attributeType instanceof TimeAttributeType) {
			final TimeAttributeType attType = (TimeAttributeType) attributeType;
			if (attType.isRelative()) {
				b.append("\nRelative Zeitangabe");
			} else {
				b.append("\nAbsolute Zeitangabe");
			}
			if (attType.getAccuracy() == TimeAttributeType.MILLISECONDS) {
				b.append(", millisekundengenau");
			} else if (attType.getAccuracy() == TimeAttributeType.SECONDS) {
				b.append(", sekundengenau");
			}
		} else if (attributeType instanceof StringAttributeType) {
			b.append("\nTextwert. Maximle Länge: ");
			final StringAttributeType sat = (StringAttributeType) attributeType;
			if (0 == sat.getMaxLength()) {
				b.append("Unbegrenzt");
			} else {
				b.append(sat.getMaxLength());
				b.append(" Zeichen");
			}
		}
		b.append("\nKurzinfo: ");
		b.append(attributeType.getInfo().getShortInfo());
		final String beschreibung = attributeType.getInfo().getDescription();
		if (beschreibung.length() > 0) {
			b.append("\nBeschreibung: ");
			if (kurz && (beschreibung.length() > 100)) {
				b.append(beschreibung.substring(0, 100));
				b.append("...");
			} else {
				b.append(beschreibung);
			}
		}
		return b.toString();
	}
}
