/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Logger;

import com.bitctrl.lib.eclipse.log.PluginLogger;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectComparatorAlphabetical;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * Ein Objekt dieser Klasse dient als Backend für die Teilansicht
 * "Parametertypen" aus Abbildung 5-41.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public final class ParametertypenCache {

	/**
	 * Das Feld aller parametrierenden Attributgruppen, für die es auch Objekte
	 * gibt.
	 */
	private final AttributeGroup[] parametrierendeAttributGruppen;

	/**
	 * Das Feld aller Typen, die mindestens eine parametrierende Attributgruppe
	 * haben.
	 */
	private final SystemObjectType[] parametrierbareSystemObjektTypen;

	/**
	 * eine Zuordnung von Typen zu parametrierbaren Attributgruppen, die an diesem
	 * Typ wirklich definiert sind.
	 */
	private final Map<SystemObjectType, List<AttributeGroup>> systemObjektParameterMap = new LinkedHashMap<>();

	/**
	 * Eine Zuordnung von Konfigurationsbereichen zu darin enthaltenen
	 * parametrierbaren Typen.
	 */
	private final Map<ConfigurationArea, List<SystemObjectType>> configAreaMap = new LinkedHashMap<>();

	/**
	 * Das Singleton.
	 */
	private static ParametertypenCache instanz;

	/**
	 * eine Zuordnung von Attributgruppen zu Typen, an denen diese direkt oder
	 * indirekt definiert ist.
	 */
	private final Map<AttributeGroup, Set<SystemObjectType>> attributeGroupParents = new LinkedHashMap<>();

	/**
	 * Dieses Objekt weist jedem virtuellen Typ, der in einer Hierarchiedefinition
	 * vorkommt, die Menge der von ihm beeinflussten Typen zu.
	 *
	 * Beispiel: Schlüssel=typ.hierarchie.bw.MQ ->
	 * Werte=[typ.messQuerschnittAllgemein]
	 */
	private final Map<SystemObjectType, Set<SystemObjectType>> hierarchieBeeinflussteTypen = new LinkedHashMap<>();

	/**
	 * Konstruktor erstellt die finalen Listen. Diese können sich nicht mehr ändern,
	 * weil die hier präsentierten Daten aus der Konfiguration stammen.
	 *
	 * @param factory
	 *            Objektfactory zum Zugriff auf das Datenmodell
	 */
	private ParametertypenCache(final ObjektFactory factory) {
		final DataModel model = factory.getDav().getDataModel();
		final List<AttributeGroup> patgs = new ArrayList<>();
		final List<SystemObject> alleTypen = model.getType("typ.typ").getElements();
		final SystemObjectType hierarchieDefinitionType = model.getType("typ.hierarchieDefinition");
		final List<SystemObject> hierarchien = hierarchieDefinitionType.getElements();
		final Map<SystemObject, List<SystemObjectType>> typVerwendungenInHierarchien = new LinkedHashMap<>();
		for (final SystemObject o : hierarchien) {
			final Data definitionData = o.getConfigurationData(model.getAttributeGroup("atg.hierarchieDefinition"));
			int loop;
			final List<HierarchieObjekteEintrag> hoe = new ArrayList<>();
			final List<SystemObjectType> currentObjects = new ArrayList<>();
			final Array hierarchieObjekteArray = definitionData.getArray("HierarchieObjekte");
			for (loop = 0; loop < hierarchieObjekteArray.getLength(); ++loop) {
				final Data currentItem = hierarchieObjekteArray.getItem(loop);
				hoe.add(new HierarchieObjekteEintrag(currentItem, model));
			}
			/*
			 * DEBUG System.out.println("\n\nMantis#1196: VOR Sortierung HOE=\n" + hoe +
			 * "\n");
			 */
			sortiereHierarchieObjektEintraege(hoe);
			/*
			 * DEBUG System.out.println("\n\nMantis#1196: NACH Sortierung HOE=\n" + hoe +
			 * "\n");
			 */
			for (final HierarchieObjekteEintrag e : hoe) {
				SystemObjectType type = e.getObjektTyp();
				if (!currentObjects.contains(type)) {
					currentObjects.add(type);
				}
				type = e.getNachfolgerTyp();
				if ((type != null) && !currentObjects.contains(type)) {
					currentObjects.add(type);
				}
			}

			typVerwendungenInHierarchien.put(o, currentObjects);
			int currentObjectLoop;
			for (currentObjectLoop = 0; currentObjectLoop < (currentObjects.size() - 1); ++currentObjectLoop) {
				final SystemObjectType virtualKeyType = currentObjects.get(currentObjectLoop);
				final Set<SystemObjectType> mappedRealTypes = new LinkedHashSet<>();
				int mappedTypeLoop;
				for (mappedTypeLoop = currentObjectLoop + 1; mappedTypeLoop < currentObjects.size(); ++mappedTypeLoop) {
					mappedRealTypes.add(currentObjects.get(mappedTypeLoop));
				}
				hierarchieBeeinflussteTypen.put(virtualKeyType, mappedRealTypes);
			}
		}
		final List<SystemObjectType> parametrierbareTypen = new ArrayList<>();
		for (final SystemObject o : alleTypen) {
			boolean typeHasParameter = false;
			final SystemObjectType type = (SystemObjectType) o;
			final List<AttributeGroup> localPAtgs = new ArrayList<>();

			// Für jeden Typ bauen wir für unsere Zwecke neue Liste von
			// Attributgruppen,
			// die an ihm definiert sind. Diese Liste enthält:
			// - alle ATGs, die direkt am Typ definiert sind
			// - alle ATGs von allen Typen, die Nachfolger von diesem Typ
			// in einer Hierarchiedefinition sind
			// HINWEIS: Rekursion unterstützen, d.h. eine Hierarchiedefinition
			// kann
			// als ein Element wieder eine Hierarchiedefinition enthalten.
			// Oder??
			final Set<AttributeGroup> virtualAtgsOfType = new LinkedHashSet<>();
			virtualAtgsOfType.addAll(type.getAttributeGroups());
			// Jetzt nachschauen, ob der untersuchte Typ evtl. in einer
			// Hierarchiedefinition ALS ERSTES ELEMENT verwendet wird
			for (final Entry<SystemObject, List<SystemObjectType>> entry : typVerwendungenInHierarchien.entrySet()) {
				final SystemObject h = entry.getKey();
				final List<SystemObjectType> typenInHierarchie = entry.getValue();
				final SystemObjectType t = typenInHierarchie.get(0);
				if (type.equals(t)) {
					int loop;
					for (loop = 1; loop < typenInHierarchie.size(); ++loop) {
						final SystemObjectType nachfolger = typenInHierarchie.get(loop);
						virtualAtgsOfType.addAll(nachfolger.getAttributeGroups());
					}
				}
			}
			for (final AttributeGroup atg : virtualAtgsOfType) {
				if (atg.isParameter()) {
					if (!patgs.contains(atg)) {
						patgs.add(atg);
					}
					localPAtgs.add(atg);
					typeHasParameter = true;
				}
			}
			for (final AttributeGroup atg : localPAtgs) {
				Set<SystemObjectType> localAtgParents;
				localAtgParents = attributeGroupParents.get(atg);
				if (localAtgParents == null) {
					localAtgParents = new LinkedHashSet<>();
					attributeGroupParents.put(atg, localAtgParents);
				}
				localAtgParents.add(type);
			}
			if (typeHasParameter) {
				parametrierbareTypen.add(type);
				systemObjektParameterMap.put(type, localPAtgs);

				final ConfigurationArea kb = type.getConfigurationArea();
				if (configAreaMap.containsKey(kb)) {
					configAreaMap.get(kb).add(type);
				} else {
					final List<SystemObjectType> kbSystemObjects = new ArrayList<>();
					kbSystemObjects.add(type);
					configAreaMap.put(kb, kbSystemObjects);
				}
			}
		}
		Collections.sort(patgs, new SystemObjectComparatorAlphabetical(false));
		Collections.sort(parametrierbareTypen, new SystemObjectComparatorAlphabetical(false));
		parametrierbareSystemObjektTypen = parametrierbareTypen
				.toArray(new SystemObjectType[parametrierbareTypen.size()]);
		parametrierendeAttributGruppen = patgs.toArray(new AttributeGroup[patgs.size()]);
		final String msg = alleTypen.size() + " Typen insgesamt, davon haben " + parametrierbareTypen.size()
				+ " Typen Parameter in " + configAreaMap.keySet().size() + " Konfigurationsbereichen mit insgesamt "
				+ patgs.size() + " parametrierende Attributgruppen";
		if (ParamPlugin.getDefault() == null) {
			Logger.getLogger(getClass().getName()).info(msg);
		} else {
			ParamPlugin.getDefault().getLogger().info(msg);
		}
	}

	/**
	 * Sortiert die Einträge der atl.hierarchieObjekt gemäß der konfigurierten
	 * Vorgänger-Nachfolger-Beziehung (Mantis#1196).
	 *
	 * @param hoe
	 *            die (potentiell) unsortierte Liste, so wie sie z.B. aus der
	 *            Konfiguration ausgelesen wurde.
	 */
	private void sortiereHierarchieObjektEintraege(final List<HierarchieObjekteEintrag> hoe) {
		if (hoe.size() <= 1) {
			/* nichts zu tun */
			return;
		}
		if (2 == hoe.size()) {
			if (hoe.get(0).getNachfolgerTyp().equals(hoe.get(1).getObjektTyp())) {
				/* Alles gut: Nachfolger des Elemetes 0 ist Typ von Element 1 */
				return;
			}
			Collections.swap(hoe, 0, 1);
			return;
		}
		final List<Integer> moeglicheEndeEntries = new ArrayList<>();
		int loop;
		for (loop = 0; loop < hoe.size(); ++loop) {
			final HierarchieObjekteEintrag e = hoe.get(loop);
			final SystemObjectType n = e.getNachfolgerTyp();
			if (null == n) {
				moeglicheEndeEntries.add(loop);
				continue;
			}
			/*
			 * Ist n irgendwo als Objekttyp enthalten? Wenn nicht, ist es ein mögliches
			 * Kettenende.
			 */
			int innerLoop;
			boolean found = false;
			for (innerLoop = 0; innerLoop < hoe.size(); ++innerLoop) {
				if (innerLoop == loop) {
					continue;
				}
				if (hoe.get(innerLoop).getObjektTyp().equals(n)) {
					found = true;
					break;
				}
			}
			if (!found) {
				moeglicheEndeEntries.add(loop);
			}
		}
		final PluginLogger plog = ParamPlugin.getDefault().getLogger();
		if (1 != moeglicheEndeEntries.size()) {
			plog.warning("Hierarchie ist nicht konsistent. Es gibt " + moeglicheEndeEntries.size()
					+ " mögliche Endeobjekte");
			return;
		}
		/* Das gefundene Endeelement ans Ende stellen, wenn nicht schon dort */
		final int endeIndex = hoe.size() - 1;
		final int istIndes = moeglicheEndeEntries.get(0).intValue();
		if (endeIndex != istIndes) {
			plog.info("Fixiere Endeelement von " + istIndes + " nach " + endeIndex);
			Collections.swap(hoe, endeIndex, istIndes);
		}
		/*
		 * Jetzt rückwärts das Feld sortieren, basierend darauf, daß das Endelement in
		 * der richtigen Position ist.
		 */
		for (loop = endeIndex; loop > 0; --loop) {
			final int currentIdx = loop - 1;
			final SystemObjectType gesuchterNachfolgerTyp = hoe.get(loop).getObjektTyp();
			int innerLoop;
			boolean found = false;
			for (innerLoop = 0; innerLoop < loop; ++innerLoop) {
				if (gesuchterNachfolgerTyp.equals(hoe.get(innerLoop).getNachfolgerTyp())) {
					found = true;
					break;
				}
			}
			if (!found) {
				plog.warning("Unterbrochene Hierarchie - Sortierung wird abgebrochen.");
				return;
			}
			if (innerLoop != currentIdx) {
				plog.info("Tausche Elemente " + currentIdx + " und " + innerLoop);
				Collections.swap(hoe, currentIdx, innerLoop);
			}
		}

		return;
	}

	/**
	 * Liefert die Liste der parametrierenden Attributgruppen, welche aktuell in
	 * Verwendung, d.h. einem Typ zugeordnet sind
	 *
	 * @return die Liste
	 */
	public AttributeGroup[] getParametrierendeAttributGruppen() {
		return parametrierendeAttributGruppen;
	}

	/**
	 * Liefert die Liste aller Typen, welche mind. eine parametrierende
	 * Attributgruppe haben
	 *
	 * @return die Liste
	 */
	public SystemObjectType[] getParametrierbareSystemObjektTypen() {
		return parametrierbareSystemObjektTypen;
	}

	/**
	 * Liefert die Liste parametrierender Attributgruppen zu einem bestimmten Typ.
	 *
	 * @param type
	 *            der Typ
	 * @return die Liste. Kann null sein.
	 */
	public List<AttributeGroup> getAttributeGroups(final SystemObjectType type) {
		return systemObjektParameterMap.get(type);
	}

	/**
	 * Liefert die Menge aller Konfigurationsbereiche, die im Cache erfaßt wurden.
	 *
	 * @return die Menge
	 */
	public Set<ConfigurationArea> getConfigurationAreas() {
		return configAreaMap.keySet();
	}

	/**
	 * Liefert die Liste aller parametrierbaren Typen, die im übergebenen
	 * Konfigurationsbereich definiert sind.
	 *
	 * @param kb
	 *            der fragliche Konfigurationsbereich
	 * @return die Liste. Kann leer, aber nicht null sein.
	 */
	public List<SystemObjectType> getSystemObjects(final ConfigurationArea kb) {
		final List<SystemObjectType> result = new ArrayList<>();
		final List<SystemObjectType> types = configAreaMap.get(kb);
		if (types != null) {
			result.addAll(types);
		}
		return result;
	}

	/**
	 * Liefert die Menge aller Systemobjekttypen, an denen die übergebene
	 * Attributgruppe benutzt wird. Im Regelfall wird die Menge genau ein Element
	 * enthalten. Eine leere Menge ist DaV-technisch zulässig, da Attributgruppen
	 * definiert sein können, ohne einem Typen zugeordnet zu sein, aber da wir im
	 * Konstruktor die Attributgruppen per Typ suchen, kann es eigentlich nicht
	 * passieren.
	 *
	 * Es können aber auch mehr als ein Element drin stehen, da Vererbung mehrfach
	 * zählt, ebenso indirekte Verweise über Hierarchiedefinitionen.
	 *
	 * @param atg
	 *            die Attributgruppe
	 * @return die Menge. Kann leer, aber nicht null sein.
	 */
	public Set<SystemObjectType> getParents(final AttributeGroup atg) {
		final Set<SystemObjectType> result = new LinkedHashSet<>();
		final Set<SystemObjectType> parents = attributeGroupParents.get(atg);
		if (parents != null) {
			result.addAll(parents);
		}
		return result;
	}

	/**
	 * Liefert die Menge aller Systemobjekttypen, an denen die übergebene
	 * Attributgruppe tatsächlich referenziert wird. Im Regelfall wird die Menge
	 * genau ein Element enthalten. Eine leere Menge ist DaV-technisch zulässig, da
	 * Attributgruppen definiert sein können, ohne einem Typen zugeordnet zu sein,
	 * aber da wir im Konstruktor die Attributgruppen per Typ suchen, kann es
	 * eigentlich nicht passieren.
	 *
	 * Die Menge kann auch mehrere Elemente enthalten, und zwar in genau zwei
	 * Fällen:
	 *
	 * a) Die Attributgruppe ist mehreren Typen zugeordnet, die zueinander nicht in
	 * einer Vererbungsbeziehung stehen.
	 *
	 * b) Der Typ, an dem die Attributgruppe definiert ist, hat Subtypen
	 *
	 * @param atg
	 *            die Attributgruppe
	 * @return die Menge. Kann leer, aber nicht null sein.
	 */
	public Set<SystemObjectType> getRealParents(final AttributeGroup atg) {
		final Set<SystemObjectType> result = new LinkedHashSet<>();
		final Set<SystemObjectType> parents = attributeGroupParents.get(atg);
		if (parents != null) {
			for (final SystemObjectType t : parents) {
				if (t.getAttributeGroups().contains(atg)) {
					result.add(t);
				}
			}
		}
		return result;
	}

	/**
	 * Liefert die Menge aller Systemobjekttypen, an denen die übergebene
	 * Attributgruppe wirklich definiert ist. Die Menge enthält also nicht die
	 * Subtypen.
	 *
	 * @param atg
	 *            die Attributgruppe
	 * @return die Menge. Kann leer, aber nicht null sein.
	 */
	public Set<SystemObjectType> getDirectParents(final AttributeGroup atg) {
		final Set<SystemObjectType> result = new LinkedHashSet<>();
		final Set<SystemObjectType> realParents = getRealParents(atg);
		result.addAll(realParents);
		for (final SystemObjectType r : realParents) {
			final Iterator<SystemObjectType> iter = result.iterator();
			while (iter.hasNext()) {
				final SystemObjectType t = iter.next();
				if (r.getSubTypes().contains(t)) {
					iter.remove();
				}
			}
		}
		return result;
	}

	/**
	 * Liefert eine Menge von Typen, welche durch die Parametrierung eines
	 * virtuellen Typen beeinflusst werden können.
	 *
	 * @param virtualType
	 *            ein virtueller Typ, der in einer Hierarchiedefinition in Beziehung
	 *            zu anderen Typen gesetzt wird.
	 * @return die Menge beeinflusster Typen. Kann leer, aber nicht
	 *         <code>null</code> sein.
	 */
	public Set<SystemObjectType> getMappedTypes(final SystemObjectType virtualType) {
		final Set<SystemObjectType> result = new LinkedHashSet<>();
		final Set<SystemObjectType> knownMappedTypes = hierarchieBeeinflussteTypen.get(virtualType);
		if (knownMappedTypes != null) {
			result.addAll(knownMappedTypes);
		}
		return result;
	}

	/**
	 * Erzeugt das Singleton. Wenn es schon besteht, passiert gar nichts.
	 *
	 * @param factory
	 *            benötigte Objektfactory
	 */
	public static void erzeugeInstanz(final ObjektFactory factory) {
		if ((ParametertypenCache.instanz == null) && factory.isVerbunden()) {
			ParametertypenCache.instanz = new ParametertypenCache(factory);
		}
	}

	/**
	 * Liefert das Singleton. Falls es noch nicht angelegt wurde, gibt es eine
	 * {@link IllegalStateException}
	 *
	 * @return die Instanz
	 */
	public static ParametertypenCache getInstanz() {
		if (ParametertypenCache.instanz == null) {
			throw new IllegalStateException("Das Singleton der Klasse " + ParametertypenCache.class.getSimpleName()
					+ " muss zunächst mit der Methode createInstanz erzeugt werden!");
		}
		return ParametertypenCache.instanz;
	}

}
