/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors;

import org.eclipse.ui.forms.editor.IFormPage;

/**
 * Beschreibt eine Klasse, mit Hilfe derer Instanzen man {@link IFormPage}s
 * erstellen kann, welche das Bearbeiten einer bestimmten
 * {@link de.bsvrz.dav.daf.main.config.AttributeGroup} im
 * {@link ParameterEditor} erlauben.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public interface IParameterFormPageFactory {

	/**
	 * Grenzwert für die Eignung von Editoren für die Bearbeitung eines gegebenen
	 * Editorinputs. Angepasste Parametereditoren nach TBuV-138 sollten in der
	 * Methode {@link #getAdequacy(ParameterEditorInput)} einen Wert größer oder
	 * gleich diesem zurückliefern.
	 */
	int ADEQUACY_SPECIAL = 10;

	/**
	 * Erstellt die Editorseite.
	 *
	 * @param parameterEditor
	 *            die Instanz des Editors, in welchen die neue Seite eingebaut wird.
	 *            Implementierer könnten sich den
	 *            {@link org.eclipse.ui.IEditorInput} holen, er ist vom Typ
	 *            {@link ParameterEditorInput}, und an diesem die zu editierende
	 *            Attributgruppe erfragen, um sicher zu gehen, dass dies korrekt
	 *            ist. In der Regel ist das aber unnötig, da vorher die Methode
	 *            {@link #canEdit(ParameterEditorInput)} aufgerufen wird, und wenn
	 *            diese <code>false</code> liefert, so wird die Seite gar nicht
	 *            abgefordert.
	 * @param index
	 *            Der aktuelle Index der zu erzeugenden Seite. Es gilt 0 <= index <
	 *            {@link #getNumFormPages(ParameterEditor)}
	 * @return die neue Seite
	 */
	IFormPage createFormPage(ParameterEditor parameterEditor, int index);

	/**
	 * Erfragt die Anzahl von Seiten, aus denen der Editor bestehen wird.
	 *
	 * @param parameterEditor
	 *            die Instanz des Editors, in welchen die Seiten eingebaut werden.
	 *            Implementierer könnten sich den
	 *            {@link org.eclipse.ui.IEditorInput} holen, er ist vom Typ
	 *            {@link ParameterEditorInput}, und an diesem die zu editierende
	 *            Objekt/Typ/Attributgruppe erfragen, da diese die Anzahl der Seiten
	 *            bestimmen könnte.
	 * @return die Anzahl. Muss größer als 0 sein.
	 */
	int getNumFormPages(ParameterEditor parameterEditor);

	/**
	 * Diese Methode bestimmt, ob die von dieser Factory über die Methode
	 * {@link #createFormPage(ParameterEditor, int)} erstellbare Seite das
	 * übergebene Input-Objekt behandeln (d.h. editieren) kann.
	 *
	 * @param editorInput
	 *            die zu bearbeitenden Parameter
	 *
	 * @return true - editorInput kann behandelt werden
	 */
	boolean canEdit(final ParameterEditorInput editorInput);

	/**
	 * Liefert ein Objekt, welches den Editor näher beschreibt.
	 *
	 * @return das Info-Objekt
	 */
	ParameterEditorInfo getEditorInfo();

	/**
	 * Liefert den Defaultwert für die Art, wie das Ändern eines Attributs sich auf
	 * die korrespondierenden Attribute in den anderen editierten Parametern
	 * auswirkt.
	 *
	 * @return die Art
	 */
	AttributePropagationMode getDefaultAttributePropagationMode();

	/**
	 * Bestimmt die Eignung eines Editors zur Bearbeitung einer Menge von
	 * Parametern, repräsentiert durch das übergebene Input-Objekt. Interne Editoren
	 * liefern Werte kleiner {@value #ADEQUACY_SPECIAL}.
	 *
	 * @param editorInput
	 *            die zu bearbeitenden Parameter
	 * @return die Eignung. Je größer dieser Wert, umso eher wird dieser Editor vom
	 *         Framework als Default genommen.
	 */
	int getAdequacy(final ParameterEditorInput editorInput);

	/**
	 * Liefert den Identifizierer für die Kontext-sensitive Hilfe für unsere
	 * Editoren.
	 *
	 * @return die Id. Darf nicht null oder leer sein.
	 */
	String getHelpContextId();

	/**
	 * Liefert die Stylebits für das Master ScrolledForm, welches vom Toolkit
	 * erzeugt sind. Erlaubt sind die logisch ODER Kombination folgender Bits:
	 * <ul>
	 * <li>SWT.H_SCROLL</li>
	 * <li>SWT.V_SCROLL</li>
	 * </ul>
	 *
	 * @return die Bits oder 0.
	 */
	int getFormStyleBits();
}
