/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.buv.rw.basislib.legende.ILegende;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;

/**
 * Basisklasse für alle Parametereditoren. Sie ist inspiriert von
 * {@link com.bitctrl.lib.eclipse.editors.ExtendableFormEditor}, benutzt aber
 * nicht die Schnittstelle
 * {@link com.bitctrl.lib.eclipse.editors.IFormPageFactory}, sondern die
 * Schnittstelle {@link IParameterFormPageFactory}. Diese erlaubt ebenso das
 * Abfordern der eigentlichen {@link org.eclipse.ui.forms.editor.FormPage}, hat
 * aber eine zusätzliche Methode
 * {@link IParameterFormPageFactory#canEdit(ParameterEditorInput)}, welche es
 * erlaubt abzufragen, ob die im Editor-Input enthaltene Attributgruppe
 * unterstützt wird. Dadurch haben wir beim Öffnen des Editors die Möglichkeit,
 * alle verfügbaren Editoren für einen bestimmten Parameter (=
 * {@link de.bsvrz.dav.daf.main.config.AttributeGroup}) abzufragen und dem
 * Nutzer die Auswahl zu überlassen, welcher verwendet wird. Nur von diesem
 * rufen wir dann createFormPage wirklich auf.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public class ParameterEditor extends FormEditor implements RwPrintable {

	/** die ID des Editors. */
	public static final String EDITOR_ID = ParameterEditor.class.getName();

	/** die Kontext-Hilfe-ID. */
	public static final String HILFE_ID = ParamPlugin.PLUGIN_ID + "." + ParameterEditor.class.getSimpleName();

	/**
	 * Der Kontext für die Parameter-Editoren. Damit wird der ShortCut STRG+E zum
	 * Exportieren der Parameter aktiviert und deaktiviert.
	 */
	public static final String PARAMETER_EDITOR_KONTEXT = "de.bsvrz.buv.plugin.param.editor.context";

	/**
	 * Der Index derjenigen Page, welche die editierbaren Composites enthält (Reiter
	 * für Aspekt "Vorgabe", siehe Abb 5-43).
	 *
	 */
	public static final int EDITED_INDEX = 0;

	/**
	 * Die Beschriftung der möglichen Tab-Reiter. Daraus ergibt sich nicht nur die
	 * Beschriftung für Reiter n, sondern auch, welche Indices gültig sind, nämlich
	 * 0 <= n < {@link #VALID_INDICES}.length.
	 */
	public static final String[] VALID_INDICES = { "Vorgabe", "Soll", "Default", "Ist" };

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.editor.FormEditor#addPages()
	 */
	@Override
	protected void addPages() {
		final IParameterFormPageFactory selectedFactory = getEditorInput().getSelectedFactory();
		final Shell shell = PlatformUI.getWorkbench().getDisplay().getActiveShell();
		Cursor waitCursor = null;
		try {
			if (null != shell) {
				waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
				shell.setCursor(waitCursor);
			}
			int index;
			for (index = 0; index < selectedFactory.getNumFormPages(this); ++index) {
				addPage(index, selectedFactory.createFormPage(this, index));
			}
		} catch (final PartInitException e) {
			ParamPlugin.getDefault().getLogger().error("Parametereditor konnte nicht geöffnet werden", e);
		} finally {
			if (null != shell) {
				shell.setCursor(null);
			}
			if (null != waitCursor) {
				waitCursor.dispose();
				waitCursor = null;
			}
		}
		setPartName(getEditorInput().getName());
		firePropertyChange(org.eclipse.ui.IWorkbenchPart.PROP_TITLE);

		// Online-Hilfe hinzufügen
		final String helpContextId = selectedFactory.getHelpContextId();
		PlatformUI.getWorkbench().getHelpSystem().setHelp(getContainer(), helpContextId);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.forms.editor.FormEditor#createToolkit(org.eclipse.swt.
	 * widgets.Display)
	 */
	@Override
	protected FormToolkit createToolkit(final Display display) {
		final int formStyleBits;
		if ((getEditorInput() != null) && (getEditorInput().getSelectedFactory() != null)) {
			formStyleBits = getEditorInput().getSelectedFactory().getFormStyleBits();
		} else {
			formStyleBits = SWT.H_SCROLL | SWT.V_SCROLL;
		}
		return new ParameterFormToolkit(display, formStyleBits);
	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
		final IFormPage editedFormPage = (IFormPage) pages.get(ParameterEditor.EDITED_INDEX);
		for (final Object page : pages) {
			if (page instanceof IFormPage) {
				final IFormPage formPage = (IFormPage) page;
				if (editedFormPage == formPage) {
					handleSave(monitor, formPage);
				} else {
					handleStaleAspects(editedFormPage, formPage);
				}
			}
		}
	}

	@Override
	public void doSaveAs() {
		throw new UnsupportedOperationException(
				"Das Abspeichern von Parametern an anderen Objekten wird momentan nicht untersützt.");
	}

	/**
	 * {@inheritDoc}
	 *
	 * Das suppress warnings ist ok. Die Methodensignatur kann nicht korrigiert
	 * werden, da die Schnittstelle von Eclipse kommt.
	 */
	@Override
	public Object getAdapter(final Class adapter) {
		ILegende legende = null;
		if (ILegende.class.isAssignableFrom(adapter)) {
			if (-1 == getActivePage()) {
				return null;
			}
			final IFormPage activeFormPage = (IFormPage) pages.get(getActivePage());
			if (activeFormPage instanceof ILegende) {
				legende = (ILegende) activeFormPage;
			} else {
				Object o = ((IAdaptable) activeFormPage).getAdapter(ILegende.class);
				if (o instanceof ILegende) {
					legende = (ILegende) o;
				} else {
					o = Platform.getAdapterManager().getAdapter(activeFormPage, ILegende.class);
					if (o instanceof ILegende) {
						legende = (ILegende) o;
					}
				}
			}
			if (null == legende) {
				legende = new ParameterEditorLegendenAbschnitt(getEditorInput());
			}
			return legende;
		}
		return super.getAdapter(adapter);
	}

	/**
	 * Liefert einen Default-Header für einen Parameter-Editor, z.B. zum Drucken. Er
	 * benennt Attributgruppe, Aspekt und Simulationsvariante sowie die Anzahl der
	 * im Editor befindlichen Objekte.
	 *
	 * @param oneLine
	 *            true - alle Informationen auf eine Zeile.
	 * @return der Text
	 */
	public String getDefaultHeader(final boolean oneLine) {
		final char lineBreak;
		if (oneLine) {
			lineBreak = ' ';
		} else {
			lineBreak = '\n';
		}
		final Parameter[] parameters = getEditorInput().getParameters();
		final ParameterInfo info = parameters[0].getInfo();
		final StringBuilder b = new StringBuilder("Parameter der Attributgruppe '");
		b.append(info.getAtg().toString());
		b.append("'");
		b.append(lineBreak);
		b.append("Aspekt: ");
		final int activePageIndex = getActivePage();
		if ((activePageIndex < 0) || (activePageIndex >= ParameterEditor.VALID_INDICES.length)) {
			b.append("Unbekannt");
		} else {
			b.append(ParameterEditor.VALID_INDICES[activePageIndex]);
		}
		b.append(lineBreak);
		b.append("Simulationsvariante ");
		b.append(info.getSim());
		b.append(" - ");
		b.append(RahmenwerkService.getService().getArtDesParametersatzesProvider()
				.getSimulationsVarianteString(info.getSim()));
		if (1 < parameters.length) {
			b.append(lineBreak);
			b.append("Insgesamt ");
			b.append(parameters.length);
			b.append(" Objekte");
		}
		return b.toString();
	}

	@Override
	public PagePrint getDruckAuftrag() {
		PagePrint druckAuftrag = null;
		if (-1 == getActivePage()) {
			return null;
		}
		final IFormPage activeFormPage = (IFormPage) pages.get(getActivePage());
		if (activeFormPage instanceof RwPrintable) {
			druckAuftrag = ((RwPrintable) activeFormPage).getDruckAuftrag();
		}
		return druckAuftrag;
	}

	@Override
	public ParameterEditorInput getEditorInput() {
		return (ParameterEditorInput) super.getEditorInput();
	}

	@Override
	public String getTitel() {
		final IFormPage activeFormPage = (IFormPage) pages.get(getActivePage());
		if (activeFormPage instanceof RwPrintable) {
			final String specificTitel = ((RwPrintable) activeFormPage).getTitel();
			if (null != specificTitel) {
				return specificTitel;
			}
		}
		return getDefaultHeader(true);
	}

	/**
	 * Die eigentliche Speichern-Routine.
	 *
	 * @param monitor
	 *            Progress-Monitor für Eclipse GUI
	 * @param formPage
	 *            die zu speichernde Seite
	 */
	private void handleSave(final IProgressMonitor monitor, final IFormPage formPage) {
		formPage.doSave(monitor);
		if (!formPage.isDirty()) {
			final AbstractParamPluginContentProvider contentProvider = getEditorInput().getContentProvider();
			if (contentProvider != null) {
				final Display display = PlatformUI.getWorkbench().getDisplay();
				final Shell shell = display.getActiveShell();
				Cursor waitCursor = null;
				try {
					if (null != shell) {
						waitCursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
						shell.setCursor(waitCursor);
					}
					contentProvider.reloadCurrentInput(true);
				} catch (final ParameterClientException e) {
					ParamPlugin.getDefault().getLogger().error("Fehler beim Aktualisieren der Ansicht", e);
				} finally {
					if (null != shell) {
						shell.setCursor(null);
					}
					if (null != waitCursor) {
						waitCursor.dispose();
						waitCursor = null;
					}
				}
			}
		}
	}

	/**
	 * Nachbehandlung nach erfolgreichem Speichern: Die nun aktuellen Sollparameter
	 * abfragen und in die Soll-Reiter laden. War dieser bereits offen, reicht es,
	 * an diesem markStale aufzurufen - dort erfolgt dann der Abruf. Wenn die Seite
	 * noch nie geöffnet war, muss man hier die Sollparameter abrufen, da aufgrund
	 * des lazy loading vom Eclipse das ManagedForm null ist.
	 *
	 * @param editedFormPage
	 *            die bearbeitete Seite.
	 * @param formPage
	 *            die zu aktualisierende Seite.
	 */
	private void handleStaleAspects(final IFormPage editedFormPage, final IFormPage formPage) {
		if (!editedFormPage.isDirty()) {
			if (null != formPage.getManagedForm()) {
				for (final IFormPart formPart : formPage.getManagedForm().getParts()) {
					if (formPart instanceof AbstractFormPart) {
						((AbstractFormPart) formPart).markStale();
					}
				}
			} else if (1 == formPage.getIndex()) {
				try {
					getEditorInput().sollParameterAbfragen();
					getEditorInput().copySollToVorgabe(null);
				} catch (final ParameterClientException e) {
					ParamPlugin.getDefault().getLogger()
							.error("Fehler beim Abfragen der " + "aktualisierten Sollparameter", e);
				}
			}
		}
	}

	@Override
	public boolean isSaveAsAllowed() {
		/*
		 * Das Abspeichern von Parametern an anderen Objekten wird momentan nicht
		 * untersützt.
		 */
		return false;
	}

	/**
	 * Neuladen von individuellen Parametern.
	 *
	 * @param parameters
	 *            die Liste von Parametern, deren Werte geladen werden sollen
	 * @param force
	 *            true - Sofort neu laden, false - Erst in einem Userdialog fragen
	 */
	public void selectiveReload(final Parameter[] parameters, final boolean force) {
		final List<Parameter> availableParameters = new ArrayList<>();
		for (final Parameter p : parameters) {
			for (final Parameter editedParameter : getEditorInput().getParameters()) {
				if (editedParameter.getInfo().equals(p.getInfo())) {
					availableParameters.add(p);
					break;
				}
			}
		}

		// Zunächst mal die aktuellen Sollparameter neu einlesen
		try {
			getEditorInput().sollParameterAbfragen();
		} catch (final ParameterClientException e) {
			ParamPlugin.getDefault().getLogger().error("Fehler beim Abfragen der " + "aktualisierten Sollparameter", e);
		}

		boolean doReload;
		if (force) {
			doReload = true;
		} else {
			final StringBuilder b = new StringBuilder(
					"Möchten Sie die folgenden " + availableParameters.size() + " Parameter neu laden?\n");
			final int maxObjects = 40;
			int loop = 0;
			for (final Parameter parameter : availableParameters) {
				b.append('\n');
				b.append(parameter.getObjekt().toString());
				++loop;
				if (loop >= maxObjects) {
					b.append(", ...");
					break;
				}
			}
			doReload = MessageDialog.openQuestion(null, "Parameter neu laden?", b.toString());
		}
		if (doReload) {
			getEditorInput().copySollToVorgabe(availableParameters);
		}

		final IFormPage vorgabeFormPage = (IFormPage) pages.get(ParameterEditor.EDITED_INDEX);
		for (final IFormPart formPart : vorgabeFormPage.getManagedForm().getParts()) {
			if (formPart instanceof AbstractFormPart) {
				((AbstractFormPart) formPart).refresh();
			}
		}
		getEditorInput().getSelectedParameters().clear();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.part.MultiPageEditorPart#setFocus()
	 */
	@Override
	public void setFocus() {
		getContainer().setFocus();
	}
}
