/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors;

import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;

import de.bsvrz.buv.plugin.param.ParamSharedImage;

/**
 * Dialog, der eine Menge von Editoren für einen Parameter anzeigt und den
 * Nutzer auswählen läßt, welcher benutzt werden soll.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class ParameterEditorAuswahlDialog extends TitleAreaDialog {

	/**
	 * die Liste verfügbarer Editoren.
	 */
	private final List<IParameterFormPageFactory> factories;

	/**
	 * Die ausgewählte Factory.
	 */
	private IParameterFormPageFactory selectedFactory;

	/**
	 * Konstruktor setzt Member und prüft Argumente.
	 *
	 * @param parentShell
	 *            Eltern-Shell
	 * @param factories
	 *            die Liste verfügbarer Editoren
	 * @param preSelectedFactory
	 *            Editor, der vorzuselektieren ist.
	 */
	public ParameterEditorAuswahlDialog(final Shell parentShell, final List<IParameterFormPageFactory> factories,
			final IParameterFormPageFactory preSelectedFactory) {
		super(parentShell);
		this.factories = factories;
		Assert.isNotNull(factories);
		Assert.isTrue(factories.size() > 1);
		selectedFactory = preSelectedFactory;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.dialogs.TitleAreaDialog#createDialogArea(org.eclipse
	 * .swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(final Composite parent) {
		parent.getShell().setText("Auswahl des Parameter-Editors");
		parent.getShell().setImage(ParamSharedImage.QUESTION_MARK.getImage());
		setTitle("Es sind mehrere Parameter-Editoren verfügbar");
		setMessage("Bitte wählen Sie den Editor aus!");

		final Composite panel = new Composite(parent, SWT.NONE);
		panel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		final GridLayout gl = new GridLayout(1, false);
		gl.marginLeft = 10;
		gl.marginRight = 10;
		gl.verticalSpacing = 10;
		panel.setLayout(gl);

		GridData gd;
		final Group factoriesGroup = new Group(panel, SWT.NONE);
		final GridLayout factoriesGroupLayout = new GridLayout(1, false);
		factoriesGroup.setLayout(factoriesGroupLayout);
		factoriesGroup.setText("Verfügbare Editoren");
		factoriesGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		int loop;
		for (loop = 0; loop < factories.size(); ++loop) {
			final IParameterFormPageFactory factory = factories.get(loop);
			final ParameterEditorInfo factoryInfo = factory.getEditorInfo();
			final Button button = new Button(factoriesGroup, SWT.RADIO | SWT.WRAP);
			button.setData(factory);
			gd = new GridData(SWT.LEFT, SWT.TOP, false, false);
			gd.horizontalSpan = 1;
			gd.horizontalAlignment = SWT.LEFT;
			button.setLayoutData(gd);
			button.setSelection(factory == selectedFactory);
			button.setEnabled(true);
			button.setText(factoryInfo.getName() + " - " + factoryInfo.getManufacturer());
			button.addSelectionListener(new SelectionListener() {

				@Override
				public void widgetSelected(final SelectionEvent e) {
					final Button b = (Button) e.widget;
					if (b.getSelection()) {
						ParameterEditorAuswahlDialog.this.selectedFactory = (IParameterFormPageFactory) b.getData();
					}
				}

				@Override
				public void widgetDefaultSelected(final SelectionEvent e) {
					widgetSelected(e);
				}
			});
		}

		return panel;
	}

	/**
	 * Liefert die momentan ausgewählte Factory.
	 *
	 * @return die Factory
	 */
	public IParameterFormPageFactory getSelectedFactory() {
		return selectedFactory;
	}

}
