/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

import de.bsvrz.buv.plugin.param.Zeichenketten;
import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.lib.ParameterManagerAllgemein;
import de.bsvrz.buv.plugin.param.provider.AbstractParamPluginContentProvider;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.ParameterInfo;
import de.bsvrz.puk.param.lib.ParameterManager;
import de.bsvrz.puk.param.lib.daten.DataWithTime;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Input-Objekt für einen {@link ParameterEditor}.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public class ParameterEditorInput implements IEditorInput {

	/**
	 * Debug-Logger für alle ParaEditComposite-Instanzen. Ein Logger pro Instanz
	 * lohnt nicht.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Farbzuordnung für die Einfärbung der eidtierten Attribute.
	 */
	private static final Map<List<Boolean>, Color> COLORS = new LinkedHashMap<>();

	/**
	 * Die Vorgabe-Parameter.
	 */
	private final Parameter[] parameters;

	/**
	 * Die Soll-Parameter.
	 */
	private final Parameter[] sollParameters;

	/**
	 * Die Default-Parameter.
	 */
	private final Parameter[] defaultParameters;

	/**
	 * Die Ist-Parameter.
	 */
	private final Parameter[] istParameters;

	/**
	 * Aktuell ausgewählte Parameter (relevant für die nächste Aktion).
	 */
	private final Set<Integer> selectedParameters = new HashSet<>();

	/**
	 * der Contentprovider des
	 * {@link de.bsvrz.buv.plugin.param.views.AbstractParamPluginView}, aus dem
	 * dieser Editor heraus geöffnet wurde. Kann null sein.
	 */
	private final AbstractParamPluginContentProvider contentProvider;

	/**
	 * Die Art und Weise, wie die Änderung eines Attributs auf das korrespondierende
	 * Attribut auf die anderenn Parameter übertragen werden soll.
	 */
	private AttributePropagationMode attributePropagationMode = AttributePropagationMode.ONLY_EDITED;

	/**
	 * Bestimmt, ob beim Spaichern alle oder nur die geänderten Parameter
	 * geschrieben werden sollen.
	 */
	private ParameterEditorSaveMode saveMode = ParameterEditorSaveMode.ONLY_MODIFIED;

	/**
	 * Der zu benutzende Editor (kommt aus einem anderen Plugin).
	 */
	private IParameterFormPageFactory selectedFactory;

	/**
	 * Bestimmt, welchen dirty-State der Editor initial hat.
	 */
	private InitialDirtyStateMode initialDirtyStateMode = InitialDirtyStateMode.ONLY_ON_DIFFERENCES_TO_SOLL;

	static {
		/* Reihenfolge: soll ist default */
		/*
		 * false: Es besteht ein Unterschied, true - Es besteht KEIN Unterschied
		 */
		// GELB
		ParameterEditorInput.setVorgabeColor(Arrays.asList(false, true, true),
				new Color(Display.getDefault(), 255, 255, 0));
		// HELLBLAU
		ParameterEditorInput.setVorgabeColor(Arrays.asList(true, false, true),
				new Color(Display.getDefault(), 89, 131, 255));
		// ROT
		ParameterEditorInput.setVorgabeColor(Arrays.asList(true, true, false),
				new Color(Display.getDefault(), 255, 30, 30));
		// ORANGE
		ParameterEditorInput.setVorgabeColor(Arrays.asList(false, true, false),
				new Color(Display.getDefault(), 255, 128, 0));
		// VIOLETT
		ParameterEditorInput.setVorgabeColor(Arrays.asList(true, false, false),
				new Color(Display.getDefault(), 244, 0, 244));
		// GRÜN
		ParameterEditorInput.setVorgabeColor(Arrays.asList(false, false, true),
				new Color(Display.getDefault(), 50, 230, 30));
		// BRAUN
		ParameterEditorInput.setVorgabeColor(Arrays.asList(false, false, false),
				new Color(Display.getDefault(), 225, 80, 0));
	}

	/**
	 * Liefert den Wert eines DAF-Datums als Textwert ohne Zusätze wie Maßeinheiten
	 * o.ä.
	 *
	 * @param plainData DAF-Datum
	 * @return der Wert
	 */
	public static String getPlainValueFromPlainData(final Data plainData) {
		String value = DataValueTextArrayMemberCombinator.IDENTIFIER_NICHT_DEFINIERT;
		final String plainDataValueStr = plainData.valueToString();
		if ((null != plainDataValueStr)
				&& !DataValueTextArrayMemberCombinator.IDENTIFIER_NICHT_DEFINIERT.equals(plainDataValueStr)) {
			final AttributeType attributeType = plainData.getAttributeType();
			if (attributeType instanceof StringAttributeType) {
				value = plainData.asTextValue().getValueText();
			} else if (attributeType instanceof IntegerAttributeType) {
				value = plainData.asUnscaledValue().getValueText();
			} else if (attributeType instanceof TimeAttributeType) {
				value = plainData.asTimeValue().getValueText();
			} else if (attributeType instanceof DoubleAttributeType) {
				value = plainData.asUnscaledValue().getValueText();
			} else if (attributeType instanceof ReferenceAttributeType) {
				final SystemObject o = plainData.asReferenceValue().getSystemObject();
				if (null != o) {
					value = o.getPidOrId();
				} else {
					value = "undefiniert";
				}
			}
		}
		return value;
	}

	/**
	 * Liefert die Farbe zu einer Kombination aus Unterschieden zwischen den
	 * einzelnen Aspekten.
	 *
	 * @param key Kombination aus Unterschieden zu Aspekten
	 *
	 * @return die Farbe oder <code>null</code>, wenn keine besondere Farbe
	 *         definiert ist
	 */
	public static Color getVorgabeColor(final List<Boolean> key) {
		return ParameterEditorInput.COLORS.get(key);
	}

	/**
	 * Weist einer Kombination aus Unterschieden zu Aspekten eine Farbe zu. Das
	 * vorherige Mapping für key wird, falls es existierte, gelöscht und die Farbe
	 * freigegeben (Color#dispose()). Deshalb sollte man dieser Methode keine
	 * SystemColors übergeben.
	 *
	 * @param key   Kombination aus Unterschieden zu Aspekten. Diese muss genau 3
	 *              Elemente enthalten, die Reihenfolge der Aspekte ist: soll ist
	 *              default
	 * @param color die Farbe. Wenn man null übergibt, wird das Mapping für den
	 *              übergebenen key gelöscht.
	 */
	public static void setVorgabeColor(final List<Boolean> key, final Color color) {
		Assert.isNotNull(key, "Keine Kombination aus Unterschieden zu Aspekten übergeben");
		Assert.isTrue(3 == key.size(),
				"Übergebene Kombination aus Unterschieden zu Aspekten muss genau 3 Elemente enthalten");
		final Color old = ParameterEditorInput.COLORS.get(key);
		if (null != old) {
			old.dispose();
			if (null == color) {
				ParameterEditorInput.COLORS.remove(key);
			}
		} else {
			if (null != color) {
				ParameterEditorInput.COLORS.put(key, color);
			}
		}
	}

	/**
	 * Konstruktor.
	 *
	 * @param parameters      die Ausgangswerte der zu editierenden Parameter
	 * @param contentProvider ein Contentprovider
	 * @throws ParameterClientException Falls beim Abfragen der Soll-, Ist- oder
	 *                                  Defaultparameter ein Fehler auftrat.
	 */
	public ParameterEditorInput(final Parameter[] parameters, final AbstractParamPluginContentProvider contentProvider)
			throws ParameterClientException {

		this.parameters = ParameterManagerAllgemein.getFilledParameterArray(parameters);
		this.contentProvider = contentProvider;
		final String message = "Es müssen Parameter zur Bearbeitung übergeben werden";
		Assert.isNotNull(parameters, message);
		Assert.isTrue(parameters.length > 0, message);
		Assert.isNotNull(parameters[0], message);

		int loop;
		final ParameterInfo[] infos = new ParameterInfo[parameters.length];
		for (loop = 0; loop < parameters.length; ++loop) {
			infos[loop] = new ParameterInfo(parameters[loop].getObjekt(), parameters[loop].getTyp(),
					parameters[loop].getAtg(), parameters[loop].getSim());
		}
		final ParameterManager hierarchie = MethodenBibliothek
				.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav());
		sollParameters = hierarchie.getParameter(infos);
		defaultParameters = ParameterManagerAllgemein.getInstanz().getDefaultParameter(infos);
		istParameters = ParameterManagerAllgemein.getInstanz().getIstParameter(infos);
	}

	/**
	 * Kopiert die Daten der Sollparameter auf die korrespondierenden
	 * Vorgabe-Parameter. <strong> Achtung:</strong> Es wird keine implizite Abfrage
	 * der Sollparameter vorher durchgeführt.
	 *
	 * @param availableParameters Liste mit zu kopierenden Parametern. null heißt:
	 *                            Alle.
	 */
	public void copySollToVorgabe(final List<Parameter> availableParameters) {
		int loop;
		selectedParameters.clear();

		for (loop = 0; loop < parameters.length; ++loop) {
			boolean doCopy = false;
			if (null == availableParameters) {
				doCopy = true;
			} else {
				for (final Parameter availableParameter : availableParameters) {
					if (availableParameter.getInfo().equals(parameters[loop].getInfo())) {
						doCopy = true;
						break;
					}
				}
			}
			if (doCopy) {
				if (null == sollParameters[loop].getData()) {
					/* Mantis#1511 */
					ParamPlugin.getDefault().getLogger().warning("Parameter " + sollParameters[loop].getInfo()
							+ Zeichenketten.PLUGIN_PARAM_HINWEIS_PARAM_PARAM_PRUEFEN);
				} else {
					parameters[loop]
							.setDataWithTime(new DataWithTime(sollParameters[loop].getData().createModifiableCopy(),
									sollParameters[loop].getZeitpunkt()));
					selectedParameters.add(loop);
				}
			}
		}
	}

	/**
	 * Bestimmt die in Frage kommenden Editoren durch Auswertung des Extension
	 * Points de.bsvrz.buv.plugin.param.editoren.
	 *
	 * @return die Liste. Kann leer sein.
	 */
	public List<IParameterFormPageFactory> determineApplicableEditors() {
		final IExtensionRegistry registry = Platform.getExtensionRegistry();
		final IExtensionPoint point = registry.getExtensionPoint(ParamPlugin.PLUGIN_ID + ".editoren");
		final IConfigurationElement[] editors = point.getConfigurationElements();
		final List<IParameterFormPageFactory> factories = new ArrayList<>();
		for (final IConfigurationElement editor : editors) {
			final String editorId = editor.getAttribute("editorId");
			if ((editorId != null) && editorId.equals(ParameterEditor.EDITOR_ID)) {
				final IConfigurationElement[] configurationElements = editor.getChildren("parameterFormPageFactory");
				for (final IConfigurationElement page : configurationElements) {
					try {
						final IParameterFormPageFactory factory = (IParameterFormPageFactory) page
								.createExecutableExtension("class");
						if (factory.canEdit(this)) {
							factories.add(factory);
						}
					} catch (final InvalidRegistryObjectException e) {
						ParamPlugin.getDefault().getLogger().error(
								"Fehler beim Auslesen der Erweiterungspunkte" + " für die Parameter-Editoren", e);
					} catch (final CoreException e) {
						ParamPlugin.getDefault().getLogger().error(
								"Fehler beim Auslesen der Erweiterungspunkte" + " für die Parameter-Editoren", e);
					}
				}
			}
		}
		return factories;
	}

	@Override
	public boolean equals(final Object obj) {
		boolean result = false;
		if (this == obj) {
			result = true;
		} else if (obj instanceof ParameterEditorInput) {
			final ParameterEditorInput otherInput = (ParameterEditorInput) obj;
			if (parameters.length == otherInput.getParameters().length) {
				final Set<ParameterInfo> myInfos = new HashSet<>();
				final Set<ParameterInfo> otherInfos = new HashSet<>();
				int loop;
				for (loop = 0; loop < parameters.length; ++loop) {
					myInfos.add(parameters[loop].getInfo());
					otherInfos.add(otherInput.getParameters()[loop].getInfo());
				}
				result = myInfos.equals(otherInfos);
			}
		}
		return result;
	}

	@Override
	public boolean exists() {
		boolean result = true;
		for (final Parameter p : parameters) {
			if (!p.getObjekt().isValid()) {
				result = false;
				break;
			}
		}
		return result;
	}

	/**
	 * {@inheritDoc}
	 *
	 * Das suppress warnings ist ok. Die Methodensignatur kann nicht korrigiert
	 * werden, da die Schnittstelle von Eclipse kommt.
	 *
	 * Später: public <T> T getAdapter(Class<T> adapter)
	 */
	@Override
	public Object getAdapter(final Class adapter) {
		if (AttributeGroup.class.equals(adapter)) {
			return parameters[0].getAtg();
		}
		return null;
	}

	/**
	 * Bestimmt den Modus, wie sich Änderungen eins Attributs auf die
	 * korrespondierenden Attribute der anderen im Editor befindlichen Parameter
	 * auswirken sollen.
	 *
	 * @return der Modus
	 */
	public AttributePropagationMode getAttributePropagationMode() {
		return attributePropagationMode;
	}

	/**
	 * Sucht aus einer Liste von Parametereditoren denjenigen heraus, der einen
	 * gegebenen Editorinput am besten bearbeiten kann.
	 *
	 * @param factories die Liste von Parametereditoren
	 * @return der Index in der Liste, welcher den besten Editor liefert
	 */
	public int getBestFactory(final List<IParameterFormPageFactory> factories) {
		if ((null == factories) || (factories.size() <= 0)) {
			return -1;
		}
		int bestFactoryIdx = 0;
		int adequacy = factories.get(bestFactoryIdx).getAdequacy(this);
		int loop;
		for (loop = 1; loop < factories.size(); ++loop) {
			final IParameterFormPageFactory currentFactory = factories.get(loop);
			final int currentAadequacy = currentFactory.getAdequacy(this);
			if (currentAadequacy > adequacy) {
				adequacy = currentAadequacy;
				bestFactoryIdx = loop;
			}
		}
		return bestFactoryIdx;
	}

	/**
	 * Liefert den Contentprovider des
	 * {@link de.bsvrz.buv.plugin.param.views.AbstractParamPluginView}, aus dem
	 * dieser Editor heraus geöffnet wurde.
	 *
	 * @return der contentProvider oder null
	 */
	public AbstractParamPluginContentProvider getContentProvider() {
		return contentProvider;
	}

	/**
	 * Liefert die Default-Parameter für die ursprünglich übergebenen
	 * defaultParameters. Die Simulationsvariante bleibt hierbei unberücksichtigt.
	 *
	 * @return die Default-Parameter
	 */
	public Parameter[] getDefaultParameters() {
		return defaultParameters;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		// leer
		return null;
	}

	/**
	 * Bestimmt, welchen dirty-State der Editor initial hat.
	 *
	 * @return der Modus
	 */
	public InitialDirtyStateMode getInitialDirtyStateMode() {
		return initialDirtyStateMode;
	}

	/**
	 * Liefert zuletzt abgefragten Ist-Parameter für die ursprünglich übergebenen
	 * istParameters.
	 *
	 * @see #istParameterAbfragen()
	 *
	 * @return die Ist-Parameter
	 */
	public Parameter[] getIstParameters() {
		return istParameters;
	}

	@Override
	public String getName() {
		return parameters[0].getAtg().getName();
	}

	/**
	 * Liefert die Parameter, die in diesem Input-Objekt verkapselt werden.
	 *
	 * @return the parameters
	 */
	public Parameter[] getParameters() {
		return parameters;
	}

	@Override
	public IPersistableElement getPersistable() {
		// leer, wir unterstützen Persistenz über Sitzungen hinweg nicht
		return null;
	}

	/**
	 * Bestimmt die Art, die Menge der zu schreibenden Parameter beim
	 * Speichern-Vorgang zu bestimmen.
	 *
	 * @return die Art
	 */
	public ParameterEditorSaveMode getSaveMode() {
		return saveMode;
	}

	/**
	 * Liefert den zu benutzenden Editor (kommt aus einem anderen Plugin).
	 *
	 * @return der Editor
	 */
	public IParameterFormPageFactory getSelectedFactory() {
		return selectedFactory;
	}

	/**
	 * Liefert die Indices aktuell ausgewählten Parameter (relevant für die nächste
	 * Aktion). Kann leer, aber nicht null sein.
	 *
	 * @return die Indices
	 */
	public Set<Integer> getSelectedParameters() {
		return selectedParameters;
	}

	/**
	 * Liefert einen Kurztext, der das Objekt besser beschreibt als toString.
	 *
	 * @return der Text
	 */
	public String getShortText() {
		final StringBuilder b = new StringBuilder("Auswahl aus ");
		b.append(parameters.length);
		b.append(" Parametern an Attributgruppe ");
		b.append(getName());
		b.append(".\n ");
		final int maxCount = 10;
		int loop = 0;
		boolean ellipsis = false;
		for (final Parameter p : parameters) {
			b.append(p.getObjekt().toString());
			b.append("\n ");
			++loop;
			if (loop > maxCount) {
				ellipsis = true;
				break;
			}
		}
		if (ellipsis) {
			b.append(" ...");
		}
		return b.toString();
	}

	/**
	 * Liefert die aktuellen Soll-Parameter für die ursprünglich übergebenen
	 * sollParameters, in der korrespondierenden Simulationsvariante.
	 *
	 * @return die Sollparameter
	 */
	public Parameter[] getSollParameters() {
		return sollParameters;
	}

	@Override
	public String getToolTipText() {
		return "Bearbeiten des Parameters '" + getName() + "'";
	}

	/**
	 * Bestimmt die Farbgebung eines einzelnen Attributs gemäß TAnf-137-12.
	 *
	 * @param info                 Identifizierung des Parameters
	 * @param pathToPlainAttribute Pfad zu einem einfachen Attribut, welches gerade
	 *                             eingefärbt werden soll
	 * @param value                Wert ohne Einheiten oder Zusätze (Bei Zahlen:
	 *                             UnscaledValue. Bei Zuständen: Der Name.
	 *                             Objektreferenzen: Pid.
	 *
	 * @return die Farbe oder <code>null</code> für Standard
	 */
	public Color getVorgabeColor(final ParameterInfo info, final String pathToPlainAttribute, final String value) {
		int loop;
		int aspektLoop;
		Color color = null;
		if ((null != pathToPlainAttribute) && (pathToPlainAttribute.length() > 0)) {
			final Parameter[] vergleichsParameter = new Parameter[4];
			final String[] plainValues = new String[4];
			plainValues[0] = value;
			for (loop = 0; loop < parameters.length; ++loop) {
				if (parameters[loop].getInfo().equals(info)) {
					vergleichsParameter[0] = parameters[loop];
					break;
				}
			}
			if (loop < parameters.length) {
				vergleichsParameter[1] = sollParameters[loop];
				vergleichsParameter[2] = istParameters[loop];
				vergleichsParameter[3] = defaultParameters[loop];
				final String[] splitPath = pathToPlainAttribute.split("\\.");
				for (aspektLoop = 1; aspektLoop < 4; ++aspektLoop) {
					Data d = vergleichsParameter[aspektLoop].getData();
					plainValues[aspektLoop] = null;
					if (null == d) {
						continue;
					}
					for (loop = 0; loop < splitPath.length; ++loop) {
						if (d.isList()) {
							d = d.getItem(splitPath[loop]);
						} else if (d.isArray()) {
							final Array array = d.asArray();
							if (Zeichenketten.EMPTY_INDICATOR.equals(splitPath[loop])) {
								if (array.getLength() <= 0) {
									plainValues[aspektLoop] = plainValues[0];
									break;
								}
								d = array.getItem(0);
							} else {
								try {
									final int index = Integer.parseInt(splitPath[loop]);
									if (array.getLength() <= index) {
										break;
									}
									d = array.getItem(index);
								} catch (final NumberFormatException e) {
									ParameterEditorInput.DEBUG.warning("Path-Data-Mismatch", e);
									break;
								}
							}
						}
						if (d.isPlain()) {
							plainValues[aspektLoop] = ParameterEditorInput.getPlainValueFromPlainData(d);
							break;
						}
					}
				}
				final DataValueTextArrayMemberCombinator kombinator = new DataValueTextArrayMemberCombinator(
						plainValues, true);
				kombinator.run();
				final List<Boolean> outputs = kombinator.getOutputs();
				color = ParameterEditorInput.COLORS.get(outputs);
			}
		}
		return color;
	}

	@Override
	public int hashCode() {
		return parameters[0].getObjekt().hashCode();
	}

	/**
	 * Abfrage der Ist-Parameter über die verallgemeinerte Methodenbibliothek.
	 */
	public void istParameterAbfragen() {
		int loop;
		final ParameterInfo[] infos = new ParameterInfo[parameters.length];
		for (loop = 0; loop < parameters.length; ++loop) {
			infos[loop] = new ParameterInfo(parameters[loop].getObjekt(), parameters[loop].getTyp(),
					parameters[loop].getAtg(), parameters[loop].getSim());
		}
		final Parameter[] neueIstParameters = ParameterManagerAllgemein.getInstanz().getIstParameter(infos);
		for (loop = 0; loop < neueIstParameters.length; ++loop) {
			istParameters[loop] = neueIstParameters[loop];
		}
	}

	/**
	 * Wählt alle Parameter aus.
	 */
	public void selectAll() {
		int loop;
		selectedParameters.clear();
		for (loop = 0; loop < parameters.length; ++loop) {
			selectedParameters.add(loop);
		}
	}

	/**
	 * Wählt diejenigen (Vorgabe-)Parameter aus, deren Werte sich vom
	 * korrespondierenden Sollparameter unterscheiden.
	 *
	 * Achtung: Da die DAF-Data-Objekte kein equals implementieren, benutzen wir
	 * toString und vergleichen diese.
	 */
	public void selectVorgabeParameterDifferringToSoll() {
		int loop;
		selectedParameters.clear();
		for (loop = 0; loop < parameters.length; ++loop) {
			final Data vorgabeData = parameters[loop].getData();
			final Data sollData = sollParameters[loop].getData();
			if (vorgabeData == sollData) {
				// This covers both being null as well
				continue;
			}
			if (((vorgabeData != null) && (sollData == null)) || ((vorgabeData == null) && (sollData != null))) {
				selectedParameters.add(loop);
				continue;
			}
			if ((vorgabeData != null) && (sollData != null)) {
				final String vorgabeDataString = vorgabeData.toString();
				final String sollDataString = sollData.toString();
				if (!vorgabeDataString.equals(sollDataString)) {
					selectedParameters.add(loop);
				}
				// Zur Information: Diesen Algorithmus hatte ich temporär im
				// Einsatz. Er vergleicht zwei Datensätze, wobei die
				// Urlasserinformationen ignoriert werden, also sowas wie
				// Data.equalsIgnoreUrlasser().
				// final AttributeGroup atg = parameters[loop].getAtg();
				// for (Attribute a : atg.getAttributes()) {
				// System.out.print("Bearbeite Attribut " + a.getName());
				// if ("atl.urlasser".equals(a.getAttributeType().getPid())) {
				// System.out.println(" ... übersprungen");
				// continue;
				// }
				// final Data vorgabeItem = vorgabeData.getItem(a.getName());
				// final Data sollItem = sollData.getItem(a.getName());
				// final String vorgabeDataString = vorgabeItem.toString();
				// final String sollDataString = sollItem.toString();
				// if (!vorgabeDataString.equals(sollDataString)) {
				// System.out.println(" ... UNGLEICH: Vorgabe="
				// + vorgabeDataString + ", Soll="
				// + sollDataString);
				// selectedParameters.add(loop);
				// break;
				// }
				// System.out.println(" ... ok");
				// }
			}
		}
	}

	/**
	 * Setzt den Modus, wie sich Änderungen eins Attributs auf die
	 * korrespondierenden Attribute der anderen im Editor befindlichen Parameter
	 * auswirken sollen.
	 *
	 * @param attributePropagationMode der neue Modus
	 */
	public void setAttributePropagationMode(final AttributePropagationMode attributePropagationMode) {
		this.attributePropagationMode = attributePropagationMode;
	}

	/**
	 * Setzt, welchen dirty-State der Editor initial hat.
	 *
	 * @param initialDirtyStateMode der Modus
	 */
	public void setInitialDirtyStateMode(final InitialDirtyStateMode initialDirtyStateMode) {
		this.initialDirtyStateMode = initialDirtyStateMode;
	}

	/**
	 * Setzt die Art, die Menge der zu schreibenden Parameter beim Speichern-Vorgang
	 * zu bestimmen.
	 *
	 * @param saveMode die Art
	 */
	public void setSaveMode(final ParameterEditorSaveMode saveMode) {
		this.saveMode = saveMode;
	}

	/**
	 * Setzt den zu benutzenden Editor (kommt aus einem anderen Plugin).
	 *
	 * @param selectedFactory der Editor
	 */
	public void setSelectedFactory(final IParameterFormPageFactory selectedFactory) {
		this.selectedFactory = selectedFactory;
	}

	/**
	 * Abfrage der Soll-Parameter über die Methodenbibliothek.
	 *
	 * @throws ParameterClientException Fehlermeldung der Methodenbibliothek
	 */
	public void sollParameterAbfragen() throws ParameterClientException {
		int loop;
		final ParameterInfo[] infos = new ParameterInfo[parameters.length];
		for (loop = 0; loop < parameters.length; ++loop) {
			infos[loop] = new ParameterInfo(parameters[loop].getObjekt(), parameters[loop].getTyp(),
					parameters[loop].getAtg(), parameters[loop].getSim());
		}
		final ParameterManager hierarchie = MethodenBibliothek
				.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav());
		final Parameter[] neueSollParameters = hierarchie.getParameter(infos);
		Assert.isTrue(neueSollParameters.length == sollParameters.length);
		for (loop = 0; loop < neueSollParameters.length; ++loop) {
			sollParameters[loop] = neueSollParameters[loop];
		}
	}

	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder(getClass().getSimpleName());
		b.append(", ");
		b.append(parameters.length);
		b.append(" Parameter.\n");
		for (final Parameter p : parameters) {
			b.append(p.getInfo().toString());
		}
		b.append('\n');
		return b.toString();
	}

}
