/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors;

import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.buv.plugin.param.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.param.views.ObjektauswahlView;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenSender;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.puk.param.lib.MethodenBibliothek;
import de.bsvrz.puk.param.lib.Parameter;
import de.bsvrz.puk.param.lib.ParameterClientException;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Benutzer;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Statische Hilfsklasse zur Abwicklung der allgemeinen Funktionalität
 * 'Speichern von Parametern'.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 *
 */
public final class ParameterSaver {

	/**
	 * Logger für Debugausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Mögliche Ursachen für das Speichern.
	 *
	 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
	 *
	 *
	 */
	public enum ParameterSaveUrsache {
		/**
		 * Der Parameter wurde normal geschrieben.
		 */
		PARAMETRIERT("parametriert"),

		/**
		 * Der Parameter wurde kopiert.
		 */
		KOPIERT("kopiert"),

		/**
		 * Der Parameter wurde gelöscht.
		 */
		GELOESCHT("gelöscht"),

		/**
		 * Der Parameter wurde getauscht.
		 */
		GETAUSCHT("getauscht");

		/**
		 * Konstruktor initialisiert nur den nutzer-lesbaren Namen.
		 *
		 * @param s
		 *            der nutzer-lesbare Name.
		 */
		private ParameterSaveUrsache(final String s) {
			this.userFriendlyName = s;
		}

		/**
		 * der nutzer-lesbare Name.
		 */
		private String userFriendlyName;

		@Override
		public String toString() {
			return userFriendlyName;
		}
	}

	/**
	 * der letzte Fehler als Klartext.
	 */
	private static String lastError = "";

	/**
	 * Erstellt einen Text, der als Vorbelegung für das "Ursache"-Feld im
	 * UrlasserDatenDialog verwendet werden kann.
	 *
	 * @param inputParameter
	 *            das Feld der Quellparameter. Kann <code>null</code> sein.
	 * @param editedParameters
	 *            das Feld der zu speichernden Parameter
	 * @param saveUrsache
	 *            Grund für das Speichern. Siehe {@link ParameterSaveUrsache}
	 *
	 * @return der Text
	 */
	public static String buildDefaultUrsache(final Parameter[] inputParameter, final Parameter[] editedParameters,
			final ParameterSaveUrsache saveUrsache) {
		if ((null == editedParameters) || (editedParameters.length <= 0)) {
			return "Keine Parameter zum Speichern ausgewählt!";
		}
		int loop;
		final StringBuilder b = new StringBuilder();
		if (ParameterSaveUrsache.KOPIERT.equals(saveUrsache)) {
			b.append("Daten der ");
		}
		b.append("Attributgruppe '");
		b.append(editedParameters[0].getAtg().getName());
		if (ParameterSaveUrsache.KOPIERT.equals(saveUrsache)) {
			if ((null != inputParameter) && (inputParameter.length > 0)) {
				b.append("' von Objekt '");
				b.append(inputParameter[0].getObjekt().toString());
			}
			b.append("' auf ");
		} else if (ParameterSaveUrsache.GETAUSCHT.equals(saveUrsache)) {
			b.append("zwischen '");
			if ((null != inputParameter) && (inputParameter.length > 0)) {
				b.append(inputParameter[0].getObjekt().toString());
				b.append("' und '");
			} else {
				b.append("einem Objekt");
			}
			b.append(editedParameters[0].getObjekt().toString());
			b.append("' getauscht");
			return b.toString();
		} else {
			b.append("' an ");
		}
		b.append(editedParameters.length);
		b.append(" Objekt");
		if (editedParameters.length > 1) {
			if (ParameterSaveUrsache.KOPIERT.equals(saveUrsache)) {
				b.append('e');
			} else {
				b.append("en");
			}
		}
		b.append(' ');
		b.append(saveUrsache.toString());
		b.append(":\n ");
		final int maxObjects = 3;
		for (loop = 0; loop < Math.min(maxObjects, editedParameters.length); ++loop) {
			if (loop > 0) {
				b.append(",\n ");
			}
			b.append(editedParameters[loop].getObjekt().toString());
		}
		if (editedParameters.length > maxObjects) {
			b.append(", ...");
		}
		return b.toString();
	}

	/**
	 * Speichert die übergebenen Parameter auf dem DaV, inkl. Urlasserabfrage.
	 *
	 * @param editedParameters
	 *            die zu speichernden Parameter
	 * @param defaultUrsache
	 *            Text, mit dem das Ursache-Feld des {@link UrlasserInfoDatenDialog}
	 *            vorinitialisiert wird
	 * @return Ergebnis des Vorgangs, siehe Dokumentation zu
	 *         {@link ParameterSaveResultEnum}
	 */
	public static synchronized ParameterSaveResultEnum saveParameters(final Parameter[] editedParameters,
			final String defaultUrsache) {
		ParameterSaveResultEnum saveResult = ParameterSaveResultEnum.PARAMETER_SAVE_CANCEL;

		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (!rahmenwerk.isOnline()) {
			return ParameterSaveResultEnum.PARAMETER_SAVE_FAILURE;
		}

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

		final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(shell, new UrlasserInfoDatenSender() {

			@Override
			public void execute(final ClientDavInterface verbindung, final UrlasserInfo urlasser) {
				// TODO Urlasserverbindung verwenden
			}
		});

		dialog.setInitialBenutzer(rahmenwerk.getBenutzerName());
		dialog.setInitialUrsache(defaultUrsache);
		dialog.setInitialVeranlasser(rahmenwerk.getBenutzerName());

		if (dialog.open() == Window.OK) {
			final UrlasserInfo urlasserInfo = dialog.getUrlasserInfo();
			final Benutzer benutzer = (Benutzer) RahmenwerkService.getService().getObjektFactory()
					.getModellobjekt(urlasserInfo.getBenutzer());
			if (benutzer == null) {
				throw new IllegalArgumentException(
						"Benutzer " + urlasserInfo.getBenutzer() + " nicht im System vorhanden");
			}
			final SystemObject benutzerReferenz = benutzer.getSystemObject();
			for (final Parameter result : editedParameters) {
				final Data data = result.getData();
				if (data != null) {
					final Data urlasserItem = data.getItem("Urlasser");
					urlasserItem.getReferenceValue("BenutzerReferenz").setSystemObject(benutzerReferenz);
					String ursache = urlasserInfo.getUrsache();
					if (ursache == null) {
						ursache = "(keine Angabe)";
					}
					urlasserItem.getTextValue("Ursache").setText(ursache);
					urlasserItem.getTextValue("Veranlasser").setText(urlasserInfo.getVeranlasser());
					// Gehört eigentlich in die Methodenbibliothek
					if (!data.isDefined()) {
						throw new IllegalStateException(
								"Unvollständiger Parametersatz für Objekt " + result.getObjekt().toString());
					}
				}
			}
			try {
				ParameterSaver.DEBUG.config("Schreibe " + editedParameters.length + " Parameter...");
				MethodenBibliothek.getParameterManager(RahmenwerkService.getService().getObjektFactory().getDav())
						.setParameter(editedParameters);
				saveResult = ParameterSaveResultEnum.PARAMETER_SAVE_SUCCESS;
			} catch (final ParameterClientException e) {
				ParamPlugin.getDefault().getLogger().error("Fehler beim Versand von Parametern", e);
				saveResult = ParameterSaveResultEnum.PARAMETER_SAVE_FAILURE;
				ParameterSaver.lastError = e.getLocalizedMessage();
			}
		}

		if (saveResult == ParameterSaveResultEnum.PARAMETER_SAVE_SUCCESS) {
			final IViewReference[] viewReferences = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage()
					.getViewReferences();
			for (final IViewReference viewRef : viewReferences) {
				final IViewPart view = viewRef.getView(false);
				if (view instanceof ObjektauswahlView) {
					((ObjektauswahlView) view).refresh();
				}
			}
		}

		return saveResult;
	}

	/**
	 * Liefert den letzten Fehler als Klartext.
	 *
	 * @return die Fehlermeldung
	 */
	public static String getLastError() {
		return ParameterSaver.lastError;
	}

	/**
	 * Leerer, privater Konstruktor, keine Instanzen erlaubt.
	 */
	private ParameterSaver() {
		// Leerer, privater Konstruktor, keine Instanzen erlaubt
	}
}
