/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.attribut;

import java.util.EventObject;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;

/**
 * Abstrakte Implementierung eines Editors für einen Attributwert. Der Editor
 * stellt lediglich ein Composite mit einem Gridlayout und ohne umgebenden
 * Rahmen zur Verfügung, und bietet eine gemeinsame als Basisklasse für
 * Eingabefelder in Dialogen, Formeditoren und als Zelleneditor in Tabellen und
 * Bäumen.
 *
 * Die konkrete Ausprägung der Editoren erfolgt in den Erweiterungen dieser
 * Basisklasse, in denen dann auch die konkreten Inhalte und Methoden zurr
 * Eingabe und Validierung der eingegebenen Daten je nach editiertem Attribut
 * bzw. Attributyp erfolgt.
 *
 * @param <T>
 *            der Typ der Daten, die der Editor verarbeitet bzw. bereitstellt
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public abstract class AbstraktAttributEditor<T> extends Composite {

	/**
	 * Listenerliste für Objekte, die sich auf Änderungen anmelden.
	 */
	private final ListenerList listeners = new ListenerList();

	/**
	 * true - Ausgabe läuft. Zur Unterbrechung von Rekursion im Konfigeditor, im
	 * Param-Plug-in nicht benutzt.
	 */
	private boolean ausgabeStatus = true;

	/**
	 * Konstrukor reicht nur durch.
	 *
	 * @param parent
	 *            Elternobjekt.
	 */
	AbstraktAttributEditor(final Composite parent) {
		super(parent, SWT.NONE);
	}

	/**
	 * erzeugt einen AttributEditor in abhängigkeit vom übergebenen Typ.
	 *
	 * @param parent
	 *            das umgebende Element
	 * @param stil
	 *            Stil-Flags für das Element
	 * @param editierbar
	 *            der Inhalt des Editors kann bearbeitet werden
	 * @param attributeType
	 *            der Typ des Attributs, dessen Wert bearbeitet werden soll
	 * @return den Editor oder null, wenn keiner erzeugt werden konnte.
	 */
	public static AbstraktAttributEditor<?> erzeugeAttributEditor(final Composite parent, final int stil,
			final boolean editierbar, final AttributeType attributeType) {
		AbstraktAttributEditor<?> result = null;
		if (attributeType instanceof IntegerAttributeType) {
			result = new IntegerAttributeEditor(parent, stil, editierbar, (IntegerAttributeType) attributeType);
		}
		if (attributeType instanceof DoubleAttributeType) {
			result = new DoubleAttributeEditor(parent, stil, editierbar, (DoubleAttributeType) attributeType);
		}
		if (attributeType instanceof StringAttributeType) {
			result = new TextAttributeEditor(parent, stil, editierbar, (StringAttributeType) attributeType);
		}
		// Zur Übersetzung dieser beiden Editortypen müssten Anpassungen
		// vorgenommen
		// werden, das ist zu aufwendig, wir haben ja schon Editoren für diese
		// Attributtypen

		// if (attributeType instanceof TimeAttributeType) {
		// result = new TimeAttributeEditor(parent, stil, editierbar,
		// (TimeAttributeType) attributeType);
		// }
		// if (attributeType instanceof ReferenceAttributeType) {
		// result = new ReferenceAttributeEditor(parent, stil, editierbar,
		// (ReferenceAttributeType) attributeType);
		// }

		if (result != null) {
			GridDataFactory.fillDefaults().grab(true, true).applyTo(result);
		}

		return result;
	}

	/**
	 * registriert einen Listener, der über geänderte Daten informiert werden soll.
	 *
	 * @param listener
	 *            der neue Listener
	 */
	public void addDatenAenderungsListener(final DatenAenderungListener listener) {
		listeners.add(listener);
	}

	/**
	 * benachrichtigt alle angemeldeten Listener über eine Änderung der Daten des
	 * Editors.
	 *
	 * @param event
	 *            die Daten des Ereignisses
	 */
	protected void fireAenderungsListener(final EventObject event) {
		if (ausgabeStatus) {
			for (final Object listener : listeners.getListeners()) {
				if (listener instanceof DatenAenderungListener) {
					((DatenAenderungListener) listener).datenGeandert(event);
				}
			}
		}
	}

	/**
	 * liefert das innerhalb des Editors bearbeitete Datum.
	 *
	 * @return den Inhalt des bearbeiteten Attributwertes.
	 */
	public abstract T getWert();

	/**
	 * füllt das übergebene Datenelement mit dem aktuellen Wert des Editors.
	 *
	 * @param daten
	 *            das Ziel für den aktuellen Wert des Editors
	 */
	public abstract void getWert(final Data daten);

	/**
	 * entfernt einen Listener, der über geänderte Daten informiert wurde.
	 *
	 * @param listener
	 *            der neue Listener
	 */
	public void removeDatenAenderungsListener(final DatenAenderungListener listener) {
		listeners.remove(listener);
	}

	/**
	 * setzt die Daten, die mit dem Attributeditor bearbeitet werden sollen.
	 *
	 * @param daten
	 *            die Daten
	 */
	public abstract void setWert(final Data daten);

	/**
	 * Setzt den Wert, der initial angezeigt wird.
	 *
	 * @param wert
	 *            der Initialwert
	 */
	public abstract void setWert(T wert);

	/**
	 * markiert den Status für die Ausgabe von Daten. Bei inaktivem Status werden
	 * keine Benachrichtigungen an Listener versendet.
	 *
	 * @param status
	 *            der Status.
	 */
	protected final void setzeAusgabestatus(final boolean status) {
		ausgabeStatus = status;
	}

}
