/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.attribut;

import java.util.EventObject;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;

import de.bsvrz.buv.plugin.param.internal.ParamPlugin;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;

/**
 * Editor zum Bearbeiten der Daten eines Attributs vom Basistyp Kommazahl.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class DoubleAttributeEditor extends AbstraktAttributEditor<Double> {

	/**
	 * Numerischer Wert.
	 */
	private Double wert;

	/**
	 * Usercontrol zur Eingabe des Werts.
	 */
	private final Text wertFeld;

	/**
	 * Die Genauigkeit. Eines von DoubleAttributeType.FLOAT oder DOUBLE.
	 */
	private final byte genauigkeit;

	/**
	 * Wert als Text.
	 */
	private String textValue = "0";

	/**
	 * Konstruktor, erzeugt eine Instanz des Editors für den übergebenen Attributtyp
	 * innerhalb des übergebenen Containerelements.
	 *
	 * @param parent
	 *            das umgebende Element
	 * @param stil
	 *            Stil-Flags für das Element
	 * @param editierbar
	 *            der Inhalt des Editors kann bearbeitet werden
	 * @param attributeType
	 *            der Typ, für den daten bearbeitet werden sollen
	 */
	public DoubleAttributeEditor(final Composite parent, final int stil, final boolean editierbar,
			final DoubleAttributeType attributeType) {
		super(parent);

		GridLayoutFactory.fillDefaults().applyTo(this);

		wert = 0.0;
		genauigkeit = attributeType.getAccuracy();
		wertFeld = new Text(this, stil);
		GridDataFactory.fillDefaults().grab(true, false).applyTo(wertFeld);
		wertFeld.setEditable(editierbar);
		if (editierbar) {
			wertFeld.addVerifyListener(new VerifyListener() {

				@Override
				public void verifyText(final VerifyEvent event) {
					final StringBuffer eingabe = new StringBuffer(wertFeld.getText().trim());
					if (event.start == event.end) {
						eingabe.insert(event.start, event.text);
					} else {
						eingabe.replace(event.start, event.end, event.text);
					}
					try {
						final double neuerWert = Double.valueOf(eingabe.toString());
						if (genauigkeit == DoubleAttributeType.FLOAT) {
							if (neuerWert > Float.MAX_VALUE) {
								throw new NumberFormatException("Die angegebene Zahl " + neuerWert
										+ " überschreitet den zulässigen Wertebereich!");
							}
						}
					} catch (final NumberFormatException e) {
						event.doit = false;
					}
				}
			});

			wertFeld.addModifyListener(new ModifyListener() {

				@Override
				public void modifyText(final ModifyEvent event) {
					try {
						final String wertStr = wertFeld.getText();
						wert = Double.valueOf(wertStr);
						textValue = wertStr;
					} catch (final NumberFormatException e) {
						// ORiginal: KonfigEditor.fehlerMeldung(null, e);
						ParamPlugin.getDefault().getLogger().error("Texteingabe nicht in Double konvertierbar", e);
					}
				}
			});

			wertFeld.addFocusListener(new FocusAdapter() {
				@Override
				public void focusLost(final FocusEvent e) {
					fireAenderungsListener(new EventObject(DoubleAttributeEditor.this));
				}
			});
		}

	}

	@Override
	public Double getWert() {
		return wert;
	}

	@Override
	public void getWert(final Data daten) {
		if (genauigkeit == DoubleAttributeType.FLOAT) {
			daten.asUnscaledValue().set(wert.floatValue());
		} else {
			daten.asUnscaledValue().set(wert);
		}
	}

	@Override
	public void setWert(final Data daten) {
		if (genauigkeit == DoubleAttributeType.FLOAT) {
			wert = (double) daten.asUnscaledValue().floatValue();
		} else {
			wert = daten.asUnscaledValue().doubleValue();
		}
		wertFeld.setText(Double.toString(wert));
	}

	@Override
	public void setWert(final Double wert) {
		this.wert = wert;
	}

	/**
	 * Liefert den Wert als Text.
	 *
	 * @return der Wert.
	 */
	public String getTextValue() {
		return textValue;
	}
}
