/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.attribut;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueState;

/**
 * {@link TitleAreaDialog}, der ein komplexes Steuerelement vom Typ
 * {@link IntegerAttributeEditor} verkapselt und die eingegebenen Daten zur
 * Verfügung stellt - auch, nachdem der Dialog geschlossen wurde.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 *
 */
public class IntegerAttributDialog extends TitleAreaDialog {

	/**
	 * zu bearbeitender Typ. Darf nicht null sein.
	 */
	private final IntegerAttributeType typ;

	/**
	 * Initialwert. Kann <code>null</code> sein, in dem Fall wird zuerst versucht,
	 * den Default-Wert des Typs zu bestimmen und zu setzen. Wenn es keinen
	 * Defaultwert gibt, wird der erste State genommen. Wenn es keine States gibt,
	 * das Minimum.
	 */
	private final long initialWert;

	/**
	 * Der verwendete Editor.
	 */
	private IntegerAttributeEditor editor;

	/**
	 * Konstruktor weist Werte zu.
	 *
	 * @param parentShell
	 *            Eltern-Shell oder <code>null</code>
	 * @param typ
	 *            zu bearbeitender Typ. Darf nicht null sein.
	 * @param initialWert
	 *            Initialwert. Kann <code>null</code> sein, in dem Fall wird zuerst
	 *            versucht, den Default-Wert des Typs zu bestimmen und zu setzen.
	 *            Wenn es keinen Defaultwert gibt, wird der erste State genommen.
	 *            Wenn es keine States gibt, das Minimum.
	 */
	public IntegerAttributDialog(final Shell parentShell, final IntegerAttributeType typ, final Long initialWert) {
		super(parentShell);
		Assert.isNotNull(typ);
		if (null == initialWert) {
			Long defaultWertLong = null;
			final String defaultWertString = typ.getDefaultAttributeValue();
			if (null != defaultWertString) {
				try {
					defaultWertLong = Long.parseLong(defaultWertString);
				} catch (final NumberFormatException e) {
					for (final IntegerValueState state : typ.getStates()) {
						if (state.getName().equals(defaultWertString)) {
							defaultWertLong = state.getValue();
							break;
						}
					}
				}
			}
			if (null == defaultWertLong) {
				if (null == typ.getStates() || typ.getStates().isEmpty()) {
					defaultWertLong = typ.getRange().getMinimum();
				} else {
					defaultWertLong = typ.getStates().get(0).getValue();
				}
			}
			this.initialWert = defaultWertLong;
		} else {
			this.initialWert = initialWert;
		}
		this.typ = typ;
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		parent.getShell().setText("Auswahl eines Integer-Wertes oder -Zustands");
		setTitle("Wert oder Zustand zuweisen");
		setMessage("Wählen Sie einen Zustand aus oder setzen Sie einen numerischen Wert");

		final Composite panel = new Composite(parent, SWT.NONE);
		GridLayoutFactory.swtDefaults().applyTo(panel);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(panel);

		editor = new IntegerAttributeEditor(panel, SWT.BORDER, true, typ);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(editor);
		editor.setWert(initialWert);

		return panel;
	}

	/**
	 * Liefert den numerischen Wert, auch dann, wenn es ein Zustand war.
	 *
	 * @return der Wert
	 */
	public Long getWert() {
		return editor.getWert();
	}

	/**
	 * Liefert eine textliche Repräsentation dieses Werts.
	 *
	 * <ul>
	 * <li>Ist der Wert ein Zustand, wird dessen Name geliefert</li>
	 * <li>Ist es ein Bereichswert, wird der Long-Wert als String geliefert. Ist
	 * eine Maßeinheit definiert, so wird diese mit angehangen, mit einem
	 * Leerzeichen Abstand.</li>
	 * </ul>
	 *
	 * @return die Textrepräsentation. Es SOLLTE mögliche sein, diese direkt der
	 *         Methode {@link de.bsvrz.dav.daf.main.Data#asTextValue()} .setText()
	 *         zu übergeben.
	 */
	public String getTextValue() {
		return editor.getTextValue();
	}
}
