/*
 * Rahmenwerk-Plug-in "Parametrierung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.param.editors.attribut;

import java.util.ArrayList;
import java.util.EventObject;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.bitctrl.math.SkalierungHelper;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Editor zum Bearbeiten der Daten eines Attributs vom Basistyp Ganzzahl.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class IntegerAttributeEditor extends AbstraktAttributEditor<Long> {

	/**
	 * Hauptcontrol zur Eingabe eines Wertes einer Range.
	 */
	private Text wertEingabeFeld;

	/**
	 * Hauptcontrol zur Auswahl aus den verfügbaren States.
	 */
	private ComboViewer selektor;

	/**
	 * Die möglichen Zustände.
	 */
	private final Map<Long, IntegerValueState> states = new HashMap<>();

	/**
	 * Der Wertebereich. Kann <code>null</code> sein.
	 */
	private final IntegerValueRange range;

	/**
	 * Anzahl der für die Darstellung des Attributs benötigten Bytes.
	 */
	private final int byteCount;

	/**
	 * Skalierungsfaktor.
	 */
	private double faktor = 1;

	/**
	 * Der aktuelle Wert.
	 */
	private Long wert = 0L;

	/**
	 * Zulässiger Minimalwert des bearbeiteten Attributs.
	 */
	private long minimum;

	/**
	 * Zulässiger Maximalwert des bearbeiteten Attributs.
	 */
	private long maximum;

	/**
	 * Wert als Text.
	 */
	private String textValue = "0";

	/**
	 * Die Maßeinheit. Kann <code>null</code> sein.
	 */
	private String unit;

	/**
	 * Konstruktor, erzeugt eines Instanz des Editors für den übergebenen
	 * Attributtyp im übergebenden Containerelement.
	 *
	 * @param parent
	 *            das umgebende Element
	 * @param stil
	 *            Stil-Flags für das Element
	 * @param editierbar
	 *            der Inhalt des Editors kann bearbeitet werden
	 * @param typ
	 *            der Typ des Attributs, dessen Daten bearbeitet werden sollen
	 */
	public IntegerAttributeEditor(final Composite parent, final int stil, final boolean editierbar,
			final IntegerAttributeType typ) {
		super(parent);

		GridLayoutFactory.fillDefaults().numColumns(2).applyTo(this);

		if (typ.getStates() != null) {
			for (final IntegerValueState state : typ.getStates()) {
				states.put(state.getValue(), state);
			}
		}
		range = typ.getRange();
		byteCount = typ.getByteCount();

		if (range == null) {
			switch (byteCount) {
			case IntegerAttributeType.BYTE:
				minimum = Byte.MIN_VALUE;
				maximum = Byte.MAX_VALUE;
				break;
			case IntegerAttributeType.INT:
				minimum = Integer.MIN_VALUE;
				maximum = Integer.MAX_VALUE;
				break;
			case IntegerAttributeType.SHORT:
				minimum = Short.MIN_VALUE;
				maximum = Short.MAX_VALUE;
				break;
			case IntegerAttributeType.LONG:
			default:
				minimum = Long.MIN_VALUE;
				maximum = Long.MAX_VALUE;
				break;
			}
		} else {
			faktor = range.getConversionFactor();
			minimum = range.getMinimum();
			maximum = range.getMaximum();
			unit = range.getUnit();
		}

		wert = minimum;

		if (states.isEmpty()) {
			wertEingabeFeld = new Text(this, stil);
			wertEingabeFeld.setText("0");
			wertEingabeFeld.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			wertEingabeFeld.addVerifyListener(new VerifyListener() {

				@Override
				public void verifyText(final VerifyEvent event) {
					final StringBuffer derText = new StringBuffer(wertEingabeFeld.getText());
					if (event.start == event.end) {
						derText.insert(event.start, event.text);
					} else {
						derText.replace(event.start, event.end, event.text);
					}
					event.doit = aktualisiereWertVomText(derText.toString());
					if (event.doit) {
						fireAenderungsListener(new EventObject(IntegerAttributeEditor.this));
					}
				}
			});

		} else {
			final CCombo combo = new CCombo(this, stil);
			combo.setVisibleItemCount(10);
			GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.BEGINNING).applyTo(combo);

			selektor = new ComboViewer(combo);
			selektor.setContentProvider(new ArrayContentProvider());
			selektor.setLabelProvider(new LabelProvider() {
				@Override
				public String getText(final Object element) {
					if (element instanceof SystemObject) {
						return ((SystemObject) element).getNameOrPidOrId();
					}
					return super.getText(element);
				}
			});
			final List<IntegerValueState> statusWerte = new ArrayList<>(states.values());
			selektor.setInput(statusWerte);
			selektor.setSelection(new StructuredSelection(statusWerte.get(0)));
			selektor.addSelectionChangedListener(new ISelectionChangedListener() {
				@Override
				public void selectionChanged(final SelectionChangedEvent event) {
					final long alterWert = wert;
					final Object element = ((IStructuredSelection) selektor.getSelection()).getFirstElement();
					if (element instanceof IntegerValueState) {
						wert = ((IntegerValueState) element).getValue();
					} else {
						aktualisiereWertVomText(selektor.getCCombo().getText().trim());
					}
					if (wert != alterWert) {
						aktualisiereWert();
						fireAenderungsListener(new EventObject(IntegerAttributeEditor.this));
					}
				}
			});
			selektor.getCCombo().addVerifyListener(new VerifyListener() {

				@Override
				public void verifyText(final VerifyEvent event) {
					final StringBuffer derText = new StringBuffer(selektor.getCCombo().getText());
					if (event.start == event.end) {
						derText.insert(event.start, event.text);
					} else {
						derText.replace(event.start, event.end, event.text);
					}
					event.doit = aktualisiereWertVomText(derText.toString());
					if (event.doit) {
						fireAenderungsListener(new EventObject(this));
					}
				}
			});
			selektor.getCCombo().addFocusListener(new FocusAdapter() {
				@Override
				public void focusLost(final FocusEvent e) {
					aktualisiereWertVomText(selektor.getCCombo().getText().trim());
					fireAenderungsListener(new EventObject(IntegerAttributeEditor.this));
				}
			});

			if ((range == null) && (!states.isEmpty())) {
				combo.setEditable(false);
			}

			selektor.getControl().setEnabled(editierbar);
		}

		if (range != null) {
			final String lunit = range.getUnit();
			if ((lunit != null) && (lunit.length() > 0)) {
				final Label unitLabel = new Label(this, SWT.NONE);
				unitLabel.setText(lunit);
				GridDataFactory.swtDefaults().applyTo(unitLabel);
			}
		}
	}

	/**
	 * Aktualisiert den Wert aus den Nutzer-Controls.
	 */
	private void aktualisiereWert() {
		setzeAusgabestatus(false);
		if (wertEingabeFeld != null) {
			if (faktor == 1) {
				wertEingabeFeld.setText(Long.toString(wert.longValue()));
			} else {
				wertEingabeFeld.setText(SkalierungHelper.getInstanz().format(wert.doubleValue(), faktor));
			}
		} else if (selektor != null) {
			final IntegerValueState status = states.get(wert);
			if (status == null) {
				if (faktor == 1) {
					selektor.getCCombo().setText(Long.toString(wert.longValue()));
				} else {
					selektor.getCCombo().setText(SkalierungHelper.getInstanz().format(wert.doubleValue(), faktor));
				}
			} else {
				selektor.setSelection(new StructuredSelection(status));
			}
		}
		setzeAusgabestatus(true);
	}

	/**
	 * Aktualisiert den Wet aus einer Textangabe von außen.
	 *
	 * @param text
	 *            Nutzertext
	 * @return true - Wert wurde gesetzt.
	 */
	private boolean aktualisiereWertVomText(final String text) {
		double neuerWert = 0;
		boolean wertGesetzt = false;
		boolean result = true;

		for (final IntegerValueState state : states.values()) {
			if (state.getName().equals(text)) {
				wert = state.getValue();
				textValue = state.getName();
				wertGesetzt = true;
				break;
			}
		}

		if (!wertGesetzt) {

			try {
				neuerWert = Long.valueOf(text).doubleValue();
			} catch (final NumberFormatException e) {
				try {
					neuerWert = Double.valueOf(text).doubleValue();
				} catch (final NumberFormatException e1) {
					result = false;
				}
			}

			if (result) {
				final Double skaliert = neuerWert / faktor;
				if ((skaliert.longValue() < minimum) || (skaliert.longValue() > maximum)) {
					result = false;
				} else {
					wert = Math.round(skaliert);
					final IntegerValueState state = states.get(wert);
					if (null == state) {
						this.textValue = String.valueOf(neuerWert);
						if (null != unit) {
							this.textValue += " " + unit;
						}
					} else {
						this.textValue = state.getName();
					}
				}
			}
		}

		return result;
	}

	@Override
	public Long getWert() {
		return wert;
	}

	@Override
	public void getWert(final Data daten) {
		switch (byteCount) {
		case IntegerAttributeType.BYTE:
			daten.asUnscaledValue().set(wert.byteValue());
			break;
		case IntegerAttributeType.INT:
			daten.asUnscaledValue().set(wert.intValue());
			break;
		case IntegerAttributeType.SHORT:
			daten.asUnscaledValue().set(wert.shortValue());
			break;
		case IntegerAttributeType.LONG:
		default:
			daten.asUnscaledValue().set(wert.longValue());
			break;
		}
	}

	@Override
	public void setWert(final Data daten) {
		final long alterWert = wert;
		switch (byteCount) {
		case IntegerAttributeType.BYTE:
			wert = Long.valueOf(daten.asUnscaledValue().byteValue());
			break;
		case IntegerAttributeType.INT:
			wert = Long.valueOf(daten.asUnscaledValue().intValue());
			break;
		case IntegerAttributeType.SHORT:
			wert = Long.valueOf(daten.asUnscaledValue().shortValue());
			break;
		case IntegerAttributeType.LONG:
		default:
			wert = Long.valueOf(daten.asUnscaledValue().longValue());
			break;
		}
		textValue = daten.valueToString();

		if (wert != alterWert) {
			aktualisiereWert();
		}
	}

	@Override
	public void setWert(final Long wert) {
		final long alterWert = this.wert;
		this.wert = wert;
		if (wert != alterWert) {
			aktualisiereWert();
		}
	}

	/**
	 * Liefert den Wert als Text.
	 *
	 * @return der Text.
	 */
	public String getTextValue() {
		return textValue;
	}
}
